// Flying Book Creator - Dashboard JavaScript

document.addEventListener('DOMContentLoaded', function() {
    initDashboard();
});

function initDashboard() {
    // Animation des stats au chargement
    animateStats();
    
    // Gestion responsive du sidebar
    handleSidebarToggle();
    
    // Mise à jour en temps réel des données
    updateDashboardData();
    
    // Gestion des actions rapides
    handleQuickActions();
    
    // Auto-refresh des statistiques
    setInterval(updateDashboardData, 30000); // Toutes les 30 secondes
}

// Animation des statistiques
function animateStats() {
    const statCards = document.querySelectorAll('.stat-card');
    
    statCards.forEach((card, index) => {
        // Animation d'apparition décalée
        card.style.opacity = '0';
        card.style.transform = 'translateY(20px)';
        
        setTimeout(() => {
            card.style.transition = 'all 0.6s ease';
            card.style.opacity = '1';
            card.style.transform = 'translateY(0)';
        }, index * 100);
        
        // Animation des chiffres
        const numberElement = card.querySelector('.stat-content h3');
        if (numberElement) {
            const finalNumber = parseInt(numberElement.textContent) || 0;
            animateNumber(numberElement, 0, finalNumber, 1000);
        }
    });
}

// Animation compteur de nombres
function animateNumber(element, start, end, duration) {
    if (start === end) return;
    
    const range = end - start;
    const increment = range / (duration / 16);
    let current = start;
    
    const timer = setInterval(() => {
        current += increment;
        if ((increment > 0 && current >= end) || (increment < 0 && current <= end)) {
            current = end;
            clearInterval(timer);
        }
        
        // Format selon le type de donnée
        if (element.textContent.includes('GB') || element.textContent.includes('MB')) {
            element.textContent = formatFileSize(Math.floor(current));
        } else {
            element.textContent = Math.floor(current);
        }
    }, 16);
}

// Gestion responsive du sidebar
function handleSidebarToggle() {
    // Créer un bouton menu mobile si pas déjà présent
    if (window.innerWidth <= 1024 && !document.querySelector('.mobile-menu-btn')) {
        createMobileMenuButton();
    }
    
    // Gestion du redimensionnement
    window.addEventListener('resize', () => {
        if (window.innerWidth > 1024) {
            const sidebar = document.querySelector('.sidebar');
            sidebar.classList.remove('open');
            removeMobileMenuButton();
        } else if (!document.querySelector('.mobile-menu-btn')) {
            createMobileMenuButton();
        }
    });
    
    // Fermer le sidebar en cliquant à l'extérieur
    document.addEventListener('click', (e) => {
        const sidebar = document.querySelector('.sidebar');
        const mobileBtn = document.querySelector('.mobile-menu-btn');
        
        if (window.innerWidth <= 1024 && 
            sidebar.classList.contains('open') && 
            !sidebar.contains(e.target) && 
            e.target !== mobileBtn) {
            sidebar.classList.remove('open');
        }
    });
}

// Créer le bouton menu mobile
function createMobileMenuButton() {
    const button = document.createElement('button');
    button.className = 'mobile-menu-btn';
    button.innerHTML = '<i class="fas fa-bars"></i>';
    button.style.cssText = `
        position: fixed;
        top: 1rem;
        left: 1rem;
        z-index: 1001;
        width: 44px;
        height: 44px;
        border: none;
        border-radius: 8px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        font-size: 1.2rem;
        cursor: pointer;
        box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
        transition: all 0.3s ease;
    `;
    
    button.addEventListener('click', () => {
        const sidebar = document.querySelector('.sidebar');
        sidebar.classList.toggle('open');
    });
    
    document.body.appendChild(button);
}

// Supprimer le bouton menu mobile
function removeMobileMenuButton() {
    const button = document.querySelector('.mobile-menu-btn');
    if (button) {
        button.remove();
    }
}

// Mise à jour des données du dashboard
async function updateDashboardData() {
    try {
        const response = await fetch('api/dashboard-stats.php');
        if (!response.ok) throw new Error('Erreur réseau');
        
        const data = await response.json();
        updateStatCards(data.stats);
        updateProgressBars(data.usage);
        
    } catch (error) {
        console.log('Mise à jour des données différée:', error.message);
        // En cas d'erreur, on continue sans alerter l'utilisateur
    }
}

// Mettre à jour les cartes statistiques
function updateStatCards(stats) {
    if (!stats) return;
    
    const cards = document.querySelectorAll('.stat-card');
    
    cards.forEach((card, index) => {
        const numberElement = card.querySelector('.stat-content h3');
        const currentValue = parseInt(numberElement.textContent) || 0;
        
        let newValue;
        switch(index) {
            case 0: newValue = stats.total_projects || 0; break;
            case 1: newValue = stats.total_storage || 0; break;
            case 2: newValue = stats.total_views || 0; break;
            default: return;
        }
        
        if (newValue !== currentValue) {
            animateNumber(numberElement, currentValue, newValue, 800);
        }
    });
}

// Mettre à jour les barres de progression
function updateProgressBars(usage) {
    if (!usage) return;
    
    const progressBars = document.querySelectorAll('.progress-fill');
    
    progressBars.forEach((bar, index) => {
        let percentage = 0;
        
        switch(index) {
            case 0: // Projets
                percentage = usage.projects_percentage || 0;
                break;
            case 1: // Stockage
                percentage = usage.storage_percentage || 0;
                break;
        }
        
        // Animation de la barre de progression
        setTimeout(() => {
            bar.style.width = Math.min(100, percentage) + '%';
        }, index * 200);
    });
}

// Gestion des actions rapides
function handleQuickActions() {
    const actionCards = document.querySelectorAll('.action-card');
    
    actionCards.forEach(card => {
        card.addEventListener('click', function(e) {
            // Animation de clic
            this.style.transform = 'scale(0.95)';
            setTimeout(() => {
                this.style.transform = '';
            }, 150);
        });
        
        // Effet de survol amélioré
        card.addEventListener('mouseenter', function() {
            const icon = this.querySelector('i');
            icon.style.transform = 'scale(1.1) rotate(5deg)';
            icon.style.transition = 'transform 0.3s ease';
        });
        
        card.addEventListener('mouseleave', function() {
            const icon = this.querySelector('i');
            icon.style.transform = '';
        });
    });
}

// Notifications toast
function showNotification(message, type = 'info') {
    // Créer le conteneur de notifications s'il n'existe pas
    let container = document.querySelector('.notification-container');
    if (!container) {
        container = document.createElement('div');
        container.className = 'notification-container';
        container.style.cssText = `
            position: fixed;
            top: 2rem;
            right: 2rem;
            z-index: 2000;
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        `;
        document.body.appendChild(container);
    }
    
    // Créer la notification
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <i class="fas fa-${getNotificationIcon(type)}"></i>
        <span>${message}</span>
        <button class="notification-close">
            <i class="fas fa-times"></i>
        </button>
    `;
    
    notification.style.cssText = `
        display: flex;
        align-items: center;
        gap: 0.75rem;
        padding: 1rem 1.25rem;
        border-radius: 8px;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
        transform: translateX(100%);
        transition: all 0.3s ease;
        min-width: 300px;
        max-width: 400px;
    `;
    
    // Couleurs selon le type
    const colors = {
        info: { bg: '#dbeafe', border: '#3b82f6', text: '#1e40af' },
        success: { bg: '#d1fae5', border: '#10b981', text: '#065f46' },
        warning: { bg: '#fef3c7', border: '#f59e0b', text: '#92400e' },
        error: { bg: '#fee2e2', border: '#ef4444', text: '#dc2626' }
    };
    
    const color = colors[type] || colors.info;
    notification.style.backgroundColor = color.bg;
    notification.style.borderLeft = `4px solid ${color.border}`;
    notification.style.color = color.text;
    
    // Ajouter au conteneur
    container.appendChild(notification);
    
    // Animation d'entrée
    setTimeout(() => {
        notification.style.transform = 'translateX(0)';
    }, 10);
    
    // Gestion de la fermeture
    const closeBtn = notification.querySelector('.notification-close');
    closeBtn.style.cssText = `
        background: none;
        border: none;
        color: inherit;
        cursor: pointer;
        padding: 0.25rem;
        border-radius: 4px;
        opacity: 0.7;
        transition: opacity 0.3s ease;
    `;
    
    closeBtn.addEventListener('click', () => closeNotification(notification));
    
    // Auto-fermeture après 5 secondes
    setTimeout(() => {
        if (notification.parentNode) {
            closeNotification(notification);
        }
    }, 5000);
}

// Fermer une notification
function closeNotification(notification) {
    notification.style.transform = 'translateX(100%)';
    notification.style.opacity = '0';
    
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 300);
}

// Icônes pour les notifications
function getNotificationIcon(type) {
    const icons = {
        info: 'info-circle',
        success: 'check-circle',
        warning: 'exclamation-triangle',
        error: 'exclamation-circle'
    };
    return icons[type] || 'info-circle';
}

// Formatage de la taille des fichiers
function formatFileSize(bytes) {
    if (bytes === 0) return '0 B';
    
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

// Gestion des erreurs globales
window.addEventListener('error', function(e) {
    console.error('Erreur JavaScript:', e.error);
    // En production, on pourrait envoyer l'erreur à un service de monitoring
});

// Gestion des erreurs de promesses non catchées
window.addEventListener('unhandledrejection', function(e) {
    console.error('Promesse rejetée:', e.reason);
    e.preventDefault();
});

// Utilitaires pour l'interaction avec l'API
const API = {
    async get(url) {
        const response = await fetch(url, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
            }
        });
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        return await response.json();
    },
    
    async post(url, data) {
        const response = await fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(data)
        });
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        return await response.json();
    }
};

// Exposer les fonctions utiles globalement
window.FlyingBook = {
    showNotification,
    formatFileSize,
    API
};