// Flying Book Creator - Authentication JavaScript

document.addEventListener('DOMContentLoaded', function() {
    initAuth();
});

function initAuth() {
    // Initialiser les fonctionnalités
    handleFormSwitching();
    setupPasswordToggles();
    setupPasswordStrength();
    setupFormValidation();
    setupFormSubmission();
    
    // Animations d'entrée
    animateFeatures();
    
    // Focus automatique sur le premier champ
    focusFirstInput();
}

// Gestion du basculement entre connexion et inscription
function handleFormSwitching() {
    const switchButton = document.getElementById('switchButton');
    const switchText = document.getElementById('switchText');
    const loginForm = document.getElementById('loginForm');
    const registerForm = document.getElementById('registerForm');
    
    if (!switchButton || !switchText || !loginForm || !registerForm) return;
    
    switchButton.addEventListener('click', function() {
        const isLoginActive = loginForm.classList.contains('active');
        
        if (isLoginActive) {
            // Passer à l'inscription
            loginForm.classList.remove('active');
            registerForm.classList.add('active');
            switchText.textContent = 'Déjà un compte ?';
            switchButton.textContent = 'Se connecter';
            
            // Animation du logo
            animateLogo();
            
            // Focus sur le premier champ
            setTimeout(() => {
                const firstInput = registerForm.querySelector('input');
                if (firstInput) firstInput.focus();
            }, 200);
            
        } else {
            // Passer à la connexion
            registerForm.classList.remove('active');
            loginForm.classList.add('active');
            switchText.textContent = 'Pas encore de compte ?';
            switchButton.textContent = 'S\'inscrire';
            
            // Animation du logo
            animateLogo();
            
            // Focus sur le premier champ
            setTimeout(() => {
                const firstInput = loginForm.querySelector('input');
                if (firstInput) firstInput.focus();
            }, 200);
        }
    });
}

// Configuration des boutons de basculement de mot de passe
function setupPasswordToggles() {
    const toggleButtons = document.querySelectorAll('.toggle-password');
    
    toggleButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            
            const targetId = this.getAttribute('onclick').match(/'([^']+)'/)[1];
            const passwordInput = document.getElementById(targetId);
            const icon = this.querySelector('i');
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
                this.setAttribute('aria-label', 'Masquer le mot de passe');
            } else {
                passwordInput.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
                this.setAttribute('aria-label', 'Afficher le mot de passe');
            }
            
            // Animation du bouton
            this.style.transform = 'scale(0.9)';
            setTimeout(() => {
                this.style.transform = '';
            }, 100);
        });
    });
}

// Fonction globale pour la compatibilité avec onclick
function togglePassword(inputId) {
    const passwordInput = document.getElementById(inputId);
    const button = passwordInput.parentNode.querySelector('.toggle-password');
    const icon = button.querySelector('i');
    
    if (passwordInput.type === 'password') {
        passwordInput.type = 'text';
        icon.classList.remove('fa-eye');
        icon.classList.add('fa-eye-slash');
    } else {
        passwordInput.type = 'password';
        icon.classList.remove('fa-eye-slash');
        icon.classList.add('fa-eye');
    }
}

// Indicateur de force du mot de passe
function setupPasswordStrength() {
    const passwordInput = document.getElementById('reg_password');
    const strengthBar = document.querySelector('.strength-bar');
    const strengthText = document.querySelector('.strength-text');
    
    if (!passwordInput || !strengthBar || !strengthText) return;
    
    passwordInput.addEventListener('input', function() {
        const password = this.value;
        const strength = calculatePasswordStrength(password);
        
        // Mettre à jour la barre de progression
        strengthBar.style.setProperty('--width', strength.percentage + '%');
        strengthBar.setAttribute('data-strength', strength.level);
        
        // Mettre à jour le texte
        strengthText.textContent = strength.message;
        strengthText.className = 'strength-text strength-' + strength.level;
    });
}

// Calculer la force du mot de passe
function calculatePasswordStrength(password) {
    if (password.length === 0) {
        return { level: 'none', percentage: 0, message: 'Minimum 6 caractères' };
    }
    
    let score = 0;
    const checks = {
        length: password.length >= 6,
        lowercase: /[a-z]/.test(password),
        uppercase: /[A-Z]/.test(password),
        numbers: /\d/.test(password),
        special: /[^A-Za-z0-9]/.test(password),
        longLength: password.length >= 12
    };
    
    // Calcul du score
    if (checks.length) score += 20;
    if (checks.lowercase) score += 15;
    if (checks.uppercase) score += 15;
    if (checks.numbers) score += 15;
    if (checks.special) score += 20;
    if (checks.longLength) score += 15;
    
    // Déterminer le niveau
    let level, message;
    if (score < 30) {
        level = 'weak';
        message = 'Faible - Ajoutez plus de caractères';
    } else if (score < 60) {
        level = 'medium';
        message = 'Moyen - Ajoutez majuscules et chiffres';
    } else if (score < 80) {
        level = 'good';
        message = 'Bon - Ajoutez des caractères spéciaux';
    } else {
        level = 'strong';
        message = 'Très fort - Parfait !';
    }
    
    return { level, percentage: Math.min(100, score), message };
}

// Validation des formulaires
function setupFormValidation() {
    const forms = document.querySelectorAll('.auth-form');
    
    forms.forEach(form => {
        const inputs = form.querySelectorAll('input[required]');
        
        inputs.forEach(input => {
            // Validation en temps réel
            input.addEventListener('blur', () => validateField(input));
            input.addEventListener('input', () => clearFieldError(input));
        });
        
        // Validation à la soumission
        form.addEventListener('submit', function(e) {
            if (!validateForm(this)) {
                e.preventDefault();
            }
        });
    });
}

// Valider un champ individuel
function validateField(input) {
    const value = input.value.trim();
    const inputGroup = input.closest('.input-group');
    let isValid = true;
    let message = '';
    
    // Supprimer les erreurs existantes
    clearFieldError(input);
    
    // Validation selon le type de champ
    switch (input.type) {
        case 'email':
            if (!value) {
                isValid = false;
                message = 'L\'email est requis';
            } else if (!isValidEmail(value)) {
                isValid = false;
                message = 'Format d\'email invalide';
            }
            break;
            
        case 'password':
            if (!value) {
                isValid = false;
                message = 'Le mot de passe est requis';
            } else if (value.length < 6) {
                isValid = false;
                message = 'Minimum 6 caractères requis';
            }
            break;
            
        case 'text':
            if (input.name === 'name' && !value) {
                isValid = false;
                message = 'Le nom est requis';
            } else if (input.name === 'confirm_password') {
                const password = document.getElementById('reg_password').value;
                if (!value) {
                    isValid = false;
                    message = 'Confirmez votre mot de passe';
                } else if (value !== password) {
                    isValid = false;
                    message = 'Les mots de passe ne correspondent pas';
                }
            }
            break;
    }
    
    // Afficher l'erreur si nécessaire
    if (!isValid) {
        showFieldError(input, message);
    }
    
    return isValid;
}

// Valider un formulaire complet
function validateForm(form) {
    const inputs = form.querySelectorAll('input[required]');
    let isValid = true;
    
    inputs.forEach(input => {
        if (!validateField(input)) {
            isValid = false;
        }
    });
    
    return isValid;
}

// Afficher une erreur sur un champ
function showFieldError(input, message) {
    const inputGroup = input.closest('.input-group') || input.closest('.form-group');
    
    // Supprimer l'erreur existante
    clearFieldError(input);
    
    // Ajouter la classe d'erreur
    inputGroup.classList.add('error');
    input.classList.add('error');
    
    // Créer le message d'erreur
    const errorDiv = document.createElement('div');
    errorDiv.className = 'field-error';
    errorDiv.textContent = message;
    
    // Style du message d'erreur
    errorDiv.style.cssText = `
        color: #dc2626;
        font-size: 0.8rem;
        margin-top: 0.25rem;
        display: flex;
        align-items: center;
        gap: 0.25rem;
        animation: slideDown 0.3s ease-out;
    `;
    
    // Ajouter une icône d'erreur
    const icon = document.createElement('i');
    icon.className = 'fas fa-exclamation-circle';
    errorDiv.insertBefore(icon, errorDiv.firstChild);
    
    // Insérer le message
    inputGroup.appendChild(errorDiv);
    
    // Style de l'input en erreur
    input.style.borderColor = '#dc2626';
    input.style.boxShadow = '0 0 0 3px rgba(220, 38, 38, 0.1)';
}

// Supprimer l'erreur d'un champ
function clearFieldError(input) {
    const inputGroup = input.closest('.input-group') || input.closest('.form-group');
    const errorDiv = inputGroup.querySelector('.field-error');
    
    if (errorDiv) {
        errorDiv.remove();
    }
    
    inputGroup.classList.remove('error');
    input.classList.remove('error');
    input.style.borderColor = '';
    input.style.boxShadow = '';
}

// Validation d'email
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

// Gestion de la soumission des formulaires
function setupFormSubmission() {
    const forms = document.querySelectorAll('.auth-form');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const submitBtn = this.querySelector('.btn-primary');
            
            // Animation de chargement
            if (submitBtn) {
                submitBtn.classList.add('loading');
                submitBtn.disabled = true;
                
                // Réactiver le bouton après 5 secondes (sécurité)
                setTimeout(() => {
                    submitBtn.classList.remove('loading');
                    submitBtn.disabled = false;
                }, 5000);
            }
        });
    });
}

// Animation des fonctionnalités
function animateFeatures() {
    const features = document.querySelectorAll('.feature');
    
    features.forEach((feature, index) => {
        feature.style.opacity = '0';
        feature.style.transform = 'translateY(30px)';
        
        setTimeout(() => {
            feature.style.transition = 'all 0.6s ease';
            feature.style.opacity = '1';
            feature.style.transform = 'translateY(0)';
        }, 300 + (index * 100));
    });
}

// Animation du logo
function animateLogo() {
    const logoIcon = document.querySelector('.logo i');
    if (logoIcon) {
        logoIcon.style.transform = 'scale(1.1) rotate(10deg)';
        setTimeout(() => {
            logoIcon.style.transform = '';
        }, 300);
    }
}

// Focus sur le premier champ
function focusFirstInput() {
    setTimeout(() => {
        const activeForm = document.querySelector('.auth-form.active');
        if (activeForm) {
            const firstInput = activeForm.querySelector('input');
            if (firstInput) {
                firstInput.focus();
            }
        }
    }, 100);
}

// Fonction globale pour le basculement (compatibilité avec onclick)
function switchForm() {
    const switchButton = document.getElementById('switchButton');
    if (switchButton) {
        switchButton.click();
    }
}

// Gestion des cookies (remember me)
function setCookie(name, value, days) {
    const expires = new Date();
    expires.setTime(expires.getTime() + (days * 24 * 60 * 60 * 1000));
    document.cookie = `${name}=${value};expires=${expires.toUTCString()};path=/;secure;samesite=strict`;
}

function getCookie(name) {
    const nameEQ = name + "=";
    const ca = document.cookie.split(';');
    for (let i = 0; i < ca.length; i++) {
        let c = ca[i];
        while (c.charAt(0) === ' ') c = c.substring(1, c.length);
        if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
    }
    return null;
}

// Auto-complétion intelligente
function setupAutoComplete() {
    const emailInputs = document.querySelectorAll('input[type="email"]');
    
    emailInputs.forEach(input => {
        input.addEventListener('input', function() {
            const value = this.value;
            const domains = ['gmail.com', 'yahoo.fr', 'hotmail.com', 'outlook.fr', 'orange.fr'];
            
            if (value.includes('@') && value.split('@')[1].length > 0) {
                const domain = value.split('@')[1];
                const suggestions = domains.filter(d => d.startsWith(domain));
                
                if (suggestions.length > 0) {
                    // Ici on pourrait afficher des suggestions
                    console.log('Suggestions:', suggestions);
                }
            }
        });
    });
}

// Détection de la force du réseau
function detectNetworkQuality() {
    if ('connection' in navigator) {
        const connection = navigator.connection;
        if (connection.effectiveType === 'slow-2g' || connection.effectiveType === '2g') {
            // Réduire les animations pour les connexions lentes
            document.body.classList.add('reduced-motion');
        }
    }
}

// Sécurité - Détection de tentatives d'injection
function sanitizeInput(input) {
    const dangerous = /<script|javascript:|on\w+=/i;
    return dangerous.test(input);
}

// Initialisation des fonctionnalités avancées
document.addEventListener('DOMContentLoaded', function() {
    setupAutoComplete();
    detectNetworkQuality();
    
    // Validation de sécurité sur tous les inputs
    document.querySelectorAll('input').forEach(input => {
        input.addEventListener('input', function() {
            if (sanitizeInput(this.value)) {
                console.warn('Tentative d\'injection détectée');
                this.value = this.value.replace(/<script.*?<\/script>/gi, '');
            }
        });
    });
});

// Gestion des erreurs globales
window.addEventListener('error', function(e) {
    console.error('Erreur JavaScript:', e.error);
});

// Gestionnaire pour les promesses rejetées
window.addEventListener('unhandledrejection', function(e) {
    console.error('Promesse rejetée:', e.reason);
    e.preventDefault();
});

// Exposer les fonctions nécessaires globalement
window.authUtils = {
    togglePassword,
    switchForm,
    validateField,
    clearFieldError,
    showFieldError
};