/**
 * Flying Book Analytics Tracker
 * Script à intégrer dans les présentations publiées pour collecter les analytics
 */

(function() {
    'use strict';

    // Configuration
    const ANALYTICS_ENDPOINT = '/flying-book/api/track.php';
    const SESSION_DURATION = 30 * 60 * 1000; // 30 minutes
    
    // Variables globales
    let sessionId = null;
    let projectId = null;
    let startTime = null;
    let currentSlide = 1;
    let maxSlideReached = 1;
    let slidesViewed = [];
    let totalSlides = 0;
    let interactions = [];
    let heartbeatInterval = null;
    let isActive = true;
    let lastActivity = Date.now();

    // Initialisation
    function init() {
        // Récupérer l'ID du projet depuis les méta-données
        const projectMeta = document.querySelector('meta[name="flying-book-project"]');
        if (!projectMeta) {
            console.warn('Flying Book Analytics: Project ID not found');
            return;
        }
        
        projectId = projectMeta.getAttribute('content');
        sessionId = generateSessionId();
        startTime = Date.now();
        
        // Détecter le nombre total de slides
        totalSlides = detectTotalSlides();
        
        // Collecter les informations de base
        collectDeviceInfo();
        
        // Démarrer le tracking
        startTracking();
        
        // Envoyer les données initiales
        sendAnalytics('page_view');
        
        console.log('Flying Book Analytics initialized for project:', projectId);
    }

    // Générer un ID de session unique
    function generateSessionId() {
        return 'fb_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
    }

    // Détecter le nombre total de slides
    function detectTotalSlides() {
        // Différentes méthodes selon le framework de présentation
        if (window.Reveal) {
            return window.Reveal.getTotalSlides();
        } else if (document.querySelectorAll('.slide').length) {
            return document.querySelectorAll('.slide').length;
        } else if (document.querySelectorAll('[data-slide]').length) {
            return document.querySelectorAll('[data-slide]').length;
        }
        return 1;
    }

    // Collecter les informations sur l'appareil
    function collectDeviceInfo() {
        return {
            screen_resolution: screen.width + 'x' + screen.height,
            viewport_size: window.innerWidth + 'x' + window.innerHeight,
            user_agent: navigator.userAgent,
            language: navigator.language,
            timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
            device_type: getDeviceType(),
            browser: getBrowserInfo(),
            os: getOSInfo(),
            referrer: document.referrer,
            url: window.location.href
        };
    }

    // Détecter le type d'appareil
    function getDeviceType() {
        const width = window.innerWidth;
        if (width <= 768) return 'mobile';
        if (width <= 1024) return 'tablet';
        return 'desktop';
    }

    // Détecter le navigateur
    function getBrowserInfo() {
        const ua = navigator.userAgent;
        if (ua.includes('Chrome')) return 'Chrome';
        if (ua.includes('Firefox')) return 'Firefox';
        if (ua.includes('Safari')) return 'Safari';
        if (ua.includes('Edge')) return 'Edge';
        return 'Other';
    }

    // Détecter l'OS
    function getOSInfo() {
        const ua = navigator.userAgent;
        if (ua.includes('Windows')) return 'Windows';
        if (ua.includes('Mac')) return 'macOS';
        if (ua.includes('Linux')) return 'Linux';
        if (ua.includes('Android')) return 'Android';
        if (ua.includes('iOS')) return 'iOS';
        return 'Other';
    }

    // Démarrer le tracking
    function startTracking() {
        // Tracking des changements de slides
        setupSlideTracking();
        
        // Tracking des interactions
        setupInteractionTracking();
        
        // Tracking de l'activité (focus/blur)
        setupActivityTracking();
        
        // Heartbeat pour maintenir la session
        startHeartbeat();
        
        // Événement avant fermeture
        window.addEventListener('beforeunload', handleBeforeUnload);
    }

    // Configuration du tracking des slides
    function setupSlideTracking() {
        // Pour Reveal.js
        if (window.Reveal) {
            window.Reveal.addEventListener('slidechanged', function(event) {
                currentSlide = event.indexh + 1;
                onSlideChange(currentSlide);
            });
        }
        
        // Pour d'autres frameworks - Observer les changements de slides
        const slideObserver = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.type === 'attributes' && 
                    (mutation.attributeName === 'class' || mutation.attributeName === 'data-slide-active')) {
                    const activeSlide = document.querySelector('.slide.active, [data-slide-active="true"]');
                    if (activeSlide) {
                        const slideNumber = parseInt(activeSlide.getAttribute('data-slide-number')) || 
                                          Array.from(activeSlide.parentNode.children).indexOf(activeSlide) + 1;
                        if (slideNumber !== currentSlide) {
                            currentSlide = slideNumber;
                            onSlideChange(currentSlide);
                        }
                    }
                }
            });
        });

        // Observer le container des slides
        const slideContainer = document.querySelector('.reveal, .slides-container, .presentation-container') || document.body;
        slideObserver.observe(slideContainer, {
            attributes: true,
            subtree: true,
            attributeFilter: ['class', 'data-slide-active']
        });
    }

    // Gestionnaire de changement de slide
    function onSlideChange(slideNumber) {
        // Mettre à jour les statistiques
        if (!slidesViewed.includes(slideNumber)) {
            slidesViewed.push(slideNumber);
        }
        
        if (slideNumber > maxSlideReached) {
            maxSlideReached = slideNumber;
        }
        
        // Enregistrer l'interaction
        recordInteraction('slide_change', {
            from_slide: currentSlide,
            to_slide: slideNumber,
            timestamp: Date.now()
        });
        
        lastActivity = Date.now();
    }

    // Configuration du tracking des interactions
    function setupInteractionTracking() {
        // Clics
        document.addEventListener('click', function(e) {
            recordInteraction('click', {
                element: e.target.tagName,
                classes: e.target.className,
                slide: currentSlide,
                timestamp: Date.now()
            });
            lastActivity = Date.now();
        });

        // Touches clavier
        document.addEventListener('keydown', function(e) {
            const navKeys = ['ArrowLeft', 'ArrowRight', 'ArrowUp', 'ArrowDown', 'Space', 'Enter'];
            if (navKeys.includes(e.key)) {
                recordInteraction('keyboard_nav', {
                    key: e.key,
                    slide: currentSlide,
                    timestamp: Date.now()
                });
                lastActivity = Date.now();
            }
        });

        // Scroll
        let scrollTimeout;
        document.addEventListener('scroll', function() {
            clearTimeout(scrollTimeout);
            scrollTimeout = setTimeout(() => {
                recordInteraction('scroll', {
                    slide: currentSlide,
                    timestamp: Date.now()
                });
                lastActivity = Date.now();
            }, 500);
        });
    }

    // Configuration du tracking d'activité
    function setupActivityTracking() {
        // Focus/Blur pour détecter quand l'utilisateur quitte/revient
        window.addEventListener('focus', function() {
            isActive = true;
            lastActivity = Date.now();
        });

        window.addEventListener('blur', function() {
            isActive = false;
        });

        // Mouvement de la souris
        document.addEventListener('mousemove', function() {
            lastActivity = Date.now();
        });
    }

    // Enregistrer une interaction
    function recordInteraction(type, data) {
        interactions.push({
            type: type,
            data: data,
            timestamp: Date.now()
        });

        // Limiter le nombre d'interactions stockées
        if (interactions.length > 100) {
            interactions = interactions.slice(-50);
        }
    }

    // Démarrer le heartbeat
    function startHeartbeat() {
        heartbeatInterval = setInterval(function() {
            // Envoyer un ping si l'utilisateur est actif
            if (isActive && (Date.now() - lastActivity) < 60000) { // Actif dans la dernière minute
                sendAnalytics('heartbeat');
            }
        }, 30000); // Toutes les 30 secondes
    }

    // Gestionnaire avant fermeture
    function handleBeforeUnload() {
        sendAnalytics('page_leave', {}, true); // Mode synchrone
    }

    // Envoyer les données analytics
    function sendAnalytics(eventType, additionalData = {}, synchronous = false) {
        const data = {
            project_id: projectId,
            session_id: sessionId,
            event_type: eventType,
            current_slide: currentSlide,
            max_slide_reached: maxSlideReached,
            slides_viewed: slidesViewed,
            total_slides: totalSlides,
            time_spent: Math.round((Date.now() - startTime) / 1000),
            completion_rate: Math.round((maxSlideReached / totalSlides) * 100),
            interactions: interactions.slice(-10), // Dernières 10 interactions
            device_info: collectDeviceInfo(),
            timestamp: new Date().toISOString(),
            ...additionalData
        };

        // Envoyer via fetch ou XMLHttpRequest selon le support
        if (synchronous) {
            // Mode synchrone pour beforeunload
            const xhr = new XMLHttpRequest();
            xhr.open('POST', ANALYTICS_ENDPOINT, false);
            xhr.setRequestHeader('Content-Type', 'application/json');
            try {
                xhr.send(JSON.stringify(data));
            } catch (e) {
                console.warn('Failed to send analytics:', e);
            }
        } else {
            // Mode asynchrone normal
            if (navigator.sendBeacon) {
                // Utiliser sendBeacon si disponible (plus fiable)
                navigator.sendBeacon(ANALYTICS_ENDPOINT, JSON.stringify(data));
            } else {
                // Fallback avec fetch
                fetch(ANALYTICS_ENDPOINT, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                }).catch(function(error) {
                    console.warn('Analytics request failed:', error);
                });
            }
        }
    }

    // API publique pour les présentations personnalisées
    window.FlyingBookAnalytics = {
        // Méthode pour signaler manuellement un changement de slide
        trackSlideChange: function(slideNumber) {
            onSlideChange(slideNumber);
        },
        
        // Méthode pour enregistrer un événement personnalisé
        trackEvent: function(eventName, data) {
            recordInteraction('custom_event', {
                event_name: eventName,
                data: data,
                slide: currentSlide,
                timestamp: Date.now()
            });
        },
        
        // Méthode pour mettre à jour le nombre total de slides
        setTotalSlides: function(total) {
            totalSlides = total;
        },
        
        // Obtenir les statistiques actuelles
        getStats: function() {
            return {
                session_id: sessionId,
                project_id: projectId,
                current_slide: currentSlide,
                max_slide_reached: maxSlideReached,
                slides_viewed: slidesViewed,
                total_slides: totalSlides,
                time_spent: Math.round((Date.now() - startTime) / 1000),
                completion_rate: Math.round((maxSlideReached / totalSlides) * 100),
                interactions_count: interactions.length
            };
        }
    };

    // Initialiser quand le DOM est prêt
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

})();