<?php
/**
 * Flying Book - Processeur PowerPoint
 * Extraction et conversion des fichiers PowerPoint
 */

require_once '../../env/.env.php';

/**
 * Traiter un fichier PowerPoint
 */
function processPowerPointFile($ppt_path, $project_id, $project_path, $pdo) {
    try {
        logError("Début du traitement PowerPoint", ['file' => $ppt_path, 'project_id' => $project_id]);
        
        // Détecter le type de fichier
        $file_extension = strtolower(pathinfo($ppt_path, PATHINFO_EXTENSION));
        
        switch ($file_extension) {
            case 'pptx':
                return processPPTX($ppt_path, $project_id, $project_path, $pdo);
            case 'ppt':
                return processPPT($ppt_path, $project_id, $project_path, $pdo);
            case 'odp':
                return processODP($ppt_path, $project_id, $project_path, $pdo);
            default:
                throw new Exception("Format de fichier non supporté: " . $file_extension);
        }
        
    } catch (Exception $e) {
        logError("Erreur traitement PowerPoint", ['error' => $e->getMessage()]);
        return [
            'success' => false,
            'message' => $e->getMessage()
        ];
    }
}

/**
 * Traiter un fichier PPTX (PowerPoint 2007+)
 */
function processPPTX($ppt_path, $project_id, $project_path, $pdo) {
    logError("Traitement PPTX", ['file' => $ppt_path]);
    
    // Créer un répertoire temporaire pour l'extraction
    $temp_dir = $project_path . '/temp/pptx_extract_' . time();
    mkdir($temp_dir, 0755, true);
    
    try {
        // Extraire le fichier PPTX (c'est un ZIP)
        $zip = new ZipArchive();
        if ($zip->open($ppt_path) !== TRUE) {
            throw new Exception("Impossible d'ouvrir le fichier PPTX");
        }
        
        $zip->extractTo($temp_dir);
        $zip->close();
        
        // Lire la structure du PowerPoint
        $slides_data = extractPPTXSlides($temp_dir);
        $images_data = extractPPTXImages($temp_dir, $project_path);
        
        // Créer les slides dans la base de données
        $slides_count = createSlidesFromData($slides_data, $images_data, $project_id, $pdo);
        
        // Nettoyer le répertoire temporaire
        deleteDirectory($temp_dir);
        
        // Mettre à jour le projet
        $stmt = $pdo->prepare("
            UPDATE flying_book_projects 
            SET total_slides = ?, status = 'draft', updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$slides_count, $project_id]);
        
        return [
            'success' => true,
            'slides_count' => $slides_count,
            'message' => "Import PPTX réussi"
        ];
        
    } catch (Exception $e) {
        // Nettoyer en cas d'erreur
        if (is_dir($temp_dir)) {
            deleteDirectory($temp_dir);
        }
        throw $e;
    }
}

/**
 * Extraire les slides d'un PPTX
 */
function extractPPTXSlides($temp_dir) {
    $slides_data = [];
    
    // Lire les slides dans ppt/slides/
    $slides_dir = $temp_dir . '/ppt/slides';
    if (!is_dir($slides_dir)) {
        throw new Exception("Structure PPTX invalide - répertoire slides non trouvé");
    }
    
    $slide_files = glob($slides_dir . '/slide*.xml');
    sort($slide_files, SORT_NATURAL);
    
    foreach ($slide_files as $slide_file) {
        $slide_number = (int) preg_replace('/.*slide(\d+)\.xml/', '$1', basename($slide_file));
        
        // Lire le contenu XML de la slide
        $xml_content = file_get_contents($slide_file);
        $xml = simplexml_load_string($xml_content);
        
        if ($xml === false) {
            logError("Erreur parsing XML slide", ['file' => $slide_file]);
            continue;
        }
        
        // Enregistrer les namespaces
        $namespaces = $xml->getNamespaces(true);
        
        // Extraire le texte de la slide
        $slide_text = extractTextFromPPTXSlide($xml, $namespaces);
        
        // Extraire les images de la slide
        $slide_images = extractImagesFromPPTXSlide($xml, $namespaces);
        
        $slides_data[] = [
            'number' => $slide_number,
            'title' => $slide_text['title'] ?? "Slide $slide_number",
            'content' => $slide_text['content'] ?? '',
            'notes' => '', // À implémenter: lecture des notes
            'images' => $slide_images
        ];
    }
    
    return $slides_data;
}

/**
 * Extraire le texte d'une slide PPTX
 */
function extractTextFromPPTXSlide($xml, $namespaces) {
    $text_data = ['title' => '', 'content' => ''];
    
    // Enregistrer les namespaces si pas déjà fait
    if (isset($namespaces['a'])) {
        $xml->registerXPathNamespace('a', $namespaces['a']);
    }
    if (isset($namespaces['p'])) {
        $xml->registerXPathNamespace('p', $namespaces['p']);
    }
    
    try {
        // Extraire tous les textes des shapes
        $text_elements = $xml->xpath('//a:t');
        $all_texts = [];
        
        if ($text_elements) {
            foreach ($text_elements as $text_element) {
                $text = trim((string) $text_element);
                if (!empty($text)) {
                    $all_texts[] = $text;
                }
            }
        }
        
        // Le premier texte significatif devient le titre
        if (!empty($all_texts)) {
            $text_data['title'] = $all_texts[0];
            
            // Le reste devient le contenu
            if (count($all_texts) > 1) {
                $text_data['content'] = implode("\n", array_slice($all_texts, 1));
            }
        }
        
    } catch (Exception $e) {
        logError("Erreur extraction texte PPTX", ['error' => $e->getMessage()]);
    }
    
    return $text_data;
}

/**
 * Extraire les images d'une slide PPTX
 */
function extractImagesFromPPTXSlide($xml, $namespaces) {
    $images = [];
    
    try {
        // Enregistrer les namespaces pour les relations
        if (isset($namespaces['r'])) {
            $xml->registerXPathNamespace('r', $namespaces['r']);
        }
        
        // Chercher les références d'images
        $image_refs = $xml->xpath('//a:blip/@r:embed');
        
        foreach ($image_refs as $ref) {
            $images[] = (string) $ref;
        }
        
    } catch (Exception $e) {
        logError("Erreur extraction images PPTX", ['error' => $e->getMessage()]);
    }
    
    return $images;
}

/**
 * Extraire et copier les images du PPTX
 */
function extractPPTXImages($temp_dir, $project_path) {
    $images_data = [];
    
    // Répertoire des médias dans le PPTX
    $media_dir = $temp_dir . '/ppt/media';
    if (!is_dir($media_dir)) {
        return $images_data; // Pas d'images
    }
    
    // Lister tous les fichiers image
    $image_files = glob($media_dir . '/*');
    
    foreach ($image_files as $image_file) {
        $filename = basename($image_file);
        $file_info = getimagesize($image_file);
        
        if ($file_info === false) {
            continue; // Pas une image valide
        }
        
        // Générer un nouveau nom de fichier
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $new_filename = 'ppt_import_' . uniqid() . '.' . $extension;
        
        // Copier vers le répertoire média du projet
        $target_path = $project_path . '/media/' . $new_filename;
        
        if (copy($image_file, $target_path)) {
            // Générer une miniature
            $thumbnail_filename = generateImageThumbnail($target_path, $project_path . '/thumbnails', $new_filename);
            
            $images_data[$filename] = [
                'original_name' => $filename,
                'stored_filename' => $new_filename,
                'file_path' => $target_path,
                'thumbnail' => $thumbnail_filename,
                'width' => $file_info[0],
                'height' => $file_info[1],
                'mime_type' => $file_info['mime']
            ];
        }
    }
    
    return $images_data;
}

/**
 * Créer les slides en base de données
 */
function createSlidesFromData($slides_data, $images_data, $project_id, $pdo) {
    $slides_count = 0;
    
    foreach ($slides_data as $slide_data) {
        // Déterminer le média principal de la slide
        $media_file = 'placeholder.jpg';
        $media_type = 'text_image';
        
        // Si la slide a des images, utiliser la première
        if (!empty($slide_data['images']) && !empty($images_data)) {
            foreach ($slide_data['images'] as $image_ref) {
                foreach ($images_data as $original_name => $image_info) {
                    if (strpos($original_name, $image_ref) !== false || strpos($image_ref, pathinfo($original_name, PATHINFO_FILENAME)) !== false) {
                        $media_file = $image_info['stored_filename'];
                        $media_type = 'image';
                        
                        // Enregistrer l'image dans la table des médias
                        $stmt = $pdo->prepare("
                            INSERT INTO flying_book_media 
                            (project_id, user_id, original_filename, stored_filename, file_path, 
                             file_type, mime_type, file_size, dimensions, thumbnail_path, upload_status, created_at)
                            VALUES (?, ?, ?, ?, ?, 'image', ?, ?, ?, ?, 'ready', NOW())
                        ");
                        
                        $file_size = file_exists($image_info['file_path']) ? filesize($image_info['file_path']) : 0;
                        $dimensions = $image_info['width'] . 'x' . $image_info['height'];
                        
                        $stmt->execute([
                            $project_id,
                            $_SESSION['user_id'],
                            $image_info['original_name'],
                            $image_info['stored_filename'],
                            $image_info['file_path'],
                            $image_info['mime_type'],
                            $file_size,
                            $dimensions,
                            $image_info['thumbnail']
                        ]);
                        
                        break 2; // Sortir des deux boucles
                    }
                }
            }
        }
        
        // Créer la slide
        $stmt = $pdo->prepare("
            INSERT INTO flying_book_slides 
            (project_id, slide_number, slide_title, media_type, media_file, 
             text_overlay, slide_notes, display_order, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $project_id,
            $slide_data['number'],
            $slide_data['title'],
            $media_type,
            $media_file,
            $slide_data['content'],
            $slide_data['notes'],
            $slide_data['number']
        ]);
        
        $slides_count++;
    }
    
    return $slides_count;
}

/**
 * Traiter un fichier PPT (PowerPoint 97-2003)
 */
function processPPT($ppt_path, $project_id, $project_path, $pdo) {
    // Pour les fichiers .ppt, on peut utiliser LibreOffice en ligne de commande
    // ou une bibliothèque PHP spécialisée
    
    // Vérifier si LibreOffice est disponible
    $libreoffice_path = trim(shell_exec('which libreoffice 2>/dev/null'));
    
    if (empty($libreoffice_path)) {
        // Essayer des chemins communs
        $common_paths = [
            '/usr/bin/libreoffice',
            '/usr/local/bin/libreoffice',
            '/opt/libreoffice/program/soffice',
            '/Applications/LibreOffice.app/Contents/MacOS/soffice'
        ];
        
        foreach ($common_paths as $path) {
            if (file_exists($path)) {
                $libreoffice_path = $path;
                break;
            }
        }
    }
    
    if (empty($libreoffice_path)) {
        throw new Exception("LibreOffice non trouvé. Installation requise pour traiter les fichiers .ppt");
    }
    
    // Convertir PPT en PPTX
    $temp_dir = $project_path . '/temp/ppt_convert_' . time();
    mkdir($temp_dir, 0755, true);
    
    try {
        // Commande de conversion
        $command = sprintf(
            '%s --headless --convert-to pptx --outdir %s %s 2>/dev/null',
            escapeshellarg($libreoffice_path),
            escapeshellarg($temp_dir),
            escapeshellarg($ppt_path)
        );
        
        exec($command, $output, $return_code);
        
        if ($return_code !== 0) {
            throw new Exception("Erreur lors de la conversion PPT vers PPTX");
        }
        
        // Chercher le fichier PPTX généré
        $pptx_files = glob($temp_dir . '/*.pptx');
        if (empty($pptx_files)) {
            throw new Exception("Fichier PPTX non généré lors de la conversion");
        }
        
        // Traiter le fichier PPTX converti
        $result = processPPTX($pptx_files[0], $project_id, $project_path, $pdo);
        
        // Nettoyer
        deleteDirectory($temp_dir);
        
        return $result;
        
    } catch (Exception $e) {
        if (is_dir($temp_dir)) {
            deleteDirectory($temp_dir);
        }
        throw $e;
    }
}

/**
 * Traiter un fichier ODP (OpenOffice/LibreOffice)
 */
function processODP($odp_path, $project_id, $project_path, $pdo) {
    // Similar à PPTX car ODP est aussi basé sur XML et ZIP
    
    $temp_dir = $project_path . '/temp/odp_extract_' . time();
    mkdir($temp_dir, 0755, true);
    
    try {
        // Extraire le fichier ODP (c'est un ZIP)
        $zip = new ZipArchive();
        if ($zip->open($odp_path) !== TRUE) {
            throw new Exception("Impossible d'ouvrir le fichier ODP");
        }
        
        $zip->extractTo($temp_dir);
        $zip->close();
        
        // Pour ODP, la structure est différente
        // content.xml contient le contenu principal
        $content_file = $temp_dir . '/content.xml';
        if (!file_exists($content_file)) {
            throw new Exception("Structure ODP invalide");
        }
        
        $slides_data = extractODPSlides($content_file);
        $images_data = extractODPImages($temp_dir, $project_path);
        
        $slides_count = createSlidesFromData($slides_data, $images_data, $project_id, $pdo);
        
        // Nettoyer
        deleteDirectory($temp_dir);
        
        $stmt = $pdo->prepare("
            UPDATE flying_book_projects 
            SET total_slides = ?, status = 'draft', updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$slides_count, $project_id]);
        
        return [
            'success' => true,
            'slides_count' => $slides_count,
            'message' => "Import ODP réussi"
        ];
        
    } catch (Exception $e) {
        if (is_dir($temp_dir)) {
            deleteDirectory($temp_dir);
        }
        throw $e;
    }
}

/**
 * Extraire les slides d'un fichier ODP
 */
function extractODPSlides($content_file) {
    $slides_data = [];
    
    $xml_content = file_get_contents($content_file);
    $xml = simplexml_load_string($xml_content);
    
    if ($xml === false) {
        throw new Exception("Impossible de parser le contenu XML ODP");
    }
    
    // Les namespaces pour ODP
    $namespaces = $xml->getNamespaces(true);
    
    // Extraire les pages de dessin (slides)
    if (isset($namespaces['draw'])) {
        $xml->registerXPathNamespace('draw', $namespaces['draw']);
    }
    if (isset($namespaces['text'])) {
        $xml->registerXPathNamespace('text', $namespaces['text']);
    }
    
    $pages = $xml->xpath('//draw:page');
    
    foreach ($pages as $index => $page) {
        $slide_number = $index + 1;
        
        // Extraire le texte de la page
        $text_elements = $page->xpath('.//text:p');
        $texts = [];
        
        if ($text_elements) {
            foreach ($text_elements as $text_element) {
                $text = trim((string) $text_element);
                if (!empty($text)) {
                    $texts[] = $text;
                }
            }
        }
        
        $slides_data[] = [
            'number' => $slide_number,
            'title' => isset($texts[0]) ? $texts[0] : "Slide $slide_number",
            'content' => count($texts) > 1 ? implode("\n", array_slice($texts, 1)) : '',
            'notes' => '',
            'images' => [] // À implémenter si nécessaire
        ];
    }
    
    return $slides_data;
}

/**
 * Extraire les images d'un fichier ODP
 */
function extractODPImages($temp_dir, $project_path) {
    $images_data = [];
    
    $pictures_dir = $temp_dir . '/Pictures';
    if (!is_dir($pictures_dir)) {
        return $images_data;
    }
    
    $image_files = glob($pictures_dir . '/*');
    
    foreach ($image_files as $image_file) {
        $filename = basename($image_file);
        $file_info = getimagesize($image_file);
        
        if ($file_info === false) {
            continue;
        }
        
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $new_filename = 'odp_import_' . uniqid() . '.' . $extension;
        $target_path = $project_path . '/media/' . $new_filename;
        
        if (copy($image_file, $target_path)) {
            $thumbnail_filename = generateImageThumbnail($target_path, $project_path . '/thumbnails', $new_filename);
            
            $images_data[$filename] = [
                'original_name' => $filename,
                'stored_filename' => $new_filename,
                'file_path' => $target_path,
                'thumbnail' => $thumbnail_filename,
                'width' => $file_info[0],
                'height' => $file_info[1],
                'mime_type' => $file_info['mime']
            ];
        }
    }
    
    return $images_data;
}

/**
 * Supprimer un répertoire récursivement
 */
function deleteDirectory($dir) {
    if (!is_dir($dir)) return;
    
    $files = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );
    
    foreach ($files as $file) {
        if ($file->isDir()) {
            rmdir($file->getRealPath());
        } else {
            unlink($file->getRealPath());
        }
    }
    
    rmdir($dir);
}
?>