<?php
/**
 * Flying Book Creator v2.0 - Collecteur Analytics
 * Collecte les données d'interaction en temps réel
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once './env/.env.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    $pdo = getDatabaseConnection();
    
    // Récupérer les données JSON
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!$data) {
        throw new Exception('Invalid JSON data');
    }
    
    // Valider les données requises
    $requiredFields = ['project_id', 'session_id', 'visitor_id', 'event'];
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            throw new Exception("Missing required field: $field");
        }
    }
    
    $projectId = intval($data['project_id']);
    $sessionId = $data['session_id'];
    $visitorId = $data['visitor_id'];
    $event = $data['event'];
    $eventData = $data['data'] ?? [];
    $timestamp = $data['timestamp'] ?? time() * 1000;
    $url = $data['url'] ?? '';
    
    // Obtenir informations visiteur
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $ipAddress = getClientIP();
    $deviceType = detectDeviceType($userAgent);
    $browser = detectBrowser($userAgent);
    $os = detectOS($userAgent);
    
    // Traiter selon le type d'événement
    switch ($event) {
        case 'slide_view':
            handleSlideView($pdo, $projectId, $sessionId, $visitorId, $eventData, $ipAddress, $userAgent, $deviceType, $browser, $os);
            break;
            
        case 'interaction':
            handleInteraction($pdo, $projectId, $sessionId, $visitorId, $eventData);
            break;
            
        case 'session_end':
            handleSessionEnd($pdo, $projectId, $sessionId, $visitorId, $eventData);
            break;
            
        default:
            // Événement personnalisé
            logCustomEvent($pdo, $projectId, $sessionId, $visitorId, $event, $eventData);
    }
    
    echo json_encode(['success' => true, 'timestamp' => date('Y-m-d H:i:s')]);
    
} catch (Exception $e) {
    logError("Analytics collector error", [
        'error' => $e->getMessage(),
        'data' => $data ?? null
    ]);
    
    http_response_code(400);
    echo json_encode(['error' => $e->getMessage()]);
}

/**
 * Gérer une vue de slide
 */
function handleSlideView($pdo, $projectId, $sessionId, $visitorId, $eventData, $ipAddress, $userAgent, $deviceType, $browser, $os) {
    $slideNumber = $eventData['slide'] ?? 1;
    $timestamp = date('Y-m-d H:i:s', ($eventData['timestamp'] ?? time() * 1000) / 1000);
    
    // Vérifier si le visiteur existe déjà
    $stmt = $pdo->prepare("
        SELECT id, max_slide_reached, slides_viewed, first_visit 
        FROM flying_book_analytics 
        WHERE project_id = ? AND session_id = ? AND visitor_ip = ?
    ");
    $stmt->execute([$projectId, $sessionId, $ipAddress]);
    $existing = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($existing) {
        // Mettre à jour l'enregistrement existant
        $slidesViewed = json_decode($existing['slides_viewed'] ?? '[]', true);
        if (!in_array($slideNumber, $slidesViewed)) {
            $slidesViewed[] = $slideNumber;
        }
        
        $maxSlideReached = max($existing['max_slide_reached'], $slideNumber);
        
        $stmt = $pdo->prepare("
            UPDATE flying_book_analytics SET
                max_slide_reached = ?,
                slides_viewed = ?,
                page_views = page_views + 1,
                last_visit = ?
            WHERE id = ?
        ");
        $stmt->execute([
            $maxSlideReached,
            json_encode($slidesViewed),
            $timestamp,
            $existing['id']
        ]);
        
    } else {
        // Créer un nouvel enregistrement
        $stmt = $pdo->prepare("
            INSERT INTO flying_book_analytics (
                project_id, visitor_ip, session_id, user_agent,
                page_views, slides_viewed, max_slide_reached,
                device_type, browser, os, first_visit, last_visit
            ) VALUES (?, ?, ?, ?, 1, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $projectId, $ipAddress, $sessionId, $userAgent,
            json_encode([$slideNumber]), $slideNumber,
            $deviceType, $browser, $os, $timestamp, $timestamp
        ]);
    }
}

/**
 * Gérer une interaction
 */
function handleInteraction($pdo, $projectId, $sessionId, $visitorId, $eventData) {
    $stmt = $pdo->prepare("
        SELECT id, interactions 
        FROM flying_book_analytics 
        WHERE project_id = ? AND session_id = ?
    ");
    $stmt->execute([$projectId, $sessionId]);
    $record = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($record) {
        $interactions = json_decode($record['interactions'] ?? '[]', true);
        $interactions[] = [
            'timestamp' => time(),
            'type' => $eventData['type'] ?? 'click',
            'x' => $eventData['x'] ?? 0,
            'y' => $eventData['y'] ?? 0,
            'element' => $eventData['element'] ?? '',
            'slide' => $eventData['slide'] ?? 1
        ];
        
        $stmt = $pdo->prepare("
            UPDATE flying_book_analytics 
            SET interactions = ? 
            WHERE id = ?
        ");
        $stmt->execute([json_encode($interactions), $record['id']]);
    }
}

/**
 * Gérer la fin de session
 */
function handleSessionEnd($pdo, $projectId, $sessionId, $visitorId, $eventData) {
    $totalTime = $eventData['totalTime'] ?? 0;
    $maxSlideReached = $eventData['maxSlideReached'] ?? 1;
    $slideViews = $eventData['slideViews'] ?? [];
    
    // Calculer le taux de completion
    $stmt = $pdo->prepare("SELECT COUNT(*) as total_slides FROM flying_book_slides WHERE project_id = ? AND is_active = 1");
    $stmt->execute([$projectId]);
    $totalSlides = $stmt->fetch(PDO::FETCH_ASSOC)['total_slides'] ?? 1;
    
    $completionRate = ($maxSlideReached / $totalSlides) * 100;
    $bounceRate = $maxSlideReached <= 1 ? 1 : 0;
    
    // Calculer le temps passé par slide
    $timePerSlide = [];
    foreach ($slideViews as $slide => $viewData) {
        if (isset($viewData['totalTime'])) {
            $timePerSlide[$slide] = $viewData['totalTime'];
        }
    }
    
    $stmt = $pdo->prepare("
        UPDATE flying_book_analytics SET
            time_spent = ?,
            completion_rate = ?,
            bounce_rate = ?,
            max_slide_reached = ?
        WHERE project_id = ? AND session_id = ?
    ");
    $stmt->execute([
        intval($totalTime / 1000), // Convertir ms en secondes
        $completionRate,
        $bounceRate,
        $maxSlideReached,
        $projectId,
        $sessionId
    ]);
}

/**
 * Logger un événement personnalisé
 */
function logCustomEvent($pdo, $projectId, $sessionId, $visitorId, $event, $eventData) {
    $stmt = $pdo->prepare("
        INSERT INTO flying_book_activity_logs (
            user_id, action, entity_type, entity_id, description, metadata
        ) VALUES (NULL, ?, 'project', ?, ?, ?)
    ");
    
    $stmt->execute([
        'custom_event_' . $event,
        $projectId,
        "Custom event: $event",
        json_encode([
            'session_id' => $sessionId,
            'visitor_id' => $visitorId,
            'event_data' => $eventData,
            'timestamp' => date('Y-m-d H:i:s')
        ])
    ]);
}

/**
 * Utilitaires de détection
 */
function getClientIP() {
    $ipKeys = ['HTTP_CF_CONNECTING_IP', 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR'];
    
    foreach ($ipKeys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

function detectDeviceType($userAgent) {
    if (preg_match('/mobile|android|iphone|ipad|phone/i', $userAgent)) {
        if (preg_match('/ipad|tablet/i', $userAgent)) {
            return 'tablet';
        }
        return 'mobile';
    }
    return 'desktop';
}

function detectBrowser($userAgent) {
    $browsers = [
        'Chrome' => '/chrome/i',
        'Firefox' => '/firefox/i',
        'Safari' => '/safari/i',
        'Edge' => '/edge/i',
        'Opera' => '/opera/i',
        'Internet Explorer' => '/msie/i'
    ];
    
    foreach ($browsers as $browser => $pattern) {
        if (preg_match($pattern, $userAgent)) {
            return $browser;
        }
    }
    
    return 'Unknown';
}

function detectOS($userAgent) {
    $os = [
        'Windows' => '/windows/i',
        'Mac' => '/macintosh|mac os x/i',
        'Linux' => '/linux/i',
        'iOS' => '/iphone|ipad/i',
        'Android' => '/android/i'
    ];
    
    foreach ($os as $osName => $pattern) {
        if (preg_match($pattern, $userAgent)) {
            return $osName;
        }
    }
    
    return 'Unknown';
}
?>