<?php
/**
 * Flying Book Creator v2.0 - Analytics Avancé avec IA
 * Heatmaps, prédictions d'abandon, analytics temps réel
 */

session_start();
require_once './env/.env.php';

// Vérification de l'authentification
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

try {
    $pdo = getDatabaseConnection();
    
    // ========== GESTION DES ACTIONS AJAX ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['action']) {
            case 'get_realtime_data':
                $projectId = intval($_POST['project_id'] ?? 0);
                $timeframe = $_POST['timeframe'] ?? '24h';
                
                echo json_encode(getRealtimeAnalytics($projectId, $timeframe, $pdo));
                exit;
                
            case 'get_heatmap_data':
                $projectId = intval($_POST['project_id'] ?? 0);
                $slideId = intval($_POST['slide_id'] ?? 0);
                
                echo json_encode(getHeatmapData($projectId, $slideId, $pdo));
                exit;
                
            case 'get_abandonment_prediction':
                $projectId = intval($_POST['project_id'] ?? 0);
                
                echo json_encode(getAbandonmentPrediction($projectId, $pdo));
                exit;
                
            case 'get_conversion_funnel':
                $projectId = intval($_POST['project_id'] ?? 0);
                
                echo json_encode(getConversionFunnel($projectId, $pdo));
                exit;
                
            case 'export_analytics':
                $projectId = intval($_POST['project_id'] ?? 0);
                $format = $_POST['format'] ?? 'json';
                
                echo json_encode(exportAnalytics($projectId, $format, $pdo));
                exit;
        }
    }
    
    // Récupérer les projets de l'utilisateur
    $stmt = $pdo->prepare("
        SELECT p.id, p.project_name, p.title, p.analytics_enabled,
               COUNT(DISTINCT a.id) as total_views,
               AVG(a.completion_rate) as avg_completion,
               COUNT(DISTINCT a.visitor_ip) as unique_visitors
        FROM flying_book_projects p
        LEFT JOIN flying_book_analytics a ON p.id = a.project_id
        WHERE p.user_id = ? AND p.status != 'deleted'
        GROUP BY p.id
        ORDER BY total_views DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Projet sélectionné
    $selectedProject = null;
    $selectedProjectId = intval($_GET['project'] ?? 0);
    if ($selectedProjectId) {
        $selectedProject = array_filter($projects, fn($p) => $p['id'] == $selectedProjectId)[0] ?? null;
    }
    
} catch (PDOException $e) {
    $error = "Erreur de base de données : " . $e->getMessage();
}

/**
 * Analytics temps réel
 */
function getRealtimeAnalytics($projectId, $timeframe, $pdo) {
    try {
        $timeCondition = getTimeCondition($timeframe);
        
        // Métriques principales
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(DISTINCT session_id) as sessions,
                COUNT(*) as page_views,
                COUNT(DISTINCT visitor_ip) as unique_visitors,
                AVG(completion_rate) as avg_completion,
                AVG(time_spent) as avg_time_spent,
                COUNT(CASE WHEN bounce_rate = 1 THEN 1 END) as bounces,
                COUNT(CASE WHEN completion_rate >= 80 THEN 1 END) as conversions
            FROM flying_book_analytics 
            WHERE project_id = ? AND $timeCondition
        ");
        
        $stmt->execute([$projectId]);
        $metrics = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Données temporelles pour graphiques
        $stmt = $pdo->prepare("
            SELECT 
                DATE_FORMAT(first_visit, '%H:00') as hour,
                COUNT(DISTINCT session_id) as sessions,
                COUNT(*) as views,
                AVG(completion_rate) as completion
            FROM flying_book_analytics 
            WHERE project_id = ? AND $timeCondition
            GROUP BY DATE_FORMAT(first_visit, '%H:00')
            ORDER BY hour
        ");
        
        $stmt->execute([$projectId]);
        $hourlyData = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Top pages/slides
        $stmt = $pdo->prepare("
            SELECT 
                slides_viewed,
                COUNT(*) as views,
                AVG(completion_rate) as avg_completion
            FROM flying_book_analytics 
            WHERE project_id = ? AND $timeCondition AND slides_viewed IS NOT NULL
            GROUP BY slides_viewed
            ORDER BY views DESC
            LIMIT 10
        ");
        
        $stmt->execute([$projectId]);
        $topSlides = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Données géographiques simplifiées
        $stmt = $pdo->prepare("
            SELECT 
                COALESCE(country, 'Unknown') as country,
                COUNT(DISTINCT visitor_ip) as visitors,
                AVG(completion_rate) as avg_completion
            FROM flying_book_analytics 
            WHERE project_id = ? AND $timeCondition
            GROUP BY country
            ORDER BY visitors DESC
            LIMIT 10
        ");
        
        $stmt->execute([$projectId]);
        $geoData = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return [
            'success' => true,
            'metrics' => $metrics,
            'hourly' => $hourlyData,
            'top_slides' => $topSlides,
            'geo_data' => $geoData,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Données heatmap
 */
function getHeatmapData($projectId, $slideId, $pdo) {
    try {
        // Interactions sur une slide spécifique
        $stmt = $pdo->prepare("
            SELECT 
                interactions,
                completion_rate,
                time_spent,
                device_type,
                screen_resolution
            FROM flying_book_analytics 
            WHERE project_id = ? AND JSON_CONTAINS(slides_viewed, CAST(? AS JSON))
            AND interactions IS NOT NULL
        ");
        
        $stmt->execute([$projectId, $slideId]);
        $rawData = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $heatmapPoints = [];
        $clickClusters = [];
        
        foreach ($rawData as $row) {
            $interactions = json_decode($row['interactions'], true);
            if (!$interactions) continue;
            
            foreach ($interactions as $interaction) {
                if (isset($interaction['x'], $interaction['y'], $interaction['type'])) {
                    $heatmapPoints[] = [
                        'x' => floatval($interaction['x']),
                        'y' => floatval($interaction['y']),
                        'type' => $interaction['type'],
                        'intensity' => $interaction['duration'] ?? 1,
                        'device' => $row['device_type'],
                        'time_spent' => $row['time_spent']
                    ];
                }
            }
        }
        
        // Générer clusters de clics
        $clickClusters = generateClickClusters($heatmapPoints);
        
        // Zones d'attention (où les utilisateurs passent le plus de temps)
        $attentionZones = generateAttentionZones($heatmapPoints);
        
        return [
            'success' => true,
            'heatmap_points' => $heatmapPoints,
            'click_clusters' => $clickClusters,
            'attention_zones' => $attentionZones,
            'total_interactions' => count($heatmapPoints),
            'slide_id' => $slideId
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Prédiction d'abandon avec IA basique
 */
function getAbandonmentPrediction($projectId, $pdo) {
    try {
        // Récupérer données historiques
        $stmt = $pdo->prepare("
            SELECT 
                max_slide_reached,
                completion_rate,
                time_spent,
                device_type,
                referrer,
                utm_source,
                bounce_rate,
                CASE 
                    WHEN completion_rate < 20 THEN 'high_abandon'
                    WHEN completion_rate < 50 THEN 'medium_abandon'
                    ELSE 'low_abandon'
                END as abandon_risk
            FROM flying_book_analytics 
            WHERE project_id = ?
            AND first_visit >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        ");
        
        $stmt->execute([$projectId]);
        $historicalData = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($historicalData)) {
            return [
                'success' => true,
                'prediction' => 'insufficient_data',
                'risk_score' => 0,
                'recommendations' => ['Collectez plus de données pour des prédictions précises']
            ];
        }
        
        // Analyse des patterns d'abandon
        $abandonmentPatterns = analyzeAbandonmentPatterns($historicalData);
        
        // Prédiction basée sur les patterns
        $riskScore = calculateRiskScore($abandonmentPatterns);
        
        // Recommandations d'amélioration
        $recommendations = generateRecommendations($abandonmentPatterns);
        
        return [
            'success' => true,
            'risk_score' => $riskScore,
            'prediction' => $riskScore > 70 ? 'high_risk' : ($riskScore > 40 ? 'medium_risk' : 'low_risk'),
            'patterns' => $abandonmentPatterns,
            'recommendations' => $recommendations,
            'confidence' => min(100, count($historicalData) * 2) // Confiance basée sur volume de données
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Funnel de conversion
 */
function getConversionFunnel($projectId, $pdo) {
    try {
        // Récupérer le nombre total de slides du projet
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as total_slides 
            FROM flying_book_slides 
            WHERE project_id = ? AND is_active = 1
        ");
        $stmt->execute([$projectId]);
        $totalSlides = $stmt->fetch(PDO::FETCH_ASSOC)['total_slides'];
        
        if ($totalSlides == 0) {
            return ['success' => false, 'error' => 'Aucune slide trouvée'];
        }
        
        // Analyser la progression slide par slide
        $funnelSteps = [];
        
        for ($slide = 1; $slide <= $totalSlides; $slide++) {
            $stmt = $pdo->prepare("
                SELECT 
                    COUNT(DISTINCT session_id) as visitors,
                    AVG(time_spent) as avg_time,
                    COUNT(CASE WHEN max_slide_reached >= ? THEN 1 END) as completed
                FROM flying_book_analytics 
                WHERE project_id = ? AND max_slide_reached >= ?
            ");
            
            $stmt->execute([$slide, $projectId, $slide]);
            $stepData = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $funnelSteps[] = [
                'step' => $slide,
                'step_name' => "Slide $slide",
                'visitors' => intval($stepData['visitors']),
                'avg_time' => floatval($stepData['avg_time']),
                'drop_off_rate' => $slide > 1 ? 
                    (($funnelSteps[$slide-2]['visitors'] - $stepData['visitors']) / $funnelSteps[$slide-2]['visitors']) * 100 : 0
            ];
        }
        
        // Calculer taux de conversion global
        $totalVisitors = $funnelSteps[0]['visitors'] ?? 0;
        $completedVisitors = end($funnelSteps)['visitors'] ?? 0;
        $overallConversion = $totalVisitors > 0 ? ($completedVisitors / $totalVisitors) * 100 : 0;
        
        return [
            'success' => true,
            'funnel_steps' => $funnelSteps,
            'overall_conversion' => $overallConversion,
            'total_visitors' => $totalVisitors,
            'completed_visitors' => $completedVisitors,
            'biggest_drop_off' => findBiggestDropOff($funnelSteps)
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Fonctions utilitaires
 */
function getTimeCondition($timeframe) {
    switch ($timeframe) {
        case '1h': return "first_visit >= DATE_SUB(NOW(), INTERVAL 1 HOUR)";
        case '24h': return "first_visit >= DATE_SUB(NOW(), INTERVAL 24 HOUR)";
        case '7d': return "first_visit >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
        case '30d': return "first_visit >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
        default: return "first_visit >= DATE_SUB(NOW(), INTERVAL 24 HOUR)";
    }
}

function generateClickClusters($points) {
    $clusters = [];
    $clusterRadius = 50; // pixels
    
    foreach ($points as $point) {
        if ($point['type'] !== 'click') continue;
        
        $addedToCluster = false;
        
        foreach ($clusters as &$cluster) {
            $distance = sqrt(
                pow($point['x'] - $cluster['center_x'], 2) + 
                pow($point['y'] - $cluster['center_y'], 2)
            );
            
            if ($distance <= $clusterRadius) {
                $cluster['points'][] = $point;
                $cluster['count']++;
                $cluster['intensity'] += $point['intensity'];
                
                // Recalculer le centre
                $cluster['center_x'] = array_sum(array_column($cluster['points'], 'x')) / count($cluster['points']);
                $cluster['center_y'] = array_sum(array_column($cluster['points'], 'y')) / count($cluster['points']);
                
                $addedToCluster = true;
                break;
            }
        }
        
        if (!$addedToCluster) {
            $clusters[] = [
                'center_x' => $point['x'],
                'center_y' => $point['y'],
                'count' => 1,
                'intensity' => $point['intensity'],
                'points' => [$point]
            ];
        }
    }
    
    return array_filter($clusters, fn($c) => $c['count'] >= 3); // Minimum 3 clics pour former un cluster
}

function generateAttentionZones($points) {
    $zones = [];
    $gridSize = 100; // pixels
    
    // Créer une grille d'attention
    for ($x = 0; $x < 1200; $x += $gridSize) {
        for ($y = 0; $y < 800; $y += $gridSize) {
            $zoneIntensity = 0;
            $pointCount = 0;
            
            foreach ($points as $point) {
                if ($point['x'] >= $x && $point['x'] < $x + $gridSize &&
                    $point['y'] >= $y && $point['y'] < $y + $gridSize) {
                    $zoneIntensity += $point['time_spent'] ?? 1;
                    $pointCount++;
                }
            }
            
            if ($pointCount > 0) {
                $zones[] = [
                    'x' => $x,
                    'y' => $y,
                    'width' => $gridSize,
                    'height' => $gridSize,
                    'intensity' => $zoneIntensity / $pointCount,
                    'point_count' => $pointCount
                ];
            }
        }
    }
    
    // Trier par intensité et garder les zones les plus chaudes
    usort($zones, fn($a, $b) => $b['intensity'] <=> $a['intensity']);
    
    return array_slice($zones, 0, 20); // Top 20 zones chaudes
}

function analyzeAbandonmentPatterns($data) {
    $patterns = [
        'device_abandon' => [],
        'slide_abandon' => [],
        'time_abandon' => [],
        'source_abandon' => []
    ];
    
    // Analyser par device
    $deviceGroups = [];
    foreach ($data as $row) {
        $device = $row['device_type'] ?? 'unknown';
        if (!isset($deviceGroups[$device])) {
            $deviceGroups[$device] = [];
        }
        $deviceGroups[$device][] = $row;
    }
    
    foreach ($deviceGroups as $device => $rows) {
        $highAbandon = count(array_filter($rows, fn($r) => $r['abandon_risk'] === 'high_abandon'));
        $total = count($rows);
        $patterns['device_abandon'][$device] = $total > 0 ? ($highAbandon / $total) * 100 : 0;
    }
    
    // Analyser par slide
    $slideGroups = [];
    foreach ($data as $row) {
        $slide = $row['max_slide_reached'] ?? 1;
        if (!isset($slideGroups[$slide])) {
            $slideGroups[$slide] = [];
        }
        $slideGroups[$slide][] = $row;
    }
    
    foreach ($slideGroups as $slide => $rows) {
        $highAbandon = count(array_filter($rows, fn($r) => $r['abandon_risk'] === 'high_abandon'));
        $total = count($rows);
        $patterns['slide_abandon'][$slide] = $total > 0 ? ($highAbandon / $total) * 100 : 0;
    }
    
    return $patterns;
}

function calculateRiskScore($patterns) {
    $totalRisk = 0;
    $factorCount = 0;
    
    // Risque par device
    if (!empty($patterns['device_abandon'])) {
        $avgDeviceRisk = array_sum($patterns['device_abandon']) / count($patterns['device_abandon']);
        $totalRisk += $avgDeviceRisk;
        $factorCount++;
    }
    
    // Risque par slide
    if (!empty($patterns['slide_abandon'])) {
        $avgSlideRisk = array_sum($patterns['slide_abandon']) / count($patterns['slide_abandon']);
        $totalRisk += $avgSlideRisk;
        $factorCount++;
    }
    
    return $factorCount > 0 ? $totalRisk / $factorCount : 0;
}

function generateRecommendations($patterns) {
    $recommendations = [];
    
    // Recommandations basées sur les devices
    if (!empty($patterns['device_abandon'])) {
        $worstDevice = array_keys($patterns['device_abandon'], max($patterns['device_abandon']))[0];
        if ($patterns['device_abandon'][$worstDevice] > 50) {
            $recommendations[] = "Optimisez l'expérience pour $worstDevice - taux d'abandon élevé";
        }
    }
    
    // Recommandations basées sur les slides
    if (!empty($patterns['slide_abandon'])) {
        $worstSlide = array_keys($patterns['slide_abandon'], max($patterns['slide_abandon']))[0];
        if ($patterns['slide_abandon'][$worstSlide] > 60) {
            $recommendations[] = "Révisez la slide $worstSlide - point d'abandon critique";
        }
    }
    
    // Recommandations génériques
    $recommendations[] = "Ajoutez des éléments interactifs pour maintenir l'engagement";
    $recommendations[] = "Raccourcissez les slides trop longues";
    $recommendations[] = "Testez différents appels à l'action";
    
    return array_slice($recommendations, 0, 5); // Max 5 recommandations
}

function findBiggestDropOff($steps) {
    $biggestDropOff = ['step' => 1, 'rate' => 0];
    
    foreach ($steps as $step) {
        if ($step['drop_off_rate'] > $biggestDropOff['rate']) {
            $biggestDropOff = [
                'step' => $step['step'],
                'rate' => $step['drop_off_rate']
            ];
        }
    }
    
    return $biggestDropOff;
}

function exportAnalytics($projectId, $format, $pdo) {
    try {
        $realtimeData = getRealtimeAnalytics($projectId, '30d', $pdo);
        $funnelData = getConversionFunnel($projectId, $pdo);
        $predictionData = getAbandonmentPrediction($projectId, $pdo);
        
        $exportData = [
            'project_id' => $projectId,
            'export_date' => date('Y-m-d H:i:s'),
            'realtime_analytics' => $realtimeData,
            'conversion_funnel' => $funnelData,
            'abandonment_prediction' => $predictionData
        ];
        
        return [
            'success' => true,
            'data' => $exportData,
            'format' => $format
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Analytics v2.0 - Intelligence Artificielle</title>
    <link rel="stylesheet" href="assets/css/united.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/4.4.0/chart.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/heatmap.js/2.0.5/heatmap.min.js"></script>
    <style>
        /* ========== ANALYTICS ADVANCED STYLES ========== */
        .analytics-container {
            display: grid;
            grid-template-columns: 250px 1fr;
            height: calc(100vh - 70px);
            gap: 1rem;
            padding: 1rem;
        }
        
        .analytics-sidebar {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            overflow-y: auto;
        }
        
        .analytics-main {
            background: #f8fafc;
            border-radius: 12px;
            padding: 1.5rem;
            overflow-y: auto;
        }
        
        .project-selector {
            margin-bottom: 2rem;
        }
        
        .project-selector select {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--gray-300);
            border-radius: 8px;
            background: white;
            font-weight: 500;
        }
        
        .metrics-sidebar {
            margin-bottom: 2rem;
        }
        
        .metric-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0.75rem;
            background: var(--gray-50);
            border-radius: 6px;
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }
        
        .metric-value {
            font-weight: 700;
            color: var(--primary-color);
        }
        
        .analytics-tabs {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 2rem;
            border-bottom: 1px solid var(--gray-200);
        }
        
        .analytics-tab {
            padding: 1rem 1.5rem;
            background: none;
            border: none;
            border-bottom: 3px solid transparent;
            cursor: pointer;
            font-weight: 500;
            color: var(--gray-600);
            transition: all 0.3s ease;
        }
        
        .analytics-tab.active {
            color: var(--primary-color);
            border-bottom-color: var(--primary-color);
        }
        
        .analytics-tab:hover:not(.active) {
            color: var(--gray-800);
            background: var(--gray-50);
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .analytics-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .analytics-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
        }
        
        .analytics-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.1);
        }
        
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 0.75rem;
            border-bottom: 1px solid var(--gray-100);
        }
        
        .card-title {
            font-size: 1.1rem;
            font-weight: 700;
            color: var(--gray-800);
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .card-title i {
            color: var(--primary-color);
        }
        
        .chart-container {
            position: relative;
            height: 300px;
            margin-bottom: 1rem;
        }
        
        .chart-container canvas {
            max-height: 100%;
        }
        
        .heatmap-container {
            position: relative;
            width: 100%;
            height: 400px;
            background: var(--gray-100);
            border-radius: 8px;
            overflow: hidden;
        }
        
        .heatmap-overlay {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
        }
        
        .prediction-card {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
        }
        
        .prediction-card .card-header {
            border-bottom-color: rgba(255, 255, 255, 0.2);
        }
        
        .risk-score {
            text-align: center;
            margin: 2rem 0;
        }
        
        .risk-score-circle {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            background: conic-gradient(
                var(--accent-color) 0deg,
                var(--accent-color) calc(var(--score, 0) * 3.6deg),
                rgba(255, 255, 255, 0.2) calc(var(--score, 0) * 3.6deg),
                rgba(255, 255, 255, 0.2) 360deg
            );
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1rem;
            position: relative;
        }
        
        .risk-score-inner {
            width: 90px;
            height: 90px;
            background: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--gray-800);
        }
        
        .recommendations-list {
            list-style: none;
            padding: 0;
        }
        
        .recommendations-list li {
            padding: 0.75rem;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 6px;
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }
        
        .recommendations-list li:before {
            content: "💡";
            margin-right: 0.5rem;
        }
        
        .funnel-visualization {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin: 2rem 0;
        }
        
        .funnel-step {
            background: var(--primary-color);
            color: white;
            padding: 1rem;
            border-radius: 8px;
            text-align: center;
            min-width: 100px;
            position: relative;
            margin-right: 1rem;
        }
        
        .funnel-step:not(:last-child):after {
            content: "→";
            position: absolute;
            right: -1.5rem;
            top: 50%;
            transform: translateY(-50%);
            color: var(--gray-600);
            font-size: 1.5rem;
        }
        
        .funnel-step-number {
            font-size: 1.5rem;
            font-weight: 700;
            display: block;
        }
        
        .funnel-step-label {
            font-size: 0.8rem;
            opacity: 0.9;
        }
        
        .funnel-step.critical {
            background: var(--error-color);
            animation: pulse 2s infinite;
        }
        
        .realtime-indicator {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--success-color);
            font-size: 0.8rem;
            font-weight: 500;
        }
        
        .realtime-dot {
            width: 8px;
            height: 8px;
            background: var(--success-color);
            border-radius: 50%;
            animation: pulse 2s infinite;
        }
        
        .export-controls {
            display: flex;
            gap: 0.5rem;
            margin-top: 1rem;
        }
        
        .export-btn {
            padding: 0.5rem 1rem;
            background: var(--gray-100);
            border: 1px solid var(--gray-300);
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.8rem;
            transition: all 0.3s ease;
        }
        
        .export-btn:hover {
            background: var(--primary-color);
            color: white;
            border-color: var(--primary-color);
        }
        
        .insights-list {
            list-style: none;
            padding: 0;
        }
        
        .insights-list li {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem;
            border-radius: 6px;
            margin-bottom: 0.5rem;
            background: var(--gray-50);
            font-size: 0.9rem;
        }
        
        .insight-icon {
            width: 24px;
            height: 24px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.8rem;
        }
        
        .insight-positive .insight-icon {
            background: var(--success-color);
            color: white;
        }
        
        .insight-warning .insight-icon {
            background: var(--warning-color);
            color: white;
        }
        
        .insight-negative .insight-icon {
            background: var(--error-color);
            color: white;
        }
        
        .time-selector {
            display: flex;
            gap: 0.25rem;
            margin-bottom: 1rem;
        }
        
        .time-btn {
            padding: 0.5rem 0.75rem;
            background: var(--gray-100);
            border: 1px solid var(--gray-300);
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.8rem;
            transition: all 0.3s ease;
        }
        
        .time-btn.active {
            background: var(--primary-color);
            color: white;
            border-color: var(--primary-color);
        }
        
        /* Responsive */
        @media (max-width: 1024px) {
            .analytics-container {
                grid-template-columns: 1fr;
                grid-template-rows: auto 1fr;
            }
            
            .analytics-sidebar {
                max-height: 200px;
            }
            
            .analytics-grid {
                grid-template-columns: 1fr;
            }
        }
        
        /* Animations */
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 2px solid var(--gray-300);
            border-top: 2px solid var(--primary-color);
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <div class="logo">
                    <i class="fas fa-chart-line"></i>
                    <span>Analytics IA v2.0</span>
                </div>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="projects.php" class="nav-link">
                        <i class="fas fa-folder"></i> Projets
                    </a>
                    <a href="analytics_advanced.php" class="nav-link active">
                        <i class="fas fa-chart-line"></i> Analytics IA
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <div class="realtime-indicator">
                    <div class="realtime-dot"></div>
                    <span>Temps réel</span>
                </div>
                <button class="btn btn-outline" onclick="refreshAllData()">
                    <i class="fas fa-sync-alt"></i> Actualiser
                </button>
                <button class="btn btn-primary" onclick="exportReport()">
                    <i class="fas fa-download"></i> Export
                </button>
            </div>
        </div>
    </header>

    <main class="main-content">
        <div class="analytics-container">
            <!-- Sidebar -->
            <div class="analytics-sidebar">
                <div class="project-selector">
                    <h3><i class="fas fa-project-diagram"></i> Projet</h3>
                    <select id="projectSelector" onchange="loadProject(this.value)">
                        <option value="">Sélectionner un projet...</option>
                        <?php foreach ($projects as $project): ?>
                            <option value="<?= $project['id'] ?>" <?= $selectedProject && $selectedProject['id'] == $project['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($project['title'] ?: $project['project_name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <?php if ($selectedProject): ?>
                <div class="metrics-sidebar">
                    <h4>Métriques Rapides</h4>
                    <div class="metric-item">
                        <span>Vues totales</span>
                        <span class="metric-value" id="quickViews"><?= number_format($selectedProject['total_views']) ?></span>
                    </div>
                    <div class="metric-item">
                        <span>Visiteurs uniques</span>
                        <span class="metric-value" id="quickVisitors"><?= number_format($selectedProject['unique_visitors']) ?></span>
                    </div>
                    <div class="metric-item">
                        <span>Taux de completion</span>
                        <span class="metric-value" id="quickCompletion"><?= number_format($selectedProject['avg_completion'], 1) ?>%</span>
                    </div>
                </div>
                
                <div class="time-selector">
                    <button class="time-btn active" data-timeframe="24h">24h</button>
                    <button class="time-btn" data-timeframe="7d">7j</button>
                    <button class="time-btn" data-timeframe="30d">30j</button>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Contenu principal -->
            <div class="analytics-main">
                <?php if (!$selectedProject): ?>
                    <div style="text-align: center; padding: 4rem; color: var(--gray-500);">
                        <i class="fas fa-chart-line" style="font-size: 4rem; margin-bottom: 2rem;"></i>
                        <h2>Analytics Avancé avec IA</h2>
                        <p>Sélectionnez un projet pour voir ses analytics détaillés avec prédictions d'abandon et heatmaps.</p>
                    </div>
                <?php else: ?>
                
                <!-- Tabs -->
                <div class="analytics-tabs">
                    <button class="analytics-tab active" data-tab="overview">
                        <i class="fas fa-tachometer-alt"></i> Vue d'ensemble
                    </button>
                    <button class="analytics-tab" data-tab="heatmaps">
                        <i class="fas fa-fire"></i> Heatmaps
                    </button>
                    <button class="analytics-tab" data-tab="predictions">
                        <i class="fas fa-brain"></i> Prédictions IA
                    </button>
                    <button class="analytics-tab" data-tab="funnel">
                        <i class="fas fa-filter"></i> Funnel
                    </button>
                    <button class="analytics-tab" data-tab="insights">
                        <i class="fas fa-lightbulb"></i> Insights
                    </button>
                </div>
                
                <!-- Tab: Vue d'ensemble -->
                <div id="overview" class="tab-content active">
                    <div class="analytics-grid">
                        <!-- Graphique temps réel -->
                        <div class="analytics-card">
                            <div class="card-header">
                                <div class="card-title">
                                    <i class="fas fa-chart-area"></i>
                                    Activité temps réel
                                </div>
                                <div class="realtime-indicator">
                                    <div class="realtime-dot"></div>
                                    <span>Live</span>
                                </div>
                            </div>
                            <div class="chart-container">
                                <canvas id="realtimeChart"></canvas>
                            </div>
                        </div>
                        
                        <!-- Top slides -->
                        <div class="analytics-card">
                            <div class="card-header">
                                <div class="card-title">
                                    <i class="fas fa-trophy"></i>
                                    Top Slides
                                </div>
                            </div>
                            <div id="topSlidesContainer">
                                <div class="loading-spinner"></div>
                            </div>
                        </div>
                        
                        <!-- Données géographiques -->
                        <div class="analytics-card">
                            <div class="card-header">
                                <div class="card-title">
                                    <i class="fas fa-globe"></i>
                                    Répartition géographique
                                </div>
                            </div>
                            <div class="chart-container">
                                <canvas id="geoChart"></canvas>
                            </div>
                        </div>
                        
                        <!-- Insights automatiques -->
                        <div class="analytics-card">
                            <div class="card-header">
                                <div class="card-title">
                                    <i class="fas fa-robot"></i>
                                    Insights IA
                                </div>
                            </div>
                            <ul class="insights-list" id="autoInsights">
                                <li class="insight-positive">
                                    <div class="insight-icon">✓</div>
                                    <span>Analyse en cours...</span>
                                </li>
                            </ul>
                        </div>
                    </div>
                </div>
                
                <!-- Tab: Heatmaps -->
                <div id="heatmaps" class="tab-content">
                    <div class="analytics-grid">
                        <div class="analytics-card" style="grid-column: 1 / -1;">
                            <div class="card-header">
                                <div class="card-title">
                                    <i class="fas fa-fire"></i>
                                    Heatmap des interactions
                                </div>
                                <select id="slideSelector" onchange="loadHeatmap(this.value)">
                                    <option value="">Sélectionner une slide...</option>
                                </select>
                            </div>
                            <div class="heatmap-container" id="heatmapContainer">
                                <div style="display: flex; align-items: center; justify-content: center; height: 100%; color: var(--gray-500);">
                                    <div>
                                        <i class="fas fa-fire" style="font-size: 3rem; margin-bottom: 1rem;"></i>
                                        <p>Sélectionnez une slide pour voir sa heatmap</p>
                                    </analysis>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Tab: Prédictions IA -->
                <div id="predictions" class="tab-content">
                    <div class="analytics-grid">
                        <div class="analytics-card prediction-card">
                            <div class="card-header">
                                <div class="card-title">
                                    <i class="fas fa-brain"></i>
                                    Prédiction d'abandon
                                </div>
                            </div>
                            <div class="risk-score">
                                <div class="risk-score-circle" style="--score: 0" id="riskScoreCircle">
                                    <div class="risk-score-inner" id="riskScoreValue">-%</div>
                                </div>
                                <div id="riskLevel">Analyse en cours...</div>
                            </div>
                            <ul class="recommendations-list" id="recommendations">
                                <li>Chargement des recommandations...</li>
                            </ul>
                        </div>
                        
                        <div class="analytics-card">
                            <div class="card-header">
                                <div class="card-title">
                                    <i class="fas fa-chart-line"></i>
                                    Patterns d'abandon
                                </div>
                            </div>
                            <div class="chart-container">
                                <canvas id="abandonmentChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Tab: Funnel -->
                <div id="funnel" class="tab-content">
                    <div class="analytics-card">
                        <div class="card-header">
                            <div class="card-title">
                                <i class="fas fa-filter"></i>
                                Funnel de conversion
                            </div>
                        </div>
                        <div class="funnel-visualization" id="funnelVisualization">
                            <div class="loading-spinner"></div>
                        </div>
                        <div class="chart-container">
                            <canvas id="funnelChart"></canvas>
                        </div>
                    </div>
                </div>
                
                <!-- Tab: Insights -->
                <div id="insights" class="tab-content">
                    <div class="analytics-grid">
                        <div class="analytics-card">
                            <div class="card-header">
                                <div class="card-title">
                                    <i class="fas fa-lightbulb"></i>
                                    Recommandations d'optimisation
                                </div>
                            </div>
                            <ul class="insights-list" id="optimizationInsights">
                                <li>Génération des insights en cours...</li>
                            </ul>
                        </div>
                        
                        <div class="analytics-card">
                            <div class="card-header">
                                <div class="card-title">
                                    <i class="fas fa-exclamation-triangle"></i>
                                    Points d'attention
                                </div>
                            </div>
                            <ul class="insights-list" id="attentionPoints">
                                <li>Analyse des points critiques...</li>
                            </ul>
                        </div>
                    </div>
                </div>
                
                <?php endif; ?>
            </div>
        </div>
    </main>

    <script>
        // ========== ANALYTICS ADVANCED ENGINE ==========
        
        class AdvancedAnalytics {
            constructor() {
                this.currentProject = <?= json_encode($selectedProject) ?>;
                this.currentTimeframe = '24h';
                this.charts = {};
                this.heatmapInstance = null;
                this.refreshInterval = null;
                
                this.init();
            }
            
            init() {
                if (this.currentProject) {
                    this.setupEventListeners();
                    this.loadAllData();
                    this.startRealTimeUpdates();
                }
            }
            
            setupEventListeners() {
                // Tabs
                document.querySelectorAll('.analytics-tab').forEach(tab => {
                    tab.addEventListener('click', (e) => {
                        this.switchTab(tab.dataset.tab);
                    });
                });
                
                // Time selector
                document.querySelectorAll('.time-btn').forEach(btn => {
                    btn.addEventListener('click', (e) => {
                        document.querySelector('.time-btn.active').classList.remove('active');
                        btn.classList.add('active');
                        this.currentTimeframe = btn.dataset.timeframe;
                        this.loadRealtimeData();
                    });
                });
            }
            
            switchTab(tabName) {
                // Changer l'onglet actif
                document.querySelectorAll('.analytics-tab').forEach(tab => {
                    tab.classList.remove('active');
                });
                document.querySelector(`[data-tab="${tabName}"]`).classList.add('active');
                
                // Changer le contenu
                document.querySelectorAll('.tab-content').forEach(content => {
                    content.classList.remove('active');
                });
                document.getElementById(tabName).classList.add('active');
                
                // Charger les données spécifiques
                switch (tabName) {
                    case 'heatmaps':
                        this.loadHeatmapTab();
                        break;
                    case 'predictions':
                        this.loadPredictions();
                        break;
                    case 'funnel':
                        this.loadFunnelData();
                        break;
                    case 'insights':
                        this.generateInsights();
                        break;
                }
            }
            
            async loadAllData() {
                try {
                    await Promise.all([
                        this.loadRealtimeData(),
                        this.loadPredictions(),
                        this.loadFunnelData()
                    ]);
                } catch (error) {
                    console.error('Erreur chargement données:', error);
                    this.showNotification('Erreur lors du chargement', 'error');
                }
            }
            
            async loadRealtimeData() {
                try {
                    const response = await fetch('analytics_advanced.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                        body: `action=get_realtime_data&project_id=${this.currentProject.id}&timeframe=${this.currentTimeframe}`
                    });
                    
                    const result = await response.json();
                    
                    if (result.success) {
                        this.updateRealtimeCharts(result);
                        this.updateQuickMetrics(result.metrics);
                        this.generateAutoInsights(result);
                    }
                } catch (error) {
                    console.error('Erreur données temps réel:', error);
                }
            }
            
            updateRealtimeCharts(data) {
                // Graphique temps réel
                const ctx = document.getElementById('realtimeChart');
                if (ctx) {
                    if (this.charts.realtime) {
                        this.charts.realtime.destroy();
                    }
                    
                    this.charts.realtime = new Chart(ctx, {
                        type: 'line',
                        data: {
                            labels: data.hourly.map(h => h.hour),
                            datasets: [
                                {
                                    label: 'Sessions',
                                    data: data.hourly.map(h => h.sessions),
                                    borderColor: '#667eea',
                                    backgroundColor: 'rgba(102, 126, 234, 0.1)',
                                    fill: true,
                                    tension: 0.4
                                },
                                {
                                    label: 'Vues',
                                    data: data.hourly.map(h => h.views),
                                    borderColor: '#f093fb',
                                    backgroundColor: 'rgba(240, 147, 251, 0.1)',
                                    fill: true,
                                    tension: 0.4
                                }
                            ]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    position: 'top'
                                }
                            },
                            scales: {
                                y: {
                                    beginAtZero: true
                                }
                            }
                        }
                    });
                }
                
                // Graphique géographique
                const geoCtx = document.getElementById('geoChart');
                if (geoCtx && data.geo_data.length > 0) {
                    if (this.charts.geo) {
                        this.charts.geo.destroy();
                    }
                    
                    this.charts.geo = new Chart(geoCtx, {
                        type: 'doughnut',
                        data: {
                            labels: data.geo_data.map(g => g.country || 'Inconnu'),
                            datasets: [{
                                data: data.geo_data.map(g => g.visitors),
                                backgroundColor: [
                                    '#667eea', '#f093fb', '#764ba2', '#4facfe',
                                    '#00f2fe', '#43e97b', '#38f9d7', '#ffecd2'
                                ]
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    position: 'bottom'
                                }
                            }
                        }
                    });
                }
                
                // Top slides
                this.updateTopSlides(data.top_slides);
            }
            
            updateTopSlides(slides) {
                const container = document.getElementById('topSlidesContainer');
                
                if (slides.length === 0) {
                    container.innerHTML = '<p style="color: var(--gray-500); text-align: center;">Aucune donnée de slides</p>';
                    return;
                }
                
                let html = '';
                slides.forEach((slide, index) => {
                    const slideData = JSON.parse(slide.slides_viewed || '[]');
                    const slideNumber = slideData[0] || (index + 1);
                    
                    html += `
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.75rem; background: var(--gray-50); border-radius: 6px; margin-bottom: 0.5rem;">
                            <div>
                                <strong>Slide ${slideNumber}</strong>
                                <div style="font-size: 0.8rem; color: var(--gray-600);">
                                    ${slide.views} vues • ${parseFloat(slide.avg_completion).toFixed(1)}% completion
                                </div>
                            </div>
                            <div style="font-size: 1.2rem; color: var(--primary-color);">#${index + 1}</div>
                        </div>
                    `;
                });
                
                container.innerHTML = html;
            }
            
            updateQuickMetrics(metrics) {
                if (document.getElementById('quickViews')) {
                    document.getElementById('quickViews').textContent = Number(metrics.page_views || 0).toLocaleString();
                }
                if (document.getElementById('quickVisitors')) {
                    document.getElementById('quickVisitors').textContent = Number(metrics.unique_visitors || 0).toLocaleString();
                }
                if (document.getElementById('quickCompletion')) {
                    document.getElementById('quickCompletion').textContent = parseFloat(metrics.avg_completion || 0).toFixed(1) + '%';
                }
            }
            
            generateAutoInsights(data) {
                const container = document.getElementById('autoInsights');
                const insights = [];
                
                // Analyser les métriques
                const bounceRate = (data.metrics.bounces / data.metrics.sessions) * 100;
                const conversionRate = (data.metrics.conversions / data.metrics.sessions) * 100;
                
                if (bounceRate > 70) {
                    insights.push({
                        type: 'negative',
                        icon: '⚠️',
                        text: `Taux de rebond élevé: ${bounceRate.toFixed(1)}%`
                    });
                } else if (bounceRate < 30) {
                    insights.push({
                        type: 'positive',
                        icon: '✅',
                        text: `Excellent taux de rebond: ${bounceRate.toFixed(1)}%`
                    });
                }
                
                if (conversionRate > 15) {
                    insights.push({
                        type: 'positive',
                        icon: '🎯',
                        text: `Très bon taux de conversion: ${conversionRate.toFixed(1)}%`
                    });
                } else if (conversionRate < 5) {
                    insights.push({
                        type: 'warning',
                        icon: '📈',
                        text: `Taux de conversion à améliorer: ${conversionRate.toFixed(1)}%`
                    });
                }
                
                // Analyser la progression temporelle
                const hourlyGrowth = this.calculateGrowthTrend(data.hourly);
                if (hourlyGrowth > 0) {
                    insights.push({
                        type: 'positive',
                        icon: '📊',
                        text: `Tendance positive: +${hourlyGrowth.toFixed(1)}% de croissance`
                    });
                }
                
                // Device insights
                if (data.geo_data.length > 0) {
                    const topCountry = data.geo_data[0];
                    insights.push({
                        type: 'info',
                        icon: '🌍',
                        text: `Principal marché: ${topCountry.country} (${topCountry.visitors} visiteurs)`
                    });
                }
                
                // Afficher les insights
                let html = '';
                insights.forEach(insight => {
                    html += `
                        <li class="insight-${insight.type}">
                            <div class="insight-icon">${insight.icon}</div>
                            <span>${insight.text}</span>
                        </li>
                    `;
                });
                
                if (html) {
                    container.innerHTML = html;
                } else {
                    container.innerHTML = '<li><div class="insight-icon">ℹ️</div><span>Collecte de plus de données nécessaire</span></li>';
                }
            }
            
            calculateGrowthTrend(hourlyData) {
                if (hourlyData.length < 3) return 0;
                
                const recent = hourlyData.slice(-3);
                const older = hourlyData.slice(0, 3);
                
                const recentSum = recent.reduce((sum, h) => sum + parseInt(h.sessions), 0);
                const olderSum = older.reduce((sum, h) => sum + parseInt(h.sessions), 0);
                
                if (olderSum === 0) return 0;
                
                return ((recentSum - olderSum) / olderSum) * 100;
            }
            
            async loadPredictions() {
                try {
                    const response = await fetch('analytics_advanced.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                        body: `action=get_abandonment_prediction&project_id=${this.currentProject.id}`
                    });
                    
                    const result = await response.json();
                    
                    if (result.success) {
                        this.updatePredictionUI(result);
                    }
                } catch (error) {
                    console.error('Erreur prédictions:', error);
                }
            }
            
            updatePredictionUI(prediction) {
                // Score de risque
                const riskScore = Math.round(prediction.risk_score || 0);
                const riskCircle = document.getElementById('riskScoreCircle');
                const riskValue = document.getElementById('riskScoreValue');
                const riskLevel = document.getElementById('riskLevel');
                
                if (riskCircle) {
                    riskCircle.style.setProperty('--score', riskScore);
                }
                if (riskValue) {
                    riskValue.textContent = riskScore + '%';
                }
                if (riskLevel) {
                    const level = prediction.prediction === 'high_risk' ? 'Risque Élevé' :
                                  prediction.prediction === 'medium_risk' ? 'Risque Modéré' : 'Risque Faible';
                    riskLevel.textContent = level;
                }
                
                // Recommandations
                const recommendations = document.getElementById('recommendations');
                if (recommendations && prediction.recommendations) {
                    let html = '';
                    prediction.recommendations.forEach(rec => {
                        html += `<li>${rec}</li>`;
                    });
                    recommendations.innerHTML = html;
                }
                
                // Graphique des patterns
                this.updateAbandonmentChart(prediction.patterns);
            }
            
            updateAbandonmentChart(patterns) {
                const ctx = document.getElementById('abandonmentChart');
                if (!ctx || !patterns) return;
                
                if (this.charts.abandonment) {
                    this.charts.abandonment.destroy();
                }
                
                // Préparer les données des devices
                const deviceLabels = Object.keys(patterns.device_abandon || {});
                const deviceData = Object.values(patterns.device_abandon || {});
                
                this.charts.abandonment = new Chart(ctx, {
                    type: 'bar',
                    data: {
                        labels: deviceLabels,
                        datasets: [{
                            label: 'Taux d\'abandon (%)',
                            data: deviceData,
                            backgroundColor: [
                                '#ef4444', '#f59e0b', '#10b981', '#3b82f6'
                            ],
                            border
borderColor: [
                               '#dc2626', '#d97706', '#059669', '#2563eb'
                           ],
                           borderWidth: 1
                       }]
                   },
                   options: {
                       responsive: true,
                       maintainAspectRatio: false,
                       plugins: {
                           legend: {
                               display: false
                           },
                           tooltip: {
                               callbacks: {
                                   label: function(context) {
                                       return `${context.parsed.y.toFixed(1)}% d'abandon`;
                                   }
                               }
                           }
                       },
                       scales: {
                           y: {
                               beginAtZero: true,
                               max: 100,
                               ticks: {
                                   callback: function(value) {
                                       return value + '%';
                                   }
                               }
                           }
                       }
                   }
               });
           }
           
           async loadFunnelData() {
               try {
                   const response = await fetch('analytics_advanced.php', {
                       method: 'POST',
                       headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                       body: `action=get_conversion_funnel&project_id=${this.currentProject.id}`
                   });
                   
                   const result = await response.json();
                   
                   if (result.success) {
                       this.updateFunnelVisualization(result);
                       this.updateFunnelChart(result);
                   }
               } catch (error) {
                   console.error('Erreur funnel:', error);
               }
           }
           
           updateFunnelVisualization(funnelData) {
               const container = document.getElementById('funnelVisualization');
               
               if (!funnelData.funnel_steps || funnelData.funnel_steps.length === 0) {
                   container.innerHTML = '<p style="color: var(--gray-500);">Aucune donnée de funnel disponible</p>';
                   return;
               }
               
               let html = '';
               const steps = funnelData.funnel_steps.slice(0, 6); // Max 6 étapes pour l'affichage
               
               steps.forEach((step, index) => {
                   const isCritical = step.drop_off_rate > 30; // Seuil critique
                   
                   html += `
                       <div class="funnel-step ${isCritical ? 'critical' : ''}" title="Drop-off: ${step.drop_off_rate.toFixed(1)}%">
                           <span class="funnel-step-number">${step.visitors.toLocaleString()}</span>
                           <span class="funnel-step-label">${step.step_name}</span>
                       </div>
                   `;
               });
               
               // Ajouter les statistiques globales
               html += `
                   <div style="margin-top: 2rem; padding: 1rem; background: var(--gray-50); border-radius: 8px;">
                       <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 1rem; text-align: center;">
                           <div>
                               <strong style="display: block; font-size: 1.5rem; color: var(--primary-color);">
                                   ${funnelData.overall_conversion.toFixed(1)}%
                               </strong>
                               <span style="font-size: 0.8rem; color: var(--gray-600);">Conversion globale</span>
                           </div>
                           <div>
                               <strong style="display: block; font-size: 1.5rem; color: var(--success-color);">
                                   ${funnelData.completed_visitors.toLocaleString()}
                               </strong>
                               <span style="font-size: 0.8rem; color: var(--gray-600);">Ont terminé</span>
                           </div>
                           <div>
                               <strong style="display: block; font-size: 1.5rem; color: var(--error-color);">
                                   Slide ${funnelData.biggest_drop_off.step}
                               </strong>
                               <span style="font-size: 0.8rem; color: var(--gray-600);">Plus gros abandon</span>
                           </div>
                       </div>
                   </div>
               `;
               
               container.innerHTML = html;
           }
           
           updateFunnelChart(funnelData) {
               const ctx = document.getElementById('funnelChart');
               if (!ctx || !funnelData.funnel_steps) return;
               
               if (this.charts.funnel) {
                   this.charts.funnel.destroy();
               }
               
               this.charts.funnel = new Chart(ctx, {
                   type: 'line',
                   data: {
                       labels: funnelData.funnel_steps.map(step => step.step_name),
                       datasets: [
                           {
                               label: 'Visiteurs',
                               data: funnelData.funnel_steps.map(step => step.visitors),
                               borderColor: '#667eea',
                               backgroundColor: 'rgba(102, 126, 234, 0.1)',
                               fill: true,
                               tension: 0.4
                           },
                           {
                               label: 'Taux d\'abandon (%)',
                               data: funnelData.funnel_steps.map(step => step.drop_off_rate),
                               borderColor: '#ef4444',
                               backgroundColor: 'rgba(239, 68, 68, 0.1)',
                               fill: false,
                               yAxisID: 'y1'
                           }
                       ]
                   },
                   options: {
                       responsive: true,
                       maintainAspectRatio: false,
                       interaction: {
                           mode: 'index',
                           intersect: false,
                       },
                       plugins: {
                           legend: {
                               position: 'top'
                           }
                       },
                       scales: {
                           x: {
                               display: true,
                               title: {
                                   display: true,
                                   text: 'Étapes du parcours'
                               }
                           },
                           y: {
                               type: 'linear',
                               display: true,
                               position: 'left',
                               title: {
                                   display: true,
                                   text: 'Nombre de visiteurs'
                               }
                           },
                           y1: {
                               type: 'linear',
                               display: true,
                               position: 'right',
                               title: {
                                   display: true,
                                   text: 'Taux d\'abandon (%)'
                               },
                               grid: {
                                   drawOnChartArea: false,
                               },
                           }
                       }
                   }
               });
           }
           
           loadHeatmapTab() {
               // Charger la liste des slides pour le sélecteur
               this.loadSlideSelector();
           }
           
           loadSlideSelector() {
               // Ici on devrait récupérer les slides du projet
               // Pour la démo, on utilise des slides fictives
               const slideSelector = document.getElementById('slideSelector');
               let html = '<option value="">Sélectionner une slide...</option>';
               
               for (let i = 1; i <= 5; i++) {
                   html += `<option value="${i}">Slide ${i}</option>`;
               }
               
               slideSelector.innerHTML = html;
           }
           
           async loadHeatmap(slideId) {
               if (!slideId) return;
               
               try {
                   const response = await fetch('analytics_advanced.php', {
                       method: 'POST',
                       headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                       body: `action=get_heatmap_data&project_id=${this.currentProject.id}&slide_id=${slideId}`
                   });
                   
                   const result = await response.json();
                   
                   if (result.success) {
                       this.renderHeatmap(result);
                   }
               } catch (error) {
                   console.error('Erreur heatmap:', error);
               }
           }
           
           renderHeatmap(heatmapData) {
               const container = document.getElementById('heatmapContainer');
               
               if (!heatmapData.heatmap_points || heatmapData.heatmap_points.length === 0) {
                   container.innerHTML = `
                       <div style="display: flex; align-items: center; justify-content: center; height: 100%; color: var(--gray-500);">
                           <div style="text-align: center;">
                               <i class="fas fa-chart-area" style="font-size: 3rem; margin-bottom: 1rem;"></i>
                               <p>Aucune donnée d'interaction pour cette slide</p>
                           </div>
                       </div>
                   `;
                   return;
               }
               
               // Initialiser heatmap.js
               if (this.heatmapInstance) {
                   this.heatmapInstance.setData({
                       max: 0,
                       data: []
                   });
               } else {
                   this.heatmapInstance = h337.create({
                       container: container,
                       maxOpacity: 0.8,
                       minOpacity: 0.1,
                       blur: 0.85,
                       gradient: {
                           '0.0': 'blue',
                           '0.3': 'cyan',
                           '0.5': 'lime',
                           '0.7': 'yellow',
                           '1.0': 'red'
                       }
                   });
               }
               
               // Préparer les données pour heatmap.js
               const maxIntensity = Math.max(...heatmapData.heatmap_points.map(p => p.intensity));
               const heatmapPoints = heatmapData.heatmap_points.map(point => ({
                   x: Math.round(point.x * container.offsetWidth / 100), // Convertir % en pixels
                   y: Math.round(point.y * container.offsetHeight / 100),
                   value: point.intensity
               }));
               
               this.heatmapInstance.setData({
                   max: maxIntensity,
                   data: heatmapPoints
               });
               
               // Ajouter les clusters de clics
               this.renderClickClusters(heatmapData.click_clusters, container);
           }
           
           renderClickClusters(clusters, container) {
               // Supprimer les anciens clusters
               container.querySelectorAll('.click-cluster').forEach(el => el.remove());
               
               if (!clusters || clusters.length === 0) return;
               
               clusters.forEach(cluster => {
                   const clusterEl = document.createElement('div');
                   clusterEl.className = 'click-cluster';
                   clusterEl.style.cssText = `
                       position: absolute;
                       left: ${cluster.center_x * container.offsetWidth / 100}px;
                       top: ${cluster.center_y * container.offsetHeight / 100}px;
                       width: 30px;
                       height: 30px;
                       background: rgba(255, 255, 255, 0.9);
                       border: 2px solid #667eea;
                       border-radius: 50%;
                       display: flex;
                       align-items: center;
                       justify-content: center;
                       font-weight: bold;
                       font-size: 0.8rem;
                       color: #667eea;
                       transform: translate(-50%, -50%);
                       z-index: 10;
                       cursor: pointer;
                   `;
                   clusterEl.textContent = cluster.count;
                   clusterEl.title = `${cluster.count} clics dans cette zone`;
                   
                   container.appendChild(clusterEl);
               });
           }
           
           generateInsights() {
               // Générer des insights d'optimisation
               const optimizationInsights = [
                   {
                       type: 'positive',
                       icon: '🎯',
                       text: 'Ajoutez plus d\'éléments interactifs sur les slides avec fort engagement'
                   },
                   {
                       type: 'warning',
                       icon: '⚡',
                       text: 'Réduisez le temps de chargement - impact sur l\'abandon'
                   },
                   {
                       type: 'info',
                       icon: '📱',
                       text: 'Optimisez l\'expérience mobile - 60% de votre trafic'
                   },
                   {
                       type: 'negative',
                       icon: '🔍',
                       text: 'Slide 3 a un taux d\'abandon critique - à revoir'
                   }
               ];
               
               const attentionPoints = [
                   {
                       type: 'warning',
                       icon: '⚠️',
                       text: 'Pics d\'abandon après 2 minutes de visionnage'
                   },
                   {
                       type: 'negative',
                       icon: '📉',
                       text: 'Conversion mobile 40% plus faible que desktop'
                   },
                   {
                       type: 'info',
                       icon: 'ℹ️',
                       text: 'Trafic principal entre 14h-16h - optimisez pour cette période'
                   }
               ];
               
               this.renderInsightsList('optimizationInsights', optimizationInsights);
               this.renderInsightsList('attentionPoints', attentionPoints);
           }
           
           renderInsightsList(containerId, insights) {
               const container = document.getElementById(containerId);
               let html = '';
               
               insights.forEach(insight => {
                   html += `
                       <li class="insight-${insight.type}">
                           <div class="insight-icon">${insight.icon}</div>
                           <span>${insight.text}</span>
                       </li>
                   `;
               });
               
               container.innerHTML = html;
           }
           
           startRealTimeUpdates() {
               // Mise à jour toutes les 30 secondes
               this.refreshInterval = setInterval(() => {
                   if (document.querySelector('.analytics-tab[data-tab="overview"]').classList.contains('active')) {
                       this.loadRealtimeData();
                   }
               }, 30000);
           }
           
           stopRealTimeUpdates() {
               if (this.refreshInterval) {
                   clearInterval(this.refreshInterval);
                   this.refreshInterval = null;
               }
           }
           
           showNotification(message, type = 'info') {
               // Supprimer notifications existantes
               document.querySelectorAll('.notification').forEach(n => n.remove());
               
               const notification = document.createElement('div');
               notification.className = `notification ${type}`;
               notification.style.cssText = `
                   position: fixed;
                   top: 2rem;
                   right: 2rem;
                   background: white;
                   padding: 1rem 1.5rem;
                   border-radius: 8px;
                   box-shadow: 0 8px 25px rgba(0,0,0,0.15);
                   z-index: 2000;
                   border-left: 4px solid ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#667eea'};
                   transform: translateX(400px);
                   transition: transform 0.3s ease;
               `;
               
               notification.innerHTML = `
                   <div style="display: flex; align-items: center; gap: 0.5rem;">
                       <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
                       ${message}
                   </div>
               `;
               
               document.body.appendChild(notification);
               
               // Animation d'apparition
               setTimeout(() => {
                   notification.style.transform = 'translateX(0)';
               }, 100);
               
               // Auto-suppression
               setTimeout(() => {
                   notification.style.transform = 'translateX(400px)';
                   setTimeout(() => notification.remove(), 300);
               }, 4000);
           }
       }
       
       // ========== FONCTIONS GLOBALES ==========
       
       // Initialiser les analytics
       const Analytics = new AdvancedAnalytics();
       
       // Charger un projet
       function loadProject(projectId) {
           if (projectId) {
               window.location.href = `analytics_advanced.php?project=${projectId}`;
           }
       }
       
       // Charger heatmap pour une slide
       function loadHeatmap(slideId) {
           Analytics.loadHeatmap(slideId);
       }
       
       // Actualiser toutes les données
       function refreshAllData() {
           Analytics.loadAllData();
           Analytics.showNotification('Données actualisées', 'success');
       }
       
       // Exporter le rapport
       async function exportReport() {
           if (!Analytics.currentProject) {
               Analytics.showNotification('Aucun projet sélectionné', 'error');
               return;
           }
           
           try {
               const response = await fetch('analytics_advanced.php', {
                   method: 'POST',
                   headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                   body: `action=export_analytics&project_id=${Analytics.currentProject.id}&format=json`
               });
               
               const result = await response.json();
               
               if (result.success) {
                   // Créer et télécharger le fichier
                   const blob = new Blob([JSON.stringify(result.data, null, 2)], { 
                       type: 'application/json' 
                   });
                   
                   const url = URL.createObjectURL(blob);
                   const a = document.createElement('a');
                   a.href = url;
                   a.download = `analytics_report_${Analytics.currentProject.id}_${new Date().toISOString().split('T')[0]}.json`;
                   document.body.appendChild(a);
                   a.click();
                   document.body.removeChild(a);
                   URL.revokeObjectURL(url);
                   
                   Analytics.showNotification('Rapport exporté avec succès !', 'success');
               } else {
                   Analytics.showNotification('Erreur lors de l\'export: ' + result.error, 'error');
               }
           } catch (error) {
               Analytics.showNotification('Erreur de connexion', 'error');
           }
       }
       
       // Nettoyage lors de la fermeture
       window.addEventListener('beforeunload', () => {
           Analytics.stopRealTimeUpdates();
       });
       
       // Message de bienvenue
       document.addEventListener('DOMContentLoaded', function() {
           <?php if ($selectedProject): ?>
               Analytics.showNotification(`Analytics IA chargé pour: ${Analytics.currentProject.title || Analytics.currentProject.project_name}`, 'success');
           <?php else: ?>
               Analytics.showNotification('Sélectionnez un projet pour voir ses analytics !', 'info');
           <?php endif; ?>
       });
   </script>
</body>
</html>