<?php
session_start();
require_once './env/.env.php';

// Vérification admin STRICT
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$message = '';
$error = '';

// Fonctions utilitaires
function getDatabaseConnection() {
    try {
        return new PDO(
            "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET,
            DB_USER,
            DB_PASS,
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
    } catch (PDOException $e) {
        throw new Exception("Erreur de connexion à la base de données");
    }
}

function formatFileSize($bytes) {
    if ($bytes == 0) return '0 B';
    $k = 1024;
    $sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
    $i = floor(log($bytes) / log($k));
    return round($bytes / pow($k, $i), 2) . ' ' . $sizes[$i];
}

function timeAgo($datetime) {
    $time = time() - strtotime($datetime);
    if ($time < 60) return 'À l\'instant';
    if ($time < 3600) return floor($time/60) . ' min';
    if ($time < 86400) return floor($time/3600) . ' h';
    if ($time < 2592000) return floor($time/86400) . ' j';
    return date('d/m/Y', strtotime($datetime));
}

try {
    $pdo = getDatabaseConnection();
    
    // ========== GESTION DES ACTIONS AJAX ==========
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax_action'])) {
        header('Content-Type: application/json');
        
        switch ($_POST['ajax_action']) {
            case 'get_system_stats':
                // Statistiques système avancées
                $stats = [];
                
                // Stats base de données
                $result = $pdo->query("SELECT COUNT(*) as total FROM users WHERE status != 'deleted'")->fetch();
                $stats['total_users'] = $result['total'];
                
                $result = $pdo->query("SELECT COUNT(*) as total FROM users WHERE status = 'active'")->fetch();
                $stats['active_users'] = $result['total'];
                
                $result = $pdo->query("SELECT COUNT(*) as total FROM flying_book_projects WHERE status != 'deleted'")->fetch();
                $stats['total_projects'] = $result['total'];
                
                $result = $pdo->query("SELECT COUNT(*) as total FROM flying_book_media WHERE upload_status = 'ready'")->fetch();
                $stats['total_media'] = $result['total'];
                
                $result = $pdo->query("SELECT COALESCE(SUM(file_size), 0) as total FROM flying_book_media WHERE upload_status = 'ready'")->fetch();
                $stats['total_storage'] = $result['total'];
                
                // Stats serveur
                $stats['php_version'] = PHP_VERSION;
                $stats['memory_usage'] = memory_get_usage(true);
                $stats['memory_limit'] = ini_get('memory_limit');
                $stats['upload_max'] = ini_get('upload_max_filesize');
                $stats['server_load'] = function_exists('sys_getloadavg') ? sys_getloadavg()[0] : 'N/A';
                
                // Espace disque
                $stats['disk_free'] = disk_free_space('/');
                $stats['disk_total'] = disk_total_space('/');
                $stats['disk_used'] = $stats['disk_total'] - $stats['disk_free'];
                
                echo json_encode(['success' => true, 'stats' => $stats]);
                exit;
                
            case 'manage_user':
                $user_target_id = (int)$_POST['user_id'];
                $action = $_POST['user_action'];
                
                if ($user_target_id === $user_id) {
                    echo json_encode(['success' => false, 'message' => 'Impossible de se modifier soi-même']);
                    exit;
                }
                
                switch ($action) {
                    case 'activate':
                        $stmt = $pdo->prepare("UPDATE users SET status = 'active' WHERE id = ?");
                        break;
                    case 'deactivate':
                        $stmt = $pdo->prepare("UPDATE users SET status = 'inactive' WHERE id = ?");
                        break;
                    case 'delete':
                        $stmt = $pdo->prepare("UPDATE users SET status = 'deleted', deleted_at = NOW() WHERE id = ?");
                        break;
                    case 'make_admin':
                        $stmt = $pdo->prepare("UPDATE users SET role = 'admin' WHERE id = ?");
                        break;
                    case 'remove_admin':
                        $stmt = $pdo->prepare("UPDATE users SET role = 'user' WHERE id = ?");
                        break;
                    default:
                        echo json_encode(['success' => false, 'message' => 'Action invalide']);
                        exit;
                }
                
                if ($stmt->execute([$user_target_id])) {
                    echo json_encode(['success' => true, 'message' => 'Utilisateur mis à jour']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erreur lors de la mise à jour']);
                }
                exit;
                
            case 'cleanup_system':
                $cleanup_type = $_POST['cleanup_type'];
                $deleted_count = 0;
                
                try {
                    $pdo->beginTransaction();
                    
                    switch ($cleanup_type) {
                        case 'draft_projects':
                            $stmt = $pdo->prepare("SELECT id FROM flying_book_projects WHERE status = 'draft' AND updated_at < DATE_SUB(NOW(), INTERVAL 30 DAY)");
                            $stmt->execute();
                            $old_drafts = $stmt->fetchAll(PDO::FETCH_COLUMN);
                            
                            if (!empty($old_drafts)) {
                                $placeholders = implode(',', array_fill(0, count($old_drafts), '?'));
                                $pdo->prepare("DELETE FROM flying_book_slides WHERE project_id IN ($placeholders)")->execute($old_drafts);
                                $pdo->prepare("DELETE FROM flying_book_analytics WHERE project_id IN ($placeholders)")->execute($old_drafts);
                                $stmt = $pdo->prepare("DELETE FROM flying_book_projects WHERE id IN ($placeholders)");
                                $stmt->execute($old_drafts);
                                $deleted_count = $stmt->rowCount();
                            }
                            break;
                            
                        case 'orphan_media':
                            $stmt = $pdo->query("
                                DELETE FROM flying_book_media 
                                WHERE user_id NOT IN (SELECT id FROM users WHERE status != 'deleted')
                            ");
                            $deleted_count = $stmt->rowCount();
                            break;
                            
                        case 'old_analytics':
                            $stmt = $pdo->prepare("DELETE FROM flying_book_analytics WHERE first_visit < DATE_SUB(NOW(), INTERVAL 1 YEAR)");
                            $stmt->execute();
                            $deleted_count = $stmt->rowCount();
                            break;
                            
                        case 'temp_files':
                            $temp_dir = $_SERVER['DOCUMENT_ROOT'] . '/flying-book/uploads/temp/';
                            if (is_dir($temp_dir)) {
                                $files = glob($temp_dir . '*');
                                foreach ($files as $file) {
                                    if (filemtime($file) < strtotime('-1 day')) {
                                        unlink($file);
                                        $deleted_count++;
                                    }
                                }
                            }
                            break;
                    }
                    
                    $pdo->commit();
                    echo json_encode(['success' => true, 'message' => "$deleted_count éléments supprimés"]);
                } catch (Exception $e) {
                    $pdo->rollback();
                    echo json_encode(['success' => false, 'message' => 'Erreur: ' . $e->getMessage()]);
                }
                exit;
                
            case 'get_error_logs':
                $logs = [];
                $log_file = $_SERVER['DOCUMENT_ROOT'] . '/flying-book/logs/error.log';
                
                if (file_exists($log_file)) {
                    $lines = file($log_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                    $logs = array_slice(array_reverse($lines), 0, 50); // 50 dernières lignes
                }
                
                echo json_encode(['success' => true, 'logs' => $logs]);
                exit;
        }
        
        echo json_encode(['success' => false, 'message' => 'Action non reconnue']);
        exit;
    }
    
    // ========== RÉCUPÉRATION DES DONNÉES ADMIN ==========
    
    // Statistiques globales
    $global_stats = $pdo->query("
        SELECT 
            (SELECT COUNT(*) FROM users WHERE status != 'deleted') as total_users,
            (SELECT COUNT(*) FROM users WHERE status = 'active') as active_users,
            (SELECT COUNT(*) FROM users WHERE role = 'admin') as admin_users,
            (SELECT COUNT(*) FROM flying_book_projects WHERE status != 'deleted') as total_projects,
            (SELECT COUNT(*) FROM flying_book_projects WHERE status = 'published') as published_projects,
            (SELECT COUNT(*) FROM flying_book_media WHERE upload_status = 'ready') as total_media,
            (SELECT COALESCE(SUM(file_size), 0) FROM flying_book_media WHERE upload_status = 'ready') as total_storage,
            (SELECT COUNT(*) FROM flying_book_analytics WHERE first_visit >= DATE_SUB(NOW(), INTERVAL 30 DAY)) as monthly_views
    ")->fetch(PDO::FETCH_ASSOC);
    
    // Utilisateurs récents
    $recent_users = $pdo->query("
        SELECT id, name, email, role, status, created_at, last_login
        FROM users 
        WHERE status != 'deleted'
        ORDER BY created_at DESC 
        LIMIT 10
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    // Projets récents
    $recent_projects = $pdo->query("
        SELECT p.*, u.name as user_name
        FROM flying_book_projects p
        LEFT JOIN users u ON p.user_id = u.id
        WHERE p.status != 'deleted'
        ORDER BY p.created_at DESC 
        LIMIT 10
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    // Activité système
    $system_activity = $pdo->query("
        SELECT 
            DATE(created_at) as date,
            COUNT(*) as activity_count
        FROM flying_book_activity_logs
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        GROUP BY DATE(created_at)
        ORDER BY date DESC
    ")->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    $error = "Erreur lors du chargement du dashboard admin: " . $e->getMessage();
    error_log("Admin dashboard error: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flying Book Creator - Admin Dashboard</title>
    <link rel="stylesheet" href="assets/css/united.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .admin-header {
            background: linear-gradient(135deg, #dc2626, #b91c1c);
            border-radius: var(--radius-xl);
            padding: var(--space-xl);
            color: var(--text-white);
            margin-bottom: var(--space-xl);
        }
        
        .admin-header h1 {
            margin-bottom: var(--space-sm);
            font-size: var(--font-size-3xl);
        }
        
        .admin-tabs {
            display: flex;
            background: var(--bg-primary);
            border-radius: var(--radius-lg);
            padding: var(--space-sm);
            margin-bottom: var(--space-xl);
            box-shadow: var(--shadow-sm);
            overflow-x: auto;
        }
        
        .admin-tab {
            flex: 1;
            min-width: 140px;
            padding: var(--space-md);
            background: none;
            border: none;
            border-radius: var(--radius-sm);
            cursor: pointer;
            transition: var(--transition-normal);
            font-weight: 600;
            color: var(--text-secondary);
            white-space: nowrap;
        }
        
        .admin-tab.active {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: var(--text-white);
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .admin-action-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: var(--space-lg);
            margin-bottom: var(--space-xl);
        }
        
        .admin-action-card {
            background: var(--bg-primary);
            border-radius: var(--radius-lg);
            padding: var(--space-xl);
            text-align: center;
            box-shadow: var(--shadow-sm);
            transition: var(--transition-normal);
            cursor: pointer;
            border: 2px solid transparent;
        }
        
        .admin-action-card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-md);
        }
        
        .admin-action-card.danger {
            border-color: rgba(239, 68, 68, 0.3);
            background: rgba(239, 68, 68, 0.05);
        }
        
        .admin-action-card.warning {
            border-color: rgba(245, 158, 11, 0.3);
            background: rgba(245, 158, 11, 0.05);
        }
        
        .admin-action-card.success {
            border-color: rgba(16, 185, 129, 0.3);
            background: rgba(16, 185, 129, 0.05);
        }
        
        .admin-action-card i {
            font-size: var(--font-size-4xl);
            margin-bottom: var(--space-md);
        }
        
        .admin-action-card.danger i {
            color: var(--error-color);
        }
        
        .admin-action-card.warning i {
            color: var(--warning-color);
        }
        
        .admin-action-card.success i {
            color: var(--success-color);
        }
        
        .admin-action-card h3 {
            margin-bottom: var(--space-sm);
            color: var(--text-primary);
        }
        
        .users-table {
            background: var(--bg-primary);
            border-radius: var(--radius-lg);
            overflow: hidden;
            box-shadow: var(--shadow-sm);
        }
        
        .users-table table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .users-table th,
        .users-table td {
            padding: var(--space-md);
            text-align: left;
            border-bottom: 1px solid var(--border-color);
        }
        
        .users-table th {
            background: var(--bg-secondary);
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .users-table tr:hover {
            background: var(--bg-secondary);
        }
        
        .status-badge {
            padding: var(--space-xs) var(--space-sm);
            border-radius: var(--radius-full);
            font-size: var(--font-size-xs);
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .status-badge.active {
            background: rgba(16, 185, 129, 0.2);
            color: var(--success-color);
        }
        
        .status-badge.inactive {
            background: rgba(107, 114, 128, 0.2);
            color: var(--text-secondary);
        }
        
        .status-badge.admin {
            background: rgba(239, 68, 68, 0.2);
            color: var(--error-color);
        }
        
        .user-actions {
            display: flex;
            gap: var(--space-xs);
        }
        
        .user-action-btn {
            padding: var(--space-xs);
            border: none;
            border-radius: var(--radius-xs);
            cursor: pointer;
            transition: var(--transition-fast);
            color: var(--text-white);
            font-size: var(--font-size-xs);
        }
        
        .user-action-btn.edit {
            background: var(--info-color);
        }
        
        .user-action-btn.delete {
            background: var(--error-color);
        }
        
        .user-action-btn.admin {
            background: var(--warning-color);
        }
        
        .system-monitor {
            background: var(--bg-primary);
            border-radius: var(--radius-lg);
            padding: var(--space-xl);
            box-shadow: var(--shadow-sm);
            margin-bottom: var(--space-xl);
        }
        
        .monitor-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: var(--space-lg);
        }
        
        .monitor-item {
            text-align: center;
            padding: var(--space-md);
            background: var(--bg-secondary);
            border-radius: var(--radius-sm);
        }
        
        .monitor-value {
            font-size: var(--font-size-2xl);
            font-weight: 700;
            color: var(--primary-color);
            margin-bottom: var(--space-xs);
        }
        
        .monitor-label {
            font-size: var(--font-size-sm);
            color: var(--text-secondary);
        }
        
        .logs-container {
            background: var(--bg-dark);
            border-radius: var(--radius-lg);
            padding: var(--space-lg);
            font-family: 'Monaco', 'Menlo', monospace;
            font-size: var(--font-size-sm);
            color: #00ff00;
            max-height: 400px;
            overflow-y: auto;
            margin-bottom: var(--space-xl);
        }
        
        .log-line {
            margin-bottom: var(--space-xs);
            word-break: break-all;
        }
        
        .log-line.error {
            color: #ff4444;
        }
        
        .log-line.warning {
            color: #ffaa00;
        }
        
        @media (max-width: 768px) {
            .admin-tabs {
                overflow-x: auto;
            }
            
            .admin-action-grid {
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            }
            
            .users-table {
                overflow-x: auto;
            }
        }
    </style>
</head>
<body>
    <!-- Header Navigation -->
    <header class="main-header">
        <div class="header-container">
            <div class="header-left">
                <a href="dashboard.php" class="logo">
                    <i class="fas fa-book-open"></i>
                    <span>Flying Book v2.0</span>
                </a>
                <nav class="main-nav">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-th-large"></i> Dashboard
                    </a>
                    <a href="admin.php" class="nav-link active">
                        <i class="fas fa-cog"></i> Admin
                    </a>
                    <a href="projects.php" class="nav-link">
                        <i class="fas fa-folder"></i> Projets
                    </a>
                    <a href="media.php" class="nav-link">
                        <i class="fas fa-images"></i> Médiathèque
                    </a>
                </nav>
            </div>
            
            <div class="header-right">
                <div class="user-menu">
                    <button class="user-button" onclick="toggleUserMenu()">
                        <div class="avatar">
                            <?= strtoupper(substr($user_name, 0, 1)) ?>
                        </div>
                        <span><?= htmlspecialchars($user_name) ?></span>
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown" id="userDropdown">
                        <a href="profile.php"><i class="fas fa-user"></i> Mon profil</a>
                        <a href="settings.php"><i class="fas fa-cog"></i> Paramètres</a>
                        <div class="dropdown-divider"></div>
                        <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Déconnexion</a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <main class="main-content">
        <div class="container">
            <!-- En-tête Admin -->
            <div class="admin-header">
                <h1><i class="fas fa-shield-alt"></i> Administration Flying Book</h1>
                <p>Panneau de contrôle administrateur - Accès restreint</p>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <!-- Statistiques système -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?= $global_stats['total_users'] ?></h3>
                        <p>Utilisateurs totaux</p>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-user-check"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?= $global_stats['active_users'] ?></h3>
                        <p>Utilisateurs actifs</p>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-book"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?= $global_stats['total_projects'] ?></h3>
                        <p>Projets créés</p>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-hdd"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?= formatFileSize($global_stats['total_storage']) ?></h3>
                        <p>Stockage utilisé</p>
                    </div>
                </div>
            </div>

            <!-- Onglets admin -->
            <div class="admin-tabs">
                <button class="admin-tab active" onclick="showTab('overview')">
                    <i class="fas fa-tachometer-alt"></i> Vue d'ensemble
                </button>
                <button class="admin-tab" onclick="showTab('users')">
                    <i class="fas fa-users"></i> Utilisateurs
                </button>
                <button class="admin-tab" onclick="showTab('system')">
                    <i class="fas fa-server"></i> Système
                </button>
                <button class="admin-tab" onclick="showTab('maintenance')">
                    <i class="fas fa-tools"></i> Maintenance
                </button>
                <button class="admin-tab" onclick="showTab('logs')">
                    <i class="fas fa-file-alt"></i> Logs
                </button>
            </div>

            <!-- Contenu des onglets -->
            
            <!-- Onglet Vue d'ensemble -->
            <div id="overview-tab" class="tab-content active">
                <div class="admin-action-grid">
                    <div class="admin-action-card success" onclick="refreshSystemStats()">
                        <i class="fas fa-sync-alt"></i>
                        <h3>Actualiser les stats</h3>
                        <p>Mettre à jour les données en temps réel</p>
                    </div>
                    
                    <div class="admin-action-card warning" onclick="showTab('maintenance')">
                        <i class="fas fa-broom"></i>
                        <h3>Maintenance</h3>
                        <p>Nettoyer et optimiser le système</p>
                    </div>
                    
                    <div class="admin-action-card" onclick="showTab('users')">
                        <i class="fas fa-user-cog"></i>
                        <h3>Gérer les utilisateurs</h3>
                        <p>Administration des comptes</p>
                    </div>
                    
                    <div class="admin-action-card danger" onclick="showTab('logs')">
                        <i class="fas fa-exclamation-triangle"></i>
                        <h3>Logs d'erreur</h3>
                        <p>Surveiller les problèmes système</p>
                    </div>
                </div>

                <!-- Projets récents -->
                <div class="recent-section">
                    <h2><i class="fas fa-clock"></i> Projets récents</h2>
                    <?php if (empty($recent_projects)): ?>
                        <div class="empty-state">
                            <i class="fas fa-folder-open"></i>
                            <h3>Aucun projet récent</h3>
                        </div>
                    <?php else: ?>
                        <?php foreach ($recent_projects as $project): ?>
                            <div class="project-mini-card">
                                <div class="project-mini-thumbnail">
                                    <i class="fas fa-presentation"></i>
                                </div>
                                <div class="project-mini-info">
                                    <h4><?= htmlspecialchars($project['title'] ?: $project['project_name']) ?></h4>
                                    <p>
                                        Par: <?= htmlspecialchars($project['user_name']) ?> • 
                                        <?= ucfirst($project['status']) ?> • 
                                        <?= timeAgo($project['created_at']) ?>
                                    </p>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Onglet Utilisateurs -->
            <div id="users-tab" class="tab-content">
                <div class="users-table">
                    <table>
                        <thead>
                            <tr>
                                <th>Nom</th>
                                <th>Email</th>
                                <th>Rôle</th>
                                <th>Statut</th>
                                <th>Inscription</th>
                                <th>Dernière connexion</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_users as $user_item): ?>
                                <tr>
                                    <td><?= htmlspecialchars($user_item['name']) ?></td>
                                    <td><?= htmlspecialchars($user_item['email']) ?></td>
                                    <td>
                                        <span class="status-badge <?= $user_item['role'] ?>">
                                            <?= ucfirst($user_item['role']) ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="status-badge <?= $user_item['status'] ?>">
                                            <?= ucfirst($user_item['status']) ?>
                                        </span>
                                    </td>
                                    <td><?= date('d/m/Y', strtotime($user_item['created_at'])) ?></td>
                                    <td><?= $user_item['last_login'] ? timeAgo($user_item['last_login']) : 'Jamais' ?></td>
                                    <td>
                                        <div class="user-actions">
                                            <?php if ($user_item['id'] != $user_id): ?>
                                                <?php if ($user_item['status'] === 'active'): ?>
                                                    <button class="user-action-btn delete" 
                                                            onclick="manageUser(<?= $user_item['id'] ?>, 'deactivate')"
                                                            title="Désactiver">
                                                        <i class="fas fa-ban"></i>
                                                    </button>
                                                <?php else: ?>
                                                    <button class="user-action-btn edit" 
                                                            onclick="manageUser(<?= $user_item['id'] ?>, 'activate')"
                                                            title="Activer">
                                                        <i class="fas fa-check"></i>
                                                    </button>
                                                <?php endif; ?>
                                                
                                                <?php if ($user_item['role'] !== 'admin'): ?>
                                                    <button class="user-action-btn admin" 
                                                            onclick="manageUser(<?= $user_item['id'] ?>, 'make_admin')"
                                                            title="Promouvoir admin">
                                                        <i class="fas fa-crown"></i>
                                                    </button>
                                                <?php else: ?>
                                                    <button class="user-action-btn edit" 
                                                            onclick="manageUser(<?= $user_item['id'] ?>, 'remove_admin')"
                                                            title="Retirer admin">
                                                        <i class="fas fa-user"></i>
                                                    </button>
                                                <?php endif; ?>
                                                
                                                <button class="user-action-btn delete" 
                                                        onclick="manageUser(<?= $user_item['id'] ?>, 'delete')"
                                                        title="Supprimer">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            <?php else: ?>
                                                <span style="color: var(--text-muted); font-size: var(--font-size-xs);">Vous</span>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Onglet Système -->
            <div id="system-tab" class="tab-content">
                <div class="system-monitor">
                    <h3><i class="fas fa-server"></i> Monitoring système</h3>
                    <div class="monitor-grid" id="systemMonitor">
                        <div class="monitor-item">
                            <div class="monitor-value" id="phpVersion">-</div>
                            <div class="monitor-label">Version PHP</div>
                        </div>
                        <div class="monitor-item">
                            <div class="monitor-value" id="memoryUsage">-</div>
                            <div class="monitor-label">Mémoire utilisée</div>
                        </div>
                        <div class="monitor-item">
                            <div class="monitor-value" id="diskUsage">-</div>
                            <div class="monitor-label">Espace disque</div>
                        </div>
                        <div class="monitor-item">
                            <div class="monitor-value" id="serverLoad">-</div>
                            <div class="monitor-label">Charge serveur</div>
                        </div>
                    </div>
                    
                    <div style="text-align: center; margin-top: var(--space-xl);">
                        <button class="btn btn-primary" onclick="refreshSystemStats()">
                            <i class="fas fa-sync-alt"></i>
                            Actualiser
                        </button>
                    </div>
                </div>
            </div>

            <!-- Onglet Maintenance -->
            <div id="maintenance-tab" class="tab-content">
                <div class="admin-action-grid">
                    <div class="admin-action-card warning" onclick="cleanupSystem('draft_projects')">
                        <i class="fas fa-folder-minus"></i>
                        <h3>Nettoyer les brouillons</h3>
                        <p>Supprimer les projets non modifiés depuis 30 jours</p>
                    </div>
                    
                    <div class="admin-action-card warning" onclick="cleanupSystem('orphan_media')">
                        <i class="fas fa-image"></i>
                        <h3>Médias orphelins</h3>
                        <p>Supprimer les fichiers sans propriétaire</p>
                    </div>
                    
                    <div class="admin-action-card warning" onclick="cleanupSystem('old_analytics')">
                        <i class="fas fa-chart-line"></i>
                        <h3>Analytics anciennes</h3>
                        <p>Nettoyer les données > 1 an</p>
                    </div>
                    
                    <div class="admin-action-card danger" onclick="cleanupSystem('temp_files')">
                        <i class="fas fa-trash"></i>
                        <h3>Fichiers temporaires</h3>
                        <p>Vider le cache et les fichiers temp</p>
                    </div>
                </div>
            </div>

            <!-- Onglet Logs -->
            <div id="logs-tab" class="tab-content">
                <div style="margin-bottom: var(--space-lg);">
                    <button class="btn btn-primary" onclick="loadErrorLogs()">
                        <i class="fas fa-sync-alt"></i>
                        Recharger les logs
                    </button>
                </div>
                
                <div class="logs-container" id="logsContainer">
                    <div class="log-line">Flying Book Admin - Logs système</div>
                    <div class="log-line">Cliquez sur "Recharger les logs" pour voir les erreurs récentes</div>
                </div>
            </div>
        </div>
    </main>

    <!-- Scripts -->
    <script>
        // ========== GESTION DES ONGLETS ==========
        function showTab(tabName) {
            // Masquer tous les contenus
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Désactiver tous les onglets
            document.querySelectorAll('.admin-tab').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Afficher le contenu sélectionné
            document.getElementById(tabName + '-tab').classList.add('active');
            
            // Activer l'onglet sélectionné
            event.target.classList.add('active');
        }
        
        // ========== STATISTIQUES SYSTÈME ==========
        async function refreshSystemStats() {
            showNotification('Actualisation des statistiques...', 'info');
            
            try {
                const response = await fetch('admin.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'ajax_action=get_system_stats'
                });
                
                const result = await response.json();
                
                if (result.success) {
                    updateSystemMonitor(result.stats);
                    showNotification('Statistiques mises à jour', 'success');
                } else {
                    showNotification('Erreur lors de la mise à jour', 'error');
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
            }
        }
        
        function updateSystemMonitor(stats) {
            document.getElementById('phpVersion').textContent = stats.php_version;
            document.getElementById('memoryUsage').textContent = formatFileSize(stats.memory_usage);
            document.getElementById('diskUsage').textContent = 
                Math.round((stats.disk_used / stats.disk_total) * 100) + '%';
            document.getElementById('serverLoad').textContent = 
                typeof stats.server_load === 'number' ? stats.server_load.toFixed(2) : stats.server_load;
        }
        
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 B';
            const k = 1024;
            const sizes = ['B', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
        
        // ========== GESTION UTILISATEURS ==========
        async function manageUser(userId, action) {
            const actionMessages = {
                'activate': 'Activer cet utilisateur ?',
                'deactivate': 'Désactiver cet utilisateur ?',
                'delete': 'ATTENTION: Supprimer définitivement cet utilisateur ?',
                'make_admin': 'Promouvoir cet utilisateur administrateur ?',
                'remove_admin': 'Retirer les droits administrateur ?'
            };
            
            if (!confirm(actionMessages[action])) {
                return;
            }
            
            if (action === 'delete' && !confirm('DERNIÈRE CONFIRMATION\n\nCette action est IRRÉVERSIBLE !')) {
                return;
            }
            
            try {
                const response = await fetch('admin.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `ajax_action=manage_user&user_id=${userId}&user_action=${action}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors de l\'action', 'error');
            }
        }
        
        // ========== NETTOYAGE SYSTÈME ==========
        async function cleanupSystem(cleanupType) {
            const cleanupMessages = {
                'draft_projects': 'Supprimer les brouillons anciens ?',
                'orphan_media': 'Supprimer les médias orphelins ?',
                'old_analytics': 'Supprimer les anciennes analytics ?',
                'temp_files': 'Vider les fichiers temporaires ?'
            };
            
            if (!confirm(cleanupMessages[cleanupType])) {
                return;
            }
            
            try {
                showNotification('Nettoyage en cours...', 'warning');
                
                const response = await fetch('admin.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `ajax_action=cleanup_system&cleanup_type=${cleanupType}`
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(result.message, 'success');
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Erreur lors du nettoyage', 'error');
            }
        }
        
        // ========== LOGS ==========
        async function loadErrorLogs() {
            showNotification('Chargement des logs...', 'info');
            
            try {
                const response = await fetch('admin.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: 'ajax_action=get_error_logs'
                });
                
                const result = await response.json();
                
                if (result.success) {
                    const logsContainer = document.getElementById('logsContainer');
                    
                    if (result.logs.length === 0) {
                        logsContainer.innerHTML = '<div class="log-line">Aucun log d\'erreur trouvé</div>';
                    } else {
                        logsContainer.innerHTML = result.logs.map(log => {
                            const className = log.toLowerCase().includes('error') ? 'error' : 
                                            log.toLowerCase().includes('warning') ? 'warning' : '';
                            return `<div class="log-line ${className}">${log}</div>`;
                        }).join('');
                    }
                    
                    showNotification('Logs chargés', 'success');
                } else {
                    showNotification('Erreur lors du chargement', 'error');
                }
            } catch (error) {
                showNotification('Erreur de connexion', 'error');
            }
        }
        
        // ========== UTILITAIRES ==========
        function toggleUserMenu() {
            const dropdown = document.getElementById('userDropdown');
            const isOpen = dropdown.style.display === 'block';
            dropdown.style.display = isOpen ? 'none' : 'block';
        }
        
        // Fermer les menus en cliquant ailleurs
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.user-menu')) {
                document.getElementById('userDropdown').style.display = 'none';
            }
        });
        
        // ========== SYSTÈME DE NOTIFICATIONS ==========
        function showNotification(message, type = 'info') {
            // Supprimer les notifications existantes
            document.querySelectorAll('.notification').forEach(n => n.remove());
            
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            
            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                warning: 'fa-exclamation-triangle',
                info: 'fa-info-circle'
            };
            
            notification.innerHTML = `
                <div style="display: flex; align-items: center; gap: 0.75rem;">
                    <i class="fas ${icons[type] || icons.info}" style="font-size: 1.2rem;"></i>
                    <div>
                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${message}</div>
                        <div style="font-size: 0.8rem; opacity: 0.8;">${new Date().toLocaleTimeString()}</div>
                    </div>
                </div>
            `;
            
            document.body.appendChild(notification);
            
            // Animation d'apparition
            setTimeout(() => notification.classList.add('show'), 100);
            
            // Auto-suppression
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => notification.remove(), 300);
            }, type === 'error' ? 8000 : 5000);
        }
        
        // ========== INITIALISATION ==========
        document.addEventListener('DOMContentLoaded', function() {
            // Charger les stats au démarrage
            refreshSystemStats();
            
            showNotification('Dashboard Admin chargé ! 🛡️', 'success');
        });
    </script>
</body>
</html>