<?php
/**
 * Flying Book Creator - Export ZIP System
 * /document/creator/export_flyingbook.php
 */

require_once __DIR__ . '/../../config/auth_check.php';
checkFlyingBookAuth();

$user = getFlyingBookUser();
$pdo = DatabaseConfig::getConnection();

$project_id = $_GET['project_id'] ?? null;

if (!$project_id) {
    die('ID de projet manquant');
}

// Vérifier que le projet appartient à l'utilisateur
$stmt = $pdo->prepare("SELECT * FROM flying_book_projects WHERE id = ? AND user_id = ?");
$stmt->execute([$project_id, $user['id']]);
$project = $stmt->fetch();

if (!$project) {
    die('Projet non trouvé');
}

// Récupérer les slides
$stmt = $pdo->prepare("
    SELECT * FROM flying_book_slides 
    WHERE project_id = ? 
    ORDER BY slide_number ASC
");
$stmt->execute([$project_id]);
$slides = $stmt->fetchAll();

// Récupérer les médias
$stmt = $pdo->prepare("
    SELECT * FROM flying_book_media 
    WHERE project_id = ? AND upload_status = 'ready'
    ORDER BY created_at ASC
");
$stmt->execute([$project_id]);
$medias = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Export - <?= htmlspecialchars($project['project_name']) ?></title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 50%, #f093fb 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .export-container {
            max-width: 800px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 40px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.1);
        }
        
        .export-header {
            text-align: center;
            margin-bottom: 40px;
        }
        
        .export-header h1 {
            color: #1a4b73;
            margin-bottom: 10px;
        }
        
        .project-info {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 30px;
        }
        
        .export-options {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .export-option {
            background: white;
            border-radius: 15px;
            padding: 25px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
            border: 2px solid transparent;
        }
        
        .export-option:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.15);
        }
        
        .export-option.selected {
            border-color: #667eea;
        }
        
        .progress-container {
            display: none;
            margin: 30px 0;
        }
        
        .progress-bar {
            background: #e9ecef;
            border-radius: 10px;
            height: 20px;
            overflow: hidden;
            margin-bottom: 15px;
        }
        
        .progress-fill {
            background: linear-gradient(45deg, #667eea, #764ba2);
            height: 100%;
            width: 0%;
            transition: width 0.3s ease;
        }
        
        .btn {
            padding: 15px 30px;
            border: none;
            border-radius: 12px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            margin: 5px;
        }
        
        .btn-primary {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .logs {
            background: #1a1a1a;
            color: #00ff00;
            border-radius: 10px;
            padding: 20px;
            font-family: 'Courier New', monospace;
            font-size: 12px;
            height: 200px;
            overflow-y: auto;
            margin-top: 20px;
            display: none;
        }
        
        .download-section {
            display: none;
            text-align: center;
            padding: 30px;
            background: rgba(40, 167, 69, 0.1);
            border-radius: 15px;
            margin-top: 30px;
        }
    </style>
</head>
<body>
    <div class="export-container">
        <div class="export-header">
            <h1>📦 Export Flying Book</h1>
            <p>Générez votre présentation pour le web</p>
        </div>
        
        <div class="project-info">
            <h3><?= htmlspecialchars($project['project_name']) ?></h3>
            <p><strong>Titre:</strong> <?= htmlspecialchars($project['title']) ?></p>
            <p><strong>Slides:</strong> <?= count($slides) ?></p>
            <p><strong>Médias:</strong> <?= count($medias) ?></p>
            <p><strong>Style:</strong> <?= htmlspecialchars($project['style_theme']) ?></p>
        </div>
        
        <div class="export-options">
            <div class="export-option" data-type="complete">
                <div style="font-size: 3rem; margin-bottom: 15px;">📁</div>
                <h4>Export Complet</h4>
                <p>HTML, CSS, JS et tous les médias</p>
                <small>Prêt pour déploiement</small>
            </div>
            
            <div class="export-option" data-type="lightweight">
                <div style="font-size: 3rem; margin-bottom: 15px;">⚡</div>
                <h4>Version Légère</h4>
                <p>Médias optimisés et compressés</p>
                <small>Plus rapide à télécharger</small>
            </div>
            
            <div class="export-option" data-type="presentation">
                <div style="font-size: 3rem; margin-bottom: 15px;">🎪</div>
                <h4>Mode Présentation</h4>
                <p>Interface simplifiée pour présenter</p>
                <small>Navigation tactile</small>
            </div>
        </div>
        
        <div style="text-align: center; margin: 30px 0;">
            <button class="btn btn-primary" onclick="startExport()">
                🚀 Générer l'export
            </button>
            <button class="btn btn-secondary" onclick="window.close()">
                Annuler
            </button>
        </div>
        
        <div class="progress-container" id="progressContainer">
            <h4>Progression de l'export</h4>
            <div class="progress-bar">
                <div class="progress-fill" id="progressFill"></div>
            </div>
            <div id="progressText">Initialisation...</div>
        </div>
        
        <div class="logs" id="logs"></div>
        
        <div class="download-section" id="downloadSection">
            <div style="font-size: 4rem; margin-bottom: 20px;">✅</div>
            <h3>Export terminé !</h3>
            <p>Votre Flying Book est prêt à être téléchargé</p>
            <button class="btn btn-primary" id="downloadBtn">
                📥 Télécharger ZIP
            </button>
        </div>
    </div>
    
    <script>
        let selectedExportType = 'complete';
        let exportData = <?= json_encode([
            'project' => $project,
            'slides' => $slides,
            'medias' => $medias,
            'user' => $user
        ]) ?>;
        
        // Sélection du type d'export
        document.querySelectorAll('.export-option').forEach(option => {
            option.addEventListener('click', function() {
                document.querySelectorAll('.export-option').forEach(opt => 
                    opt.classList.remove('selected')
                );
                this.classList.add('selected');
                selectedExportType = this.dataset.type;
            });
        });
        
        // Sélectionner le premier par défaut
        document.querySelector('.export-option').classList.add('selected');
        
        function startExport() {
            document.getElementById('progressContainer').style.display = 'block';
            document.getElementById('logs').style.display = 'block';
            
            log('🚀 Démarrage de l\'export ' + selectedExportType);
            log('📊 Projet: ' + exportData.project.project_name);
            log('📄 ' + exportData.slides.length + ' slides à traiter');
            log('🖼️ ' + exportData.medias.length + ' médias à inclure');
            
            // Simulation du processus d'export
            simulateExport();
        }
        
        function simulateExport() {
            const steps = [
                { percent: 10, text: '📁 Création de la structure de fichiers...', delay: 500 },
                { percent: 25, text: '🎨 Génération du HTML...', delay: 800 },
                { percent: 40, text: '🎪 Application du style ' + exportData.project.style_theme + '...', delay: 600 },
                { percent: 55, text: '⚡ Intégration des animations PowerPoint...', delay: 1000 },
                { percent: 70, text: '🖼️ Traitement des médias...', delay: 1200 },
                { percent: 85, text: '📦 Compression des fichiers...', delay: 800 },
                { percent: 95, text: '🔗 Génération des liens...', delay: 400 },
                { percent: 100, text: '✅ Export terminé !', delay: 300 }
            ];
            
            let currentStep = 0;
            
            function nextStep() {
                if (currentStep < steps.length) {
                    const step = steps[currentStep];
                    updateProgress(step.percent, step.text);
                    log(step.text);
                    
                    currentStep++;
                    setTimeout(nextStep, step.delay);
                } else {
                    completeExport();
                }
            }
            
            nextStep();
        }
        
        function updateProgress(percent, text) {
            document.getElementById('progressFill').style.width = percent + '%';
            document.getElementById('progressText').textContent = text;
        }
        
        function log(message) {
            const logs = document.getElementById('logs');
            const timestamp = new Date().toLocaleTimeString();
            logs.innerHTML += `[${timestamp}] ${message}\n`;
            logs.scrollTop = logs.scrollHeight;
        }
        
        function completeExport() {
            log('🎉 Export terminé avec succès !');
            log('📊 Statistiques:');
            log('   - Taille estimée: ' + calculateSize() + 'MB');
            log('   - Fichiers générés: ' + (exportData.slides.length + exportData.medias.length + 5));
            log('   - Animations: ' + countAnimations() + ' effets PowerPoint');
            
            document.getElementById('downloadSection').style.display = 'block';
            
            // Préparer le téléchargement
            prepareDownload();
        }
        
        function calculateSize() {
            // Estimation basique
            let size = 2; // Fichiers de base (HTML, CSS, JS)
            size += exportData.medias.reduce((acc, media) => acc + (media.file_size / 1024 / 1024), 0);
            return Math.round(size * 100) / 100;
        }
        
        function countAnimations() {
            return exportData.slides.reduce((acc, slide) => {
                if (slide.powerpoint_animations) {
                    try {
                        const animations = JSON.parse(slide.powerpoint_animations);
                        return acc + (Array.isArray(animations) ? animations.length : 0);
                    } catch (e) {
                        return acc;
                    }
                }
                return acc;
            }, 0);
        }
        
        function prepareDownload() {
            // Générer le contenu ZIP (simulation)
            const exportContent = generateExportContent();
            
            document.getElementById('downloadBtn').onclick = function() {
                downloadZip(exportContent);
            };
        }
        
        function generateExportContent() {
            // Ici on génèrerait le vrai contenu HTML/CSS/JS
            return {
                'index.html': generateHTML(),
                'style.css': generateCSS(),
                'script.js': generateJavaScript(),
                'config.json': JSON.stringify({
                    project: exportData.project,
                    slides: exportData.slides,
                    export_type: selectedExportType,
                    generated_at: new Date().toISOString()
                }, null, 2)
            };
        }
        
        function generateHTML() {
            return `<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${exportData.project.title}</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <div id="flyingbook-container">
        <div id="flyingbook-header">
            <h1>${exportData.project.title}</h1>
            ${exportData.project.subtitle ? `<h2>${exportData.project.subtitle}</h2>` : ''}
        </div>
        
        <div id="flyingbook-slides">
            ${exportData.slides.map((slide, index) => `
                <div class="slide" data-slide="${index + 1}">
                    <div class="slide-content">
                        ${slide.slide_title ? `<h3>${slide.slide_title}</h3>` : ''}
                        ${slide.text_overlay ? `<div class="text-overlay">${slide.text_overlay}</div>` : ''}
                        ${slide.media_file ? `<div class="media-container">
                            ${slide.media_type === 'image' ? 
                                `<img src="media/${slide.media_file}" alt="Slide ${index + 1}">` :
                                slide.media_type === 'video' ? 
                                `<video controls><source src="media/${slide.media_file}"></video>` :
                                `<div class="media-placeholder">Média: ${slide.media_file}</div>`
                            }
                        </div>` : ''}
                    </div>
                </div>
            `).join('')}
        </div>
        
        <div id="flyingbook-navigation">
            <button id="prev-btn">←</button>
            <div id="slide-counter">
                <span id="current-slide">1</span> / <span id="total-slides">${exportData.slides.length}</span>
            </div>
            <button id="next-btn">→</button>
        </div>
    </div>
    
    <script src="script.js"></script>
</body>
</html>`;
        }
        
        function generateCSS() {
            const theme = exportData.project.style_theme;
            
            return `/* Flying Book Creator - Generated CSS */
body {
    font-family: 'Inter', sans-serif;
    margin: 0;
    padding: 0;
    background: ${theme === 'moderne' ? 'linear-gradient(135deg, #667eea, #764ba2)' : '#f8f9fa'};
    min-height: 100vh;
}

#flyingbook-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
}

#flyingbook-header {
    text-align: center;
    margin-bottom: 40px;
    color: white;
}

#flyingbook-header h1 {
    font-size: 2.5rem;
    margin-bottom: 10px;
}

#flyingbook-slides {
    position: relative;
    background: white;
    border-radius: 20px;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    min-height: 500px;
}

.slide {
    display: none;
    padding: 40px;
    animation: fadeIn 0.5s ease;
}

.slide.active {
    display: block;
}

.slide-content h3 {
    color: #1a4b73;
    margin-bottom: 20px;
    font-size: 1.8rem;
}

.text-overlay {
    font-size: 1.1rem;
    line-height: 1.6;
    color: #333;
    margin-bottom: 20px;
    white-space: pre-line;
}

.media-container {
    text-align: center;
    margin: 20px 0;
}

.media-container img,
.media-container video {
    max-width: 100%;
    max-height: 400px;
    border-radius: 10px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
}

#flyingbook-navigation {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 20px;
    margin-top: 30px;
}

#flyingbook-navigation button {
    padding: 15px 20px;
    border: none;
    border-radius: 10px;
    background: rgba(255, 255, 255, 0.9);
    cursor: pointer;
    font-size: 1.2rem;
    transition: all 0.3s ease;
}

#flyingbook-navigation button:hover {
    background: white;
    transform: translateY(-2px);
}

#slide-counter {
    background: rgba(255, 255, 255, 0.9);
    padding: 10px 20px;
    border-radius: 20px;
    font-weight: 600;
}

/* Animations PowerPoint */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}

@keyframes slideInLeft {
    from { opacity: 0; transform: translateX(-50px); }
    to { opacity: 1; transform: translateX(0); }
}

@keyframes bounceIn {
    0% { opacity: 0; transform: scale(0.3); }
    50% { opacity: 1; transform: scale(1.05); }
    70% { transform: scale(0.9); }
    100% { opacity: 1; transform: scale(1); }
}

/* Responsive */
@media (max-width: 768px) {
    #flyingbook-container {
        padding: 10px;
    }
    
    .slide {
        padding: 20px;
    }
    
    #flyingbook-header h1 {
        font-size: 1.8rem;
    }
}`;
        }
        
        function generateJavaScript() {
            return `// Flying Book Creator - Generated JavaScript
let currentSlide = 1;
const totalSlides = ${exportData.slides.length};

document.addEventListener('DOMContentLoaded', function() {
    initializeFlyingBook();
});

function initializeFlyingBook() {
    showSlide(1);
    
    // Navigation
    document.getElementById('prev-btn').addEventListener('click', prevSlide);
    document.getElementById('next-btn').addEventListener('click', nextSlide);
    
    // Clavier
    document.addEventListener('keydown', function(e) {
        if (e.key === 'ArrowLeft') prevSlide();
        if (e.key === 'ArrowRight') nextSlide();
        if (e.key === 'Escape') toggleFullscreen();
    });
    
    // Touch/swipe (mobile)
    let startX = 0;
    document.addEventListener('touchstart', function(e) {
        startX = e.touches[0].clientX;
    });
    
    document.addEventListener('touchend', function(e) {
        const endX = e.changedTouches[0].clientX;
        const diff = startX - endX;
        
        if (Math.abs(diff) > 50) {
            if (diff > 0) nextSlide();
            else prevSlide();
        }
    });
    
    console.log('🎪 Flying Book initialized with ' + totalSlides + ' slides');
}

function showSlide(slideNumber) {
    // Masquer toutes les slides
    document.querySelectorAll('.slide').forEach(slide => {
        slide.classList.remove('active');
    });
    
    // Afficher la slide demandée
    const targetSlide = document.querySelector('.slide[data-slide="' + slideNumber + '"]');
    if (targetSlide) {
        targetSlide.classList.add('active');
        currentSlide = slideNumber;
        
        // Mettre à jour le compteur
        document.getElementById('current-slide').textContent = currentSlide;
        
        // Appliquer les animations PowerPoint si disponibles
        applySlideAnimations(targetSlide);
        
        // Analytics (si activé)
        trackSlideView(slideNumber);
    }
}

function nextSlide() {
    if (currentSlide < totalSlides) {
        showSlide(currentSlide + 1);
    }
}

function prevSlide() {
    if (currentSlide > 1) {
        showSlide(currentSlide - 1);
    }
}

function applySlideAnimations(slideElement) {
    // Appliquer les animations PowerPoint converties
    const animatedElements = slideElement.querySelectorAll('[data-animation]');
    
    animatedElements.forEach((element, index) => {
        const animation = element.dataset.animation;
        const delay = index * 200; // Délai entre animations
        
        setTimeout(() => {
            element.style.animation = animation + ' 0.5s ease forwards';
        }, delay);
    });
}

function trackSlideView(slideNumber) {
    // Analytics basique
    if (typeof gtag !== 'undefined') {
        gtag('event', 'slide_view', {
            'slide_number': slideNumber,
            'project_name': '${exportData.project.project_name}'
        });
    }
    
    console.log('Slide ' + slideNumber + ' viewed');
}

function toggleFullscreen() {
    if (!document.fullscreenElement) {
        document.documentElement.requestFullscreen();
    } else {
        document.exitFullscreen();
    }
}

// Utilitaires
function getSlideData() {
    return ${JSON.stringify(exportData.slides)};
}

function getProjectInfo() {
    return ${JSON.stringify(exportData.project)};
}`;
        }
        
        function downloadZip(content) {
            // Simulation du téléchargement
            const filename = exportData.project.project_slug + '_' + selectedExportType + '.zip';
            
            // Dans un vrai cas, on utiliserait JSZip pour créer le ZIP
            log('📥 Téléchargement de ' + filename);
            
            // Créer un blob avec du contenu factice pour la démo
            const blob = new Blob([JSON.stringify(content, null, 2)], { type: 'application/json' });
            const url = URL.createObjectURL(blob);
            
            const a = document.createElement('a');
            a.href = url;
            a.download = filename;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
            
            log('✅ Téléchargement lancé !');
        }
        
        console.log('🎪 Export Flying Book ready!', exportData);
    </script>
</body>
</html>