<?php
/**
 * ================================================================
 * 👥 GESTION UTILISATEURS SMG STUDIO v2.0 - PANNEAU ADMIN
 * ================================================================
 * 
 * 🎯 CRUD complet utilisateurs + tokens + rôles
 * ✅ Navigation bidirectionnelle avec dashboard
 * 🔐 Sécurité admin + logs des actions
 * 💰 Attribution tokens directe
 * 
 * 💡 BY PAPY 77 - INTERFACE ADMIN COMPLÈTE !
 * ================================================================
 */

session_start();
require_once 'config/db.php';
require_once 'helpers/common_functions.php';

// Vérification session + droits admin
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?expired=1');
    exit;
}

// Récupération utilisateur courant
try {
    $stmt = $pdo->prepare("SELECT id, name, email, role FROM users WHERE id = ? AND status = 'active'");
    $stmt->execute([$_SESSION['user_id']]);
    $current_user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$current_user || $current_user['role'] !== 'admin') {
        header('Location: dashboard.php?error=insufficient_privileges');
        exit;
    }
} catch (Exception $e) {
    error_log("Erreur vérification admin: " . $e->getMessage());
    header('Location: dashboard.php?error=1');
    exit;
}

// ================================================================
// 🔧 TRAITEMENT DES ACTIONS ADMIN
// ================================================================

$action = $_GET['action'] ?? 'list';
$message = '';
$message_type = '';
$selected_user = null;

// 👤 CRÉATION UTILISATEUR
if ($_POST['action'] ?? '' === 'create_user') {
    try {
        $name = trim($_POST['name']);
        $email = trim($_POST['email']);
        $password = $_POST['password'];
        $role = $_POST['role'];
        
        if (empty($name) || empty($email) || empty($password)) {
            throw new Exception("Tous les champs sont requis");
        }
        
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new Exception("Email invalide");
        }
        
        // Vérifier email unique
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);
        if ($stmt->fetch()) {
            throw new Exception("Email déjà utilisé");
        }
        
        // Créer utilisateur
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("
            INSERT INTO users (name, email, password, role, status, email_verified, created_at)
            VALUES (?, ?, ?, ?, 'active', 1, NOW())
        ");
        
        $stmt->execute([$name, $email, $hashed_password, $role]);
        $new_user_id = $pdo->lastInsertId();
        
        // Attribution package tokens si demandé
        if (!empty($_POST['initial_tokens']) && $_POST['initial_tokens'] > 0) {
            $tokens = intval($_POST['initial_tokens']);
            $stmt = $pdo->prepare("
                INSERT INTO user_token_packages 
                (user_id, package_name, tokens_purchased, price_paid, payment_reference, status, purchase_date)
                VALUES (?, 'admin_attribution', ?, 0.00, ?, 'active', NOW())
            ");
            $reference = 'ADMIN_GIFT_' . date('Ymd_His');
            $stmt->execute([$new_user_id, $tokens, $reference]);
        }
        
        // Log de l'action
        logAdminAction($current_user['id'], 'create_user', 'users', $new_user_id, [
            'name' => $name,
            'email' => $email,
            'role' => $role,
            'initial_tokens' => $_POST['initial_tokens'] ?? 0
        ]);
        
        $message = "✅ Utilisateur créé avec succès : $name";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur création : " . $e->getMessage();
        $message_type = 'error';
    }
}

// ✏️ MODIFICATION UTILISATEUR
if ($_POST['action'] ?? '' === 'update_user') {
    try {
        $user_id = intval($_POST['user_id']);
        $name = trim($_POST['name']);
        $email = trim($_POST['email']);
        $role = $_POST['role'];
        $status = $_POST['status'];
        
        if (empty($name) || empty($email)) {
            throw new Exception("Nom et email requis");
        }
        
        // Ne pas se modifier soi-même en non-admin
        if ($user_id == $current_user['id'] && $role !== 'admin') {
            throw new Exception("Vous ne pouvez pas retirer vos droits admin");
        }
        
        $stmt = $pdo->prepare("
            UPDATE users 
            SET name = ?, email = ?, role = ?, status = ?, updated_at = NOW()
            WHERE id = ?
        ");
        
        $stmt->execute([$name, $email, $role, $status, $user_id]);
        
        // Modification mot de passe si fourni
        if (!empty($_POST['new_password'])) {
            $hashed_password = password_hash($_POST['new_password'], PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
            $stmt->execute([$hashed_password, $user_id]);
        }
        
        logAdminAction($current_user['id'], 'update_user', 'users', $user_id, [
            'name' => $name,
            'email' => $email,
            'role' => $role,
            'status' => $status
        ]);
        
        $message = "✅ Utilisateur modifié avec succès";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur modification : " . $e->getMessage();
        $message_type = 'error';
    }
}

// 💰 ATTRIBUTION TOKENS
if ($_POST['action'] ?? '' === 'add_tokens') {
    try {
        $user_id = intval($_POST['user_id']);
        $tokens_amount = intval($_POST['tokens_amount']);
        $package_name = $_POST['package_name'] ?? 'admin_attribution';
        
        if ($tokens_amount <= 0) {
            throw new Exception("Montant tokens invalide");
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO user_token_packages 
            (user_id, package_name, tokens_purchased, price_paid, payment_reference, status, purchase_date)
            VALUES (?, ?, ?, 0.00, ?, 'active', NOW())
        ");
        
        $reference = 'ADMIN_ADD_' . date('Ymd_His') . '_' . $user_id;
        $stmt->execute([$user_id, $package_name, $tokens_amount, $reference]);
        
        logAdminAction($current_user['id'], 'add_tokens', 'user_token_packages', $pdo->lastInsertId(), [
            'user_id' => $user_id,
            'tokens_amount' => $tokens_amount,
            'package_name' => $package_name
        ]);
        
        $message = "✅ $tokens_amount tokens ajoutés avec succès";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur ajout tokens : " . $e->getMessage();
        $message_type = 'error';
    }
}

// 🗑️ SUPPRESSION UTILISATEUR (soft delete)
if ($_GET['action'] ?? '' === 'delete_user') {
    try {
        $user_id = intval($_GET['user_id']);
        
        if ($user_id == $current_user['id']) {
            throw new Exception("Vous ne pouvez pas vous supprimer vous-même");
        }
        
        $stmt = $pdo->prepare("
            UPDATE users 
            SET status = 'deleted', deleted_at = NOW()
            WHERE id = ? AND role != 'admin'
        ");
        
        $affected = $stmt->execute([$user_id]);
        
        if ($stmt->rowCount() === 0) {
            throw new Exception("Impossible de supprimer cet utilisateur");
        }
        
        logAdminAction($current_user['id'], 'delete_user', 'users', $user_id, []);
        
        $message = "✅ Utilisateur supprimé avec succès";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur suppression : " . $e->getMessage();
        $message_type = 'error';
    }
}

// ================================================================
// 📊 CHARGEMENT DONNÉES
// ================================================================

// Liste des utilisateurs avec stats
$users_list = getUsersWithStats();

// Stats globales
$global_stats = getGlobalUserStats();

// Utilisateur sélectionné pour édition
if ($action === 'edit' && isset($_GET['user_id'])) {
    $selected_user = getUserDetails(intval($_GET['user_id']));
}

// ================================================================
// 🔧 FONCTIONS MÉTIER
// ================================================================

function getUsersWithStats() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT 
                u.*,
                COALESCE(tb.current_balance, 0) as token_balance,
                COALESCE(tb.total_purchased, 0) as total_tokens_purchased,
                COALESCE(tb.total_used, 0) as total_tokens_used,
                COALESCE(ai_stats.total_requests, 0) as ai_requests,
                COALESCE(ai_stats.total_tokens_used, 0) as ai_tokens_consumed,
                COUNT(DISTINCT p.id) as posts_count,
                MAX(lc.created_at) as last_login_date
            FROM users u
            LEFT JOIN user_token_balance tb ON u.id = tb.user_id
            LEFT JOIN (
                SELECT 
                    aul.user_id,
                    COUNT(*) as total_requests,
                    SUM(aul.tokens_total) as total_tokens_used
                FROM ai_usage_logs aul
                GROUP BY aul.user_id
            ) ai_stats ON u.id = ai_stats.user_id
            LEFT JOIN posts p ON u.id = p.user_id
            LEFT JOIN log_connexions lc ON u.id = lc.user_id
            WHERE u.status != 'deleted'
            GROUP BY u.id
            ORDER BY u.created_at DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getUsersWithStats: " . $e->getMessage());
        return [];
    }
}

function getGlobalUserStats() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as total_users,
                COUNT(CASE WHEN status = 'active' THEN 1 END) as active_users,
                COUNT(CASE WHEN role = 'admin' THEN 1 END) as admin_users,
                COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 END) as new_users_30d,
                COALESCE(SUM(tb.current_balance), 0) as total_tokens_balance,
                COALESCE(SUM(tb.total_purchased), 0) as total_tokens_distributed
            FROM users u
            LEFT JOIN user_token_balance tb ON u.id = tb.user_id
            WHERE u.status != 'deleted'
        ");
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getGlobalUserStats: " . $e->getMessage());
        return [
            'total_users' => 0,
            'active_users' => 0,
            'admin_users' => 0,
            'new_users_30d' => 0,
            'total_tokens_balance' => 0,
            'total_tokens_distributed' => 0
        ];
    }
}

function getUserDetails($user_id) {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT u.*, 
                   COUNT(DISTINCT utp.id) as packages_count,
                   COALESCE(SUM(utp.tokens_purchased), 0) as total_tokens_purchased,
                   COALESCE(SUM(utp.tokens_used), 0) as total_tokens_used,
                   COUNT(DISTINCT uap.id) as ai_profiles_count
            FROM users u
            LEFT JOIN user_token_packages utp ON u.id = utp.user_id AND utp.status = 'active'
            LEFT JOIN user_ai_profiles uap ON u.id = uap.user_id AND uap.is_active = 1
            WHERE u.id = ?
            GROUP BY u.id
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getUserDetails: " . $e->getMessage());
        return null;
    }
}

function logAdminAction($admin_id, $action, $entity_type, $entity_id, $details) {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs 
            (user_id, action, entity_type, entity_id, description, details, ip_address, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $description = "Action admin: $action sur $entity_type";
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        
        $stmt->execute([
            $admin_id, 
            $action, 
            $entity_type, 
            $entity_id, 
            $description, 
            json_encode($details), 
            $ip
        ]);
    } catch (Exception $e) {
        error_log("Erreur logAdminAction: " . $e->getMessage());
    }
}

function formatDate($date) {
    return $date ? date('d/m/Y H:i', strtotime($date)) : 'Jamais';
}

function getUserStatusBadge($status) {
    $badges = [
        'active' => '<span class="badge badge-success">🟢 Actif</span>',
        'inactive' => '<span class="badge badge-warning">🟡 Inactif</span>',
        'pending' => '<span class="badge badge-info">🔵 En attente</span>',
        'deleted' => '<span class="badge badge-danger">🔴 Supprimé</span>'
    ];
    return $badges[$status] ?? $status;
}

function getRoleBadge($role) {
    $badges = [
        'admin' => '<span class="badge badge-danger">👑 Admin</span>',
        'manager' => '<span class="badge badge-primary">👔 Manager</span>',
        'user' => '<span class="badge badge-secondary">👤 Utilisateur</span>'
    ];
    return $badges[$role] ?? $role;
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>👥 Gestion Utilisateurs - SMG Studio v2.0</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }

        .main-container {
            max-width: 1600px;
            margin: 0 auto;
            padding: 20px;
        }

        .main-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 25px 35px;
            margin-bottom: 30px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header-logo {
            width: 60px;
            height: 60px;
            background: linear-gradient(45deg, #e74c3c, #c0392b);
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .nav-ia-principale {
            display: flex;
            gap: 8px;
            background: rgba(255, 107, 107, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(255, 107, 107, 0.2);
        }

        .nav-ia-principale a {
            padding: 8px 16px;
            background: rgba(255, 107, 107, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .nav-ia-principale a:hover {
            background: #ff6b6b;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        .admin-nav {
            display: flex;
            gap: 8px;
            background: rgba(231, 76, 60, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(231, 76, 60, 0.2);
        }

        .admin-nav a {
            padding: 8px 16px;
            background: rgba(231, 76, 60, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .admin-nav a:hover, .admin-nav a.active {
            background: #e74c3c;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(231, 76, 60, 0.4);
        }

        .content-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }

        .card-title {
            font-size: 20px;
            font-weight: 600;
            color: #333;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .stat-icon {
            font-size: 32px;
            margin-bottom: 10px;
            display: block;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 700;
            color: #e74c3c;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .btn {
            padding: 8px 16px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            transition: all 0.3s ease;
            font-size: 13px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            color: white;
        }

        .btn-success {
            background: linear-gradient(135deg, #27ae60, #229954);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, #f39c12, #e67e22);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            color: white;
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
        }

        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border: 1px solid transparent;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-success {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }

        .alert-error {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }

        .users-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }

        .users-table th,
        .users-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #dee2e6;
        }

        .users-table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #495057;
        }

        .users-table tr:hover {
            background: #f8f9fa;
        }

        .badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
        }

        .badge-success {
            background: #d4edda;
            color: #155724;
        }

        .badge-warning {
            background: #fff3cd;
            color: #856404;
        }

        .badge-danger {
            background: #f8d7da;
            color: #721c24;
        }

        .badge-info {
            background: #d1ecf1;
            color: #0c5460;
        }

        .badge-primary {
            background: #d1ecf1;
            color: #0c5460;
        }

        .badge-secondary {
            background: #e2e3e5;
            color: #383d41;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
        }

        .form-control {
            width: 100%;
            padding: 10px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s ease;
        }

        .form-control:focus {
            outline: none;
            border-color: #e74c3c;
            box-shadow: 0 0 0 3px rgba(231, 76, 60, 0.1);
        }

        .form-select {
            width: 100%;
            padding: 10px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            background: white;
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.8);
            z-index: 10000;
            align-items: center;
            justify-content: center;
        }

        .modal-content {
            background: white;
            border-radius: 15px;
            padding: 30px;
            max-width: 500px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }

        .modal-title {
            font-size: 18px;
            font-weight: 600;
            color: #333;
        }

        .close-btn {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #999;
        }

        .action-buttons {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        @media (max-width: 768px) {
            .main-container {
                padding: 15px;
            }
            
            .main-header {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
            
            .nav-ia-principale, .admin-nav {
                flex-wrap: wrap;
                justify-content: center;
            }
            
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .users-table {
                font-size: 12px;
            }
            
            .users-table th,
            .users-table td {
                padding: 8px 4px;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header avec Navigation -->
        <header class="main-header">
            <div class="header-left">
                <div class="header-logo">👥</div>
                <div>
                    <h1>Gestion Utilisateurs</h1>
                    <p>Administration SMG Studio v2.0</p>
                </div>
            </div>
            
            <div style="display: flex; align-items: center; gap: 15px; flex-wrap: wrap;">
                <!-- Navigation IA -->
                <div class="nav-ia-principale">
                    <a href="ai_content_generator.php">🎨 IA Générateur</a>
                    <a href="ai_dashboard.php">📊 IA Dashboard</a>
                    <a href="ai_profiles.php">⚙️ IA Profils</a>
                </div>
                
                <!-- Navigation Admin -->
                <div class="admin-nav">
                    <a href="users_management.php" class="active">👥 Utilisateurs</a>
                    <a href="tokens_management.php">💰 Tokens</a>
                    <a href="system_admin.php">⚙️ Système</a>
                </div>
                
                <!-- Retour Dashboard -->
                <a href="dashboard.php" class="btn btn-primary">
                    📊 Dashboard Principal
                </a>
            </div>
        </header>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type ?>">
                <span><?= $message ?></span>
            </div>
        <?php endif; ?>

        <!-- Stats globales -->
        <div class="stats-grid">
            <div class="stat-card">
                <span class="stat-icon">👥</span>
                <div class="stat-value"><?= $global_stats['total_users'] ?></div>
                <div class="stat-label">Total Utilisateurs</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">🟢</span>
                <div class="stat-value"><?= $global_stats['active_users'] ?></div>
                <div class="stat-label">Utilisateurs Actifs</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">👑</span>
                <div class="stat-value"><?= $global_stats['admin_users'] ?></div>
                <div class="stat-label">Administrateurs</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">🆕</span>
                <div class="stat-value"><?= $global_stats['new_users_30d'] ?></div>
                <div class="stat-label">Nouveaux (30j)</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">💰</span>
                <div class="stat-value"><?= number_format($global_stats['total_tokens_balance']) ?></div>
                <div class="stat-label">Tokens en circulation</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">📦</span>
                <div class="stat-value"><?= number_format($global_stats['total_tokens_distributed']) ?></div>
                <div class="stat-label">Tokens distribués</div>
            </div>
        </div>

        <!-- Gestion des utilisateurs -->
        <div class="content-card">
            <div class="card-header">
                <h2 class="card-title">
                    👥 Liste des Utilisateurs
                </h2>
                <div class="action-buttons">
                    <button onclick="openCreateUserModal()" class="btn btn-success">
                        ➕ Nouveau Utilisateur
                    </button>
                    <button onclick="refreshUsers()" class="btn btn-primary">
                        🔄 Actualiser
                    </button>
                </div>
            </div>
            
            <table class="users-table">
                <thead>
                    <tr>
                        <th>👤 Utilisateur</th>
                        <th>📧 Email</th>
                        <th>🎭 Rôle</th>
                        <th>📊 Statut</th>
                        <th>💰 Tokens</th>
                        <th>🤖 IA</th>
                        <th>📝 Posts</th>
                        <th>🕒 Dernière connexion</th>
                        <th>⚡ Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users_list as $user): ?>
                        <tr>
                            <td>
                                <strong><?= htmlspecialchars($user['name']) ?></strong>
                                <br><small>ID: <?= $user['id'] ?></small>
                            </td>
                            <td><?= htmlspecialchars($user['email']) ?></td>
                            <td><?= getRoleBadge($user['role']) ?></td>
                            <td><?= getUserStatusBadge($user['status']) ?></td>
                            <td>
                                <strong><?= number_format($user['token_balance']) ?></strong>
                                <br><small><?= number_format($user['total_tokens_used']) ?> utilisés</small>
                            </td>
                            <td>
                                <strong><?= number_format($user['ai_requests']) ?></strong> req.
                                <br><small><?= number_format($user['ai_tokens_consumed']) ?> tokens</small>
                            </td>
                            <td><?= number_format($user['posts_count']) ?></td>
                            <td><?= formatDate($user['last_login_date']) ?></td>
                            <td>
                                <div class="action-buttons">
                                    <button onclick="editUser(<?= $user['id'] ?>)" class="btn btn-warning btn-sm">
                                        ✏️ Éditer
                                    </button>
                                    <button onclick="addTokensToUser(<?= $user['id'] ?>, '<?= htmlspecialchars($user['name']) ?>')" class="btn btn-success btn-sm">
                                        💰 Tokens
                                    </button>
                                    <?php if ($user['id'] != $current_user['id']): ?>
                                        <button onclick="deleteUser(<?= $user['id'] ?>, '<?= htmlspecialchars($user['name']) ?>')" class="btn btn-danger btn-sm">
                                            🗑️ Suppr.
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Modal Création Utilisateur -->
    <div id="createUserModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">➕ Créer un Utilisateur</h3>
                <button type="button" class="close-btn" onclick="closeModal('createUserModal')">&times;</button>
            </div>
            
            <form method="POST">
                <input type="hidden" name="action" value="create_user">
                
                <div class="form-group">
                    <label class="form-label">👤 Nom complet</label>
                    <input type="text" name="name" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">📧 Email</label>
                    <input type="email" name="email" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">🔒 Mot de passe</label>
                    <input type="password" name="password" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">🎭 Rôle</label>
                    <select name="role" class="form-select" required>
                        <option value="user">👤 Utilisateur</option>
                        <option value="manager">👔 Manager</option>
                        <option value="admin">👑 Administrateur</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">💰 Tokens initiaux (optionnel)</label>
                    <input type="number" name="initial_tokens" class="form-control" min="0" placeholder="0">
                    <small>Laissez vide pour aucun token initial</small>
                </div>
                
                <div class="action-buttons">
                    <button type="submit" class="btn btn-success">
                        ✅ Créer l'utilisateur
                    </button>
                    <button type="button" onclick="closeModal('createUserModal')" class="btn btn-secondary">
                        ❌ Annuler
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal Édition Utilisateur -->
    <div id="editUserModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">✏️ Éditer l'Utilisateur</h3>
                <button type="button" class="close-btn" onclick="closeModal('editUserModal')">&times;</button>
            </div>
            
            <form method="POST" id="editUserForm">
                <input type="hidden" name="action" value="update_user">
                <input type="hidden" name="user_id" id="edit_user_id">
                
                <div class="form-group">
                    <label class="form-label">👤 Nom complet</label>
                    <input type="text" name="name" id="edit_name" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">📧 Email</label>
                    <input type="email" name="email" id="edit_email" class="form-control" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">🔒 Nouveau mot de passe (optionnel)</label>
                    <input type="password" name="new_password" class="form-control" placeholder="Laissez vide pour ne pas modifier">
                </div>
                
                <div class="form-group">
                    <label class="form-label">🎭 Rôle</label>
                    <select name="role" id="edit_role" class="form-select" required>
                        <option value="user">👤 Utilisateur</option>
                        <option value="manager">👔 Manager</option>
                        <option value="admin">👑 Administrateur</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">📊 Statut</label>
                    <select name="status" id="edit_status" class="form-select" required>
                        <option value="active">🟢 Actif</option>
                        <option value="inactive">🟡 Inactif</option>
                        <option value="pending">🔵 En attente</option>
                    </select>
                </div>
                
                <div class="action-buttons">
                    <button type="submit" class="btn btn-warning">
                        ✅ Sauvegarder
                    </button>
                    <button type="button" onclick="closeModal('editUserModal')" class="btn btn-secondary">
                        ❌ Annuler
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal Ajout Tokens -->
    <div id="addTokensModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">💰 Ajouter des Tokens</h3>
                <button type="button" class="close-btn" onclick="closeModal('addTokensModal')">&times;</button>
            </div>
            
            <form method="POST">
                <input type="hidden" name="action" value="add_tokens">
                <input type="hidden" name="user_id" id="tokens_user_id">
                
                <div class="form-group">
                    <label class="form-label">👤 Utilisateur</label>
                    <input type="text" id="tokens_user_name" class="form-control" readonly>
                </div>
                
                <div class="form-group">
                    <label class="form-label">💰 Nombre de tokens</label>
                    <input type="number" name="tokens_amount" class="form-control" min="1" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">📦 Type de package</label>
                    <select name="package_name" class="form-select">
                        <option value="admin_attribution">🎁 Attribution Admin</option>
                        <option value="bonus">🎉 Bonus</option>
                        <option value="refund">🔙 Remboursement</option>
                        <option value="promotion">🏷️ Promotion</option>
                    </select>
                </div>
                
                <div class="action-buttons">
                    <button type="submit" class="btn btn-success">
                        ✅ Ajouter les tokens
                    </button>
                    <button type="button" onclick="closeModal('addTokensModal')" class="btn btn-secondary">
                        ❌ Annuler
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Gestion des modals
        function openCreateUserModal() {
            document.getElementById('createUserModal').style.display = 'flex';
        }

        function closeModal(modalId) {
            document.getElementById(modalId).style.display = 'none';
        }

        // Édition utilisateur
        function editUser(userId) {
            // Récupérer les données utilisateur via AJAX ou directement
            window.location.href = `users_management.php?action=edit&user_id=${userId}`;
        }

        // Ajout tokens
        function addTokensToUser(userId, userName) {
            document.getElementById('tokens_user_id').value = userId;
            document.getElementById('tokens_user_name').value = userName;
            document.getElementById('addTokensModal').style.display = 'flex';
        }

        // Suppression utilisateur
        function deleteUser(userId, userName) {
            if (confirm(`⚠️ Êtes-vous sûr de vouloir supprimer l'utilisateur "${userName}" ?\n\nCette action est irréversible !`)) {
                window.location.href = `users_management.php?action=delete_user&user_id=${userId}`;
            }
        }

        // Actualiser la page
        function refreshUsers() {
            window.location.reload();
        }

        // Fermeture modal en cliquant à l'extérieur
        window.onclick = function(event) {
            const modals = document.querySelectorAll('.modal');
            modals.forEach(modal => {
                if (event.target === modal) {
                    modal.style.display = 'none';
                }
            });
        }

        console.log('👥 SMG Studio v2.0 - Gestion Utilisateurs Ready!');
        console.log('✅ Interface admin complète par Papy 77');
        console.log('🔧 CRUD utilisateurs + tokens + navigation IA');
    </script>
</body>
</html>