<?php
/**
 * ================================================================
 * 💰 GESTION TOKENS SMG STUDIO v2.0 - PANNEAU ADMIN
 * ================================================================
 * 
 * 🎯 Gestion complète packages + quotas + débits
 * ✅ Attribution manuelle + historique transactions
 * 📊 Analytics temps réel consommation IA
 * 🔄 Système débit automatique intégré
 * 
 * 💡 BY PAPY 77 - HUB CENTRAL TOKENS !
 * ================================================================
 */

session_start();
require_once 'config/db.php';
require_once 'helpers/common_functions.php';

// Vérification session + droits admin
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?expired=1');
    exit;
}

// Récupération utilisateur courant
try {
    $stmt = $pdo->prepare("SELECT id, name, email, role FROM users WHERE id = ? AND status = 'active'");
    $stmt->execute([$_SESSION['user_id']]);
    $current_user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$current_user || $current_user['role'] !== 'admin') {
        header('Location: dashboard.php?error=insufficient_privileges');
        exit;
    }
} catch (Exception $e) {
    error_log("Erreur vérification admin: " . $e->getMessage());
    header('Location: dashboard.php?error=1');
    exit;
}

// ================================================================
// 🔧 TRAITEMENT DES ACTIONS TOKENS
// ================================================================

$action = $_GET['action'] ?? 'overview';
$message = '';
$message_type = '';

// 💰 ATTRIBUTION MANUELLE TOKENS
if ($_POST['action'] ?? '' === 'manual_attribution') {
    try {
        $user_id = intval($_POST['user_id']);
        $tokens_amount = intval($_POST['tokens_amount']);
        $package_name = $_POST['package_name'] ?? 'admin_manual';
        $notes = trim($_POST['notes'] ?? '');
        
        if ($tokens_amount <= 0) {
            throw new Exception("Montant tokens invalide");
        }
        
        // Vérifier que l'utilisateur existe
        $stmt = $pdo->prepare("SELECT name FROM users WHERE id = ? AND status = 'active'");
        $stmt->execute([$user_id]);
        $target_user = $stmt->fetch();
        
        if (!$target_user) {
            throw new Exception("Utilisateur introuvable");
        }
        
        // Créer le package tokens
        $stmt = $pdo->prepare("
            INSERT INTO user_token_packages 
            (user_id, package_name, tokens_purchased, price_paid, payment_reference, status, purchase_date, expires_at)
            VALUES (?, ?, ?, 0.00, ?, 'active', NOW(), DATE_ADD(NOW(), INTERVAL 1 YEAR))
        ");
        
        $reference = 'MANUAL_' . date('Ymd_His') . '_' . $current_user['id'];
        $stmt->execute([$user_id, $package_name, $tokens_amount, $reference]);
        
        // Log de l'action
        logTokenAction($current_user['id'], 'manual_attribution', [
            'user_id' => $user_id,
            'user_name' => $target_user['name'],
            'tokens_amount' => $tokens_amount,
            'package_name' => $package_name,
            'notes' => $notes
        ]);
        
        $message = "✅ $tokens_amount tokens attribués à " . $target_user['name'];
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur attribution : " . $e->getMessage();
        $message_type = 'error';
    }
}

// 📦 CRÉATION PACKAGE CATALOGUE
if ($_POST['action'] ?? '' === 'create_package_catalog') {
    try {
        $package_key = trim($_POST['package_key']);
        $package_name = trim($_POST['package_name']);
        $tokens_included = intval($_POST['tokens_included']);
        $price_eur = floatval($_POST['price_eur']);
        $description = trim($_POST['description']);
        $is_popular = isset($_POST['is_popular']) ? 1 : 0;
        
        if (empty($package_key) || empty($package_name) || $tokens_included <= 0) {
            throw new Exception("Données package invalides");
        }
        
        $stmt = $pdo->prepare("
            INSERT INTO token_packages_catalog 
            (package_key, package_name, tokens_included, price_eur, description, is_popular, is_active, sort_order)
            VALUES (?, ?, ?, ?, ?, ?, 1, 0)
        ");
        
        $stmt->execute([$package_key, $package_name, $tokens_included, $price_eur, $description, $is_popular]);
        
        logTokenAction($current_user['id'], 'create_package_catalog', [
            'package_key' => $package_key,
            'package_name' => $package_name,
            'tokens_included' => $tokens_included,
            'price_eur' => $price_eur
        ]);
        
        $message = "✅ Package catalogue créé : $package_name";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur création package : " . $e->getMessage();
        $message_type = 'error';
    }
}

// 🚫 RÉVOCATION TOKENS
if ($_POST['action'] ?? '' === 'revoke_tokens') {
    try {
        $package_id = intval($_POST['package_id']);
        $reason = trim($_POST['reason']);
        
        // Récupérer info package
        $stmt = $pdo->prepare("
            SELECT utp.*, u.name as user_name
            FROM user_token_packages utp
            JOIN users u ON utp.user_id = u.id
            WHERE utp.id = ?
        ");
        $stmt->execute([$package_id]);
        $package = $stmt->fetch();
        
        if (!$package) {
            throw new Exception("Package introuvable");
        }
        
        // Marquer comme expiré
        $stmt = $pdo->prepare("
            UPDATE user_token_packages 
            SET status = 'expired', expires_at = NOW(), updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$package_id]);
        
        logTokenAction($current_user['id'], 'revoke_tokens', [
            'package_id' => $package_id,
            'user_name' => $package['user_name'],
            'tokens_revoked' => $package['tokens_remaining'],
            'reason' => $reason
        ]);
        
        $message = "✅ Tokens révoqués pour " . $package['user_name'];
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur révocation : " . $e->getMessage();
        $message_type = 'error';
    }
}

// 📊 MISE À JOUR QUOTA IA
if ($_POST['action'] ?? '' === 'update_ai_quota') {
    try {
        $user_id = intval($_POST['user_id']);
        $quota_period = $_POST['quota_period'];
        $tokens_limit = intval($_POST['tokens_limit']);
        $cost_limit = floatval($_POST['cost_limit']);
        
        if ($tokens_limit <= 0) {
            throw new Exception("Limite tokens invalide");
        }
        
        // Calculer dates période
        $start_date = date('Y-m-d');
        switch ($quota_period) {
            case 'daily':
                $end_date = date('Y-m-d', strtotime('+1 day'));
                break;
            case 'weekly':
                $end_date = date('Y-m-d', strtotime('+1 week'));
                break;
            case 'monthly':
                $end_date = date('Y-m-d', strtotime('+1 month'));
                break;
            case 'yearly':
                $end_date = date('Y-m-d', strtotime('+1 year'));
                break;
            default:
                throw new Exception("Période invalide");
        }
        
        // Créer ou mettre à jour quota
        $stmt = $pdo->prepare("
            INSERT INTO user_ai_quotas 
            (user_id, quota_period, quota_start_date, quota_end_date, tokens_limit, cost_limit, is_active)
            VALUES (?, ?, ?, ?, ?, ?, 1)
            ON DUPLICATE KEY UPDATE
            tokens_limit = VALUES(tokens_limit),
            cost_limit = VALUES(cost_limit),
            quota_end_date = VALUES(quota_end_date),
            is_active = 1,
            updated_at = NOW()
        ");
        
        $stmt->execute([$user_id, $quota_period, $start_date, $end_date, $tokens_limit, $cost_limit]);
        
        $message = "✅ Quota IA mis à jour";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur quota : " . $e->getMessage();
        $message_type = 'error';
    }
}

// ================================================================
// 📊 CHARGEMENT DONNÉES TOKENS
// ================================================================

// Vue d'ensemble tokens
$tokens_overview = getTokensOverview();

// Utilisateurs avec soldes tokens
$users_tokens = getUsersTokensBalance();

// Packages catalogue
$packages_catalog = getPackagesCatalog();

// Historique transactions récentes
$recent_transactions = getRecentTokenTransactions();

// Analytics consommation IA
$ai_consumption = getAIConsumptionStats();

// Quotas IA actifs
$active_quotas = getActiveAIQuotas();

// ================================================================
// 🔧 FONCTIONS MÉTIER TOKENS
// ================================================================

function getTokensOverview() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(DISTINCT utp.user_id) as users_with_tokens,
                COUNT(utp.id) as total_packages,
                SUM(utp.tokens_purchased) as total_tokens_distributed,
                SUM(utp.tokens_used) as total_tokens_consumed,
                SUM(utp.tokens_remaining) as total_tokens_remaining,
                SUM(utp.price_paid) as total_revenue,
                AVG(utp.tokens_purchased) as avg_package_size,
                COUNT(CASE WHEN utp.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 END) as packages_last_7d
            FROM user_token_packages utp
            WHERE utp.status = 'active'
        ");
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getTokensOverview: " . $e->getMessage());
        return [
            'users_with_tokens' => 0,
            'total_packages' => 0,
            'total_tokens_distributed' => 0,
            'total_tokens_consumed' => 0,
            'total_tokens_remaining' => 0,
            'total_revenue' => 0,
            'avg_package_size' => 0,
            'packages_last_7d' => 0
        ];
    }
}

function getUsersTokensBalance() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT 
                u.id,
                u.name,
                u.email,
                u.role,
                COALESCE(SUM(utp.tokens_purchased), 0) as total_purchased,
                COALESCE(SUM(utp.tokens_used), 0) as total_used,
                COALESCE(SUM(utp.tokens_remaining), 0) as current_balance,
                COALESCE(SUM(utp.price_paid), 0) as total_spent,
                COUNT(utp.id) as packages_count,
                MAX(utp.created_at) as last_purchase_date,
                COALESCE(ai_usage.total_tokens_consumed, 0) as ai_tokens_used,
                COALESCE(ai_usage.total_cost, 0) as ai_total_cost
            FROM users u
            LEFT JOIN user_token_packages utp ON u.id = utp.user_id AND utp.status = 'active'
            LEFT JOIN (
                SELECT 
                    user_id,
                    SUM(tokens_total) as total_tokens_consumed,
                    SUM(cost_estimate) as total_cost
                FROM ai_usage_logs
                GROUP BY user_id
            ) ai_usage ON u.id = ai_usage.user_id
            WHERE u.status = 'active'
            GROUP BY u.id
            HAVING total_purchased > 0 OR ai_tokens_used > 0
            ORDER BY current_balance DESC, total_purchased DESC
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getUsersTokensBalance: " . $e->getMessage());
        return [];
    }
}

function getPackagesCatalog() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT tpc.*,
                   COUNT(utp.id) as sales_count,
                   SUM(utp.tokens_purchased) as tokens_sold,
                   SUM(utp.price_paid) as revenue_generated
            FROM token_packages_catalog tpc
            LEFT JOIN user_token_packages utp ON tpc.package_key = utp.package_name
            WHERE tpc.is_active = 1
            GROUP BY tpc.id
            ORDER BY tpc.sort_order ASC, tpc.price_eur ASC
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getPackagesCatalog: " . $e->getMessage());
        return [];
    }
}

function getRecentTokenTransactions() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT 
                utp.*,
                u.name as user_name,
                u.email as user_email,
                admin.name as admin_name
            FROM user_token_packages utp
            JOIN users u ON utp.user_id = u.id
            LEFT JOIN activity_logs al ON al.entity_type = 'user_token_packages' 
                AND al.entity_id = utp.id 
                AND al.action IN ('manual_attribution', 'add_tokens')
            LEFT JOIN users admin ON al.user_id = admin.id
            WHERE utp.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            ORDER BY utp.created_at DESC
            LIMIT 50
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getRecentTokenTransactions: " . $e->getMessage());
        return [];
    }
}

function getAIConsumptionStats() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT 
                DATE(aul.created_at) as consumption_date,
                COUNT(*) as requests_count,
                SUM(aul.tokens_total) as tokens_consumed,
                SUM(aul.cost_estimate) as total_cost,
                AVG(aul.processing_time_ms) as avg_processing_time,
                COUNT(DISTINCT aul.user_id) as active_users,
                COUNT(CASE WHEN aul.success = 0 THEN 1 END) as failed_requests
            FROM ai_usage_logs aul
            WHERE aul.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY DATE(aul.created_at)
            ORDER BY consumption_date DESC
            LIMIT 30
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getAIConsumptionStats: " . $e->getMessage());
        return [];
    }
}

function getActiveAIQuotas() {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT 
                uaq.*,
                u.name as user_name,
                u.email as user_email,
                ROUND((uaq.tokens_used / uaq.tokens_limit) * 100, 1) as usage_percentage,
                CASE 
                    WHEN uaq.tokens_used >= uaq.tokens_limit THEN 'EXCEEDED'
                    WHEN (uaq.tokens_used / uaq.tokens_limit) >= 0.8 THEN 'WARNING'
                    ELSE 'OK'
                END as status,
                DATEDIFF(uaq.quota_end_date, CURDATE()) as days_remaining
            FROM user_ai_quotas uaq
            JOIN users u ON uaq.user_id = u.id
            WHERE uaq.is_active = 1 
            AND CURDATE() BETWEEN uaq.quota_start_date AND uaq.quota_end_date
            ORDER BY usage_percentage DESC, uaq.quota_end_date ASC
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getActiveAIQuotas: " . $e->getMessage());
        return [];
    }
}

function logTokenAction($admin_id, $action, $details) {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs 
            (user_id, action, entity_type, description, details, ip_address, created_at)
            VALUES (?, ?, 'tokens_management', ?, ?, ?, NOW())
        ");
        
        $description = "Action tokens admin: $action";
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        
        $stmt->execute([
            $admin_id, 
            $action, 
            $description, 
            json_encode($details), 
            $ip
        ]);
    } catch (Exception $e) {
        error_log("Erreur logTokenAction: " . $e->getMessage());
    }
}

function formatDate($date) {
    return $date ? date('d/m/Y H:i', strtotime($date)) : 'N/A';
}

function getStatusBadge($status) {
    $badges = [
        'OK' => '<span class="badge badge-success">🟢 OK</span>',
        'WARNING' => '<span class="badge badge-warning">🟡 Attention</span>',
        'EXCEEDED' => '<span class="badge badge-danger">🔴 Dépassé</span>',
        'active' => '<span class="badge badge-success">✅ Actif</span>',
        'expired' => '<span class="badge badge-danger">❌ Expiré</span>',
        'pending' => '<span class="badge badge-info">⏳ En attente</span>'
    ];
    return $badges[$status] ?? $status;
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>💰 Gestion Tokens - SMG Studio v2.0</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }

        .main-container {
            max-width: 1600px;
            margin: 0 auto;
            padding: 20px;
        }

        .main-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 25px 35px;
            margin-bottom: 30px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header-logo {
            width: 60px;
            height: 60px;
            background: linear-gradient(45deg, #f39c12, #e67e22);
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .nav-ia-principale {
            display: flex;
            gap: 8px;
            background: rgba(255, 107, 107, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(255, 107, 107, 0.2);
        }

        .nav-ia-principale a {
            padding: 8px 16px;
            background: rgba(255, 107, 107, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .nav-ia-principale a:hover {
            background: #ff6b6b;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        .admin-nav {
            display: flex;
            gap: 8px;
            background: rgba(243, 156, 18, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(243, 156, 18, 0.2);
        }

        .admin-nav a {
            padding: 8px 16px;
            background: rgba(243, 156, 18, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .admin-nav a:hover, .admin-nav a.active {
            background: #f39c12;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(243, 156, 18, 0.4);
        }

        .content-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }

        .card-title {
            font-size: 20px;
            font-weight: 600;
            color: #333;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .stat-icon {
            font-size: 32px;
            margin-bottom: 10px;
            display: block;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 700;
            color: #f39c12;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .stat-details {
            margin-top: 10px;
            font-size: 11px;
            color: #999;
        }

        .btn {
            padding: 8px 16px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            transition: all 0.3s ease;
            font-size: 13px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #f39c12, #e67e22);
            color: white;
        }

        .btn-success {
            background: linear-gradient(135deg, #27ae60, #229954);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, #f39c12, #e67e22);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            color: white;
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
        }

        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border: 1px solid transparent;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-success {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }

        .alert-error {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }

        .tokens-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }

        .tokens-table th,
        .tokens-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #dee2e6;
        }

        .tokens-table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #495057;
        }

        .tokens-table tr:hover {
            background: #f8f9fa;
        }

        .badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
        }

        .badge-success {
            background: #d4edda;
            color: #155724;
        }

        .badge-warning {
            background: #fff3cd;
            color: #856404;
        }

        .badge-danger {
            background: #f8d7da;
            color: #721c24;
        }

        .badge-info {
            background: #d1ecf1;
            color: #0c5460;
        }

        .progress-bar {
            width: 100%;
            height: 8px;
            background: #e9ecef;
            border-radius: 4px;
            overflow: hidden;
            margin: 5px 0;
        }

        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #27ae60, #f39c12, #e74c3c);
            transition: width 0.3s ease;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
        }

        .form-control {
            width: 100%;
            padding: 10px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s ease;
        }

        .form-control:focus {
            outline: none;
            border-color: #f39c12;
            box-shadow: 0 0 0 3px rgba(243, 156, 18, 0.1);
        }

        .form-select {
            width: 100%;
            padding: 10px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            background: white;
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.8);
            z-index: 10000;
            align-items: center;
            justify-content: center;
        }

        .modal-content {
            background: white;
            border-radius: 15px;
            padding: 30px;
            max-width: 600px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }

        .modal-title {
            font-size: 18px;
            font-weight: 600;
            color: #333;
        }

        .close-btn {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #999;
        }

        .action-buttons {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        .tabs {
            display: flex;
            gap: 5px;
            margin-bottom: 20px;
            border-bottom: 2px solid #f8f9fa;
        }

        .tab {
            padding: 10px 20px;
            background: none;
            border: none;
            border-bottom: 3px solid transparent;
            cursor: pointer;
            font-weight: 600;
            color: #666;
            transition: all 0.3s ease;
        }

        .tab.active {
            color: #f39c12;
            border-bottom-color: #f39c12;
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }

        @media (max-width: 768px) {
            .main-container {
                padding: 15px;
            }
            
            .main-header {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
            
            .nav-ia-principale, .admin-nav {
                flex-wrap: wrap;
                justify-content: center;
            }
            
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .tokens-table {
                font-size: 12px;
            }
            
            .tokens-table th,
            .tokens-table td {
                padding: 8px 4px;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header avec Navigation -->
        <header class="main-header">
            <div class="header-left">
                <div class="header-logo">💰</div>
                <div>
                    <h1>Gestion Tokens</h1>
                    <p>Hub Central SMG Studio v2.0</p>
                </div>
            </div>
            
            <div style="display: flex; align-items: center; gap: 15px; flex-wrap: wrap;">
                <!-- Navigation IA -->
                <div class="nav-ia-principale">
                    <a href="ai_content_generator.php">🎨 IA Générateur</a>
                    <a href="ai_dashboard.php">📊 IA Dashboard</a>
                    <a href="ai_profiles.php">⚙️ IA Profils</a>
                </div>
                
                <!-- Navigation Admin -->
                <div class="admin-nav">
                    <a href="users_management.php">👥 Utilisateurs</a>
                    <a href="tokens_management.php" class="active">💰 Tokens</a>
                    <a href="system_admin.php">⚙️ Système</a>
                </div>
                
                <!-- Retour Dashboard -->
                <a href="dashboard.php" class="btn btn-primary">
                    📊 Dashboard Principal
                </a>
            </div>
        </header>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type ?>">
                <span><?= $message ?></span>
            </div>
        <?php endif; ?>

        <!-- Stats Vue d'ensemble -->
        <div class="stats-grid">
            <div class="stat-card">
                <span class="stat-icon">👥</span>
                <div class="stat-value"><?= $tokens_overview['users_with_tokens'] ?></div>
                <div class="stat-label">Utilisateurs avec tokens</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">📦</span>
                <div class="stat-value"><?= number_format($tokens_overview['total_packages']) ?></div>
                <div class="stat-label">Packages actifs</div>
                <div class="stat-details"><?= $tokens_overview['packages_last_7d'] ?> cette semaine</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">💰</span>
                <div class="stat-value"><?= number_format($tokens_overview['total_tokens_distributed']) ?></div>
                <div class="stat-label">Tokens distribués</div>
                <div class="stat-details">Moy: <?= number_format($tokens_overview['avg_package_size']) ?> par package</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">⚡</span>
                <div class="stat-value"><?= number_format($tokens_overview['total_tokens_consumed']) ?></div>
                <div class="stat-label">Tokens consommés</div>
                <div class="stat-details"><?= number_format($tokens_overview['total_tokens_remaining']) ?> restants</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">��</span>
                <div class="stat-value"><?= number_format($tokens_overview['total_revenue'], 2) ?>€</div>
                <div class="stat-label">Revenus totaux</div>
            </div>
        </div>

        <!-- Onglets Gestion -->
        <div class="content-card">
            <div class="tabs">
                <button class="tab active" onclick="showTab('users_balance')">👥 Soldes Utilisateurs</button>
                <button class="tab" onclick="showTab('packages_catalog')">📦 Packages Catalogue</button>
                <button class="tab" onclick="showTab('ai_quotas')">🤖 Quotas IA</button>
                <button class="tab" onclick="showTab('transactions')">📋 Transactions</button>
                <button class="tab" onclick="showTab('analytics')">📊 Analytics</button>
            </div>

            <!-- Onglet Soldes Utilisateurs -->
            <div id="users_balance" class="tab-content active">
                <div class="card-header">
                    <h3 class="card-title">👥 Soldes Tokens Utilisateurs</h3>
                    <div class="action-buttons">
                        <button onclick="openManualAttributionModal()" class="btn btn-success">
                            ➕ Attribution Manuelle
                        </button>
                        <button onclick="refreshData()" class="btn btn-primary">
                            🔄 Actualiser
                        </button>
                    </div>
                </div>
                
                <table class="tokens-table">
                    <thead>
                        <tr>
                            <th>👤 Utilisateur</th>
                            <th>💰 Solde actuel</th>
                            <th>📦 Total acheté</th>
                            <th>⚡ Total utilisé</th>
                            <th>🤖 IA consommé</th>
                            <th>💶 Dépensé</th>
                            <th>📅 Dernier achat</th>
                            <th>⚡ Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users_tokens as $user): ?>
                            <tr>
                                <td>
                                    <strong><?= htmlspecialchars($user['name']) ?></strong>
                                    <br><small><?= htmlspecialchars($user['email']) ?></small>
                                </td>
                                <td>
                                    <strong style="color: #27ae60;"><?= number_format($user['current_balance']) ?></strong>
                                    <?php 
                                    $percentage = $user['total_purchased'] > 0 ? ($user['current_balance'] / $user['total_purchased']) * 100 : 0;
                                    $bar_color = $percentage > 50 ? '#27ae60' : ($percentage > 20 ? '#f39c12' : '#e74c3c');
                                    ?>
                                    <div class="progress-bar">
                                        <div class="progress-fill" style="width: <?= $percentage ?>%; background: <?= $bar_color ?>;"></div>
                                    </div>
                                </td>
                                <td><?= number_format($user['total_purchased']) ?></td>
                                <td><?= number_format($user['total_used']) ?></td>
                                <td>
                                    <?= number_format($user['ai_tokens_used']) ?>
                                    <br><small><?= number_format($user['ai_total_cost'], 2) ?>€</small>
                                </td>
                                <td><?= number_format($user['total_spent'], 2) ?>€</td>
                                <td><?= formatDate($user['last_purchase_date']) ?></td>
                                <td>
                                    <div class="action-buttons">
                                        <button onclick="addTokensToUser(<?= $user['id'] ?>, '<?= htmlspecialchars($user['name']) ?>')" class="btn btn-success btn-sm">
                                            ➕ Ajouter
                                        </button>
                                        <button onclick="viewUserDetails(<?= $user['id'] ?>)" class="btn btn-primary btn-sm">
                                            👁️ Détails
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Onglet Packages Catalogue -->
            <div id="packages_catalog" class="tab-content">
                <div class="card-header">
                    <h3 class="card-title">📦 Catalogue Packages</h3>
                    <button onclick="openCreatePackageModal()" class="btn btn-success">
                        ➕ Nouveau Package
                    </button>
                </div>
                
                <table class="tokens-table">
                    <thead>
                        <tr>
                            <th>📦 Package</th>
                            <th>💰 Tokens</th>
                            <th>💶 Prix</th>
                            <th>📊 Ventes</th>
                            <th>💰 Revenus</th>
                            <th>⭐ Populaire</th>
                            <th>⚡ Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($packages_catalog as $package): ?>
                            <tr>
                                <td>
                                    <strong><?= htmlspecialchars($package['package_name']) ?></strong>
                                    <br><small><?= htmlspecialchars($package['package_key']) ?></small>
                                    <?php if ($package['description']): ?>
                                        <br><em><?= htmlspecialchars(substr($package['description'], 0, 50)) ?>...</em>
                                    <?php endif; ?>
                                </td>
                                <td><strong><?= number_format($package['tokens_included']) ?></strong></td>
                                <td><strong><?= number_format($package['price_eur'], 2) ?>€</strong></td>
                                <td>
                                    <?= number_format($package['sales_count']) ?> ventes
                                    <br><small><?= number_format($package['tokens_sold']) ?> tokens vendus</small>
                                </td>
                                <td><?= number_format($package['revenue_generated'], 2) ?>€</td>
                                <td>
                                    <?php if ($package['is_popular']): ?>
                                        <span class="badge badge-warning">⭐ Populaire</span>
                                    <?php else: ?>
                                        <span class="badge badge-secondary">Standard</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <button onclick="editPackage(<?= $package['id'] ?>)" class="btn btn-warning btn-sm">
                                            ✏️ Éditer
                                        </button>
                                        <button onclick="togglePackageStatus(<?= $package['id'] ?>)" class="btn btn-primary btn-sm">
                                            🔄 Toggle
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Onglet Quotas IA -->
            <div id="ai_quotas" class="tab-content">
                <div class="card-header">
                    <h3 class="card-title">🤖 Quotas IA Actifs</h3>
                    <button onclick="openCreateQuotaModal()" class="btn btn-success">
                        ➕ Nouveau Quota
                    </button>
                </div>
                
                <table class="tokens-table">
                    <thead>
                        <tr>
                            <th>👤 Utilisateur</th>
                            <th>📅 Période</th>
                            <th>💰 Limite tokens</th>
                            <th>📊 Utilisation</th>
                            <th>💶 Coût limite</th>
                            <th>🗓️ Fin quota</th>
                            <th>⚠️ Statut</th>
                            <th>⚡ Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($active_quotas as $quota): ?>
                            <tr>
                                <td>
                                    <strong><?= htmlspecialchars($quota['user_name']) ?></strong>
                                    <br><small><?= htmlspecialchars($quota['user_email']) ?></small>
                                </td>
                                <td>
                                    <span class="badge badge-info"><?= ucfirst($quota['quota_period']) ?></span>
                                    <br><small><?= formatDate($quota['quota_start_date']) ?></small>
                                </td>
                                <td><?= number_format($quota['tokens_limit']) ?></td>
                                <td>
                                    <strong><?= number_format($quota['tokens_used']) ?></strong> / <?= number_format($quota['tokens_limit']) ?>
                                    <div class="progress-bar">
                                        <div class="progress-fill" style="width: <?= min(100, $quota['usage_percentage']) ?>%; 
                                            background: <?= $quota['status'] === 'OK' ? '#27ae60' : ($quota['status'] === 'WARNING' ? '#f39c12' : '#e74c3c') ?>;"></div>
                                    </div>
                                    <small><?= $quota['usage_percentage'] ?>%</small>
                                </td>
                                <td>
                                    <?= number_format($quota['cost_used'], 2) ?>€ / <?= number_format($quota['cost_limit'], 2) ?>€
                                </td>
                                <td>
                                    <?= formatDate($quota['quota_end_date']) ?>
                                    <br><small><?= $quota['days_remaining'] ?> jours restants</small>
                                </td>
                                <td><?= getStatusBadge($quota['status']) ?></td>
                                <td>
                                    <div class="action-buttons">
                                        <button onclick="editQuota(<?= $quota['id'] ?>)" class="btn btn-warning btn-sm">
                                            ✏️ Éditer
                                        </button>
                                        <button onclick="resetQuota(<?= $quota['id'] ?>)" class="btn btn-primary btn-sm">
                                            🔄 Reset
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Onglet Transactions -->
            <div id="transactions" class="tab-content">
                <div class="card-header">
                    <h3 class="card-title">📋 Transactions Récentes</h3>
                    <button onclick="exportTransactions()" class="btn btn-primary">
                        📊 Exporter
                    </button>
                </div>
                
                <table class="tokens-table">
                    <thead>
                        <tr>
                            <th>📅 Date</th>
                            <th>👤 Utilisateur</th>
                            <th>📦 Package</th>
                            <th>💰 Tokens</th>
                            <th>💶 Prix</th>
                            <th>📋 Référence</th>
                            <th>📊 Statut</th>
                            <th>👔 Admin</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent_transactions as $transaction): ?>
                            <tr>
                                <td><?= formatDate($transaction['created_at']) ?></td>
                                <td>
                                    <strong><?= htmlspecialchars($transaction['user_name']) ?></strong>
                                    <br><small><?= htmlspecialchars($transaction['user_email']) ?></small>
                                </td>
                                <td>
                                    <span class="badge badge-info"><?= htmlspecialchars($transaction['package_name']) ?></span>
                                </td>
                                <td><strong><?= number_format($transaction['tokens_purchased']) ?></strong></td>
                                <td><?= number_format($transaction['price_paid'], 2) ?>€</td>
                                <td><small><?= htmlspecialchars($transaction['payment_reference']) ?></small></td>
                                <td><?= getStatusBadge($transaction['status']) ?></td>
                                <td>
                                    <?php if ($transaction['admin_name']): ?>
                                        <small>👔 <?= htmlspecialchars($transaction['admin_name']) ?></small>
                                    <?php else: ?>
                                        <small>🤖 Automatique</small>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Onglet Analytics -->
            <div id="analytics" class="tab-content">
                <div class="card-header">
                    <h3 class="card-title">📊 Analytics Consommation IA</h3>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <h4>📈 Consommation quotidienne (30 derniers jours)</h4>
                </div>
                
                <table class="tokens-table">
                    <thead>
                        <tr>
                            <th>📅 Date</th>
                            <th>📊 Requêtes</th>
                            <th>💰 Tokens</th>
                            <th>💶 Coût</th>
                            <th>👥 Utilisateurs actifs</th>
                            <th>❌ Échecs</th>
                            <th>⏱️ Temps moyen</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($ai_consumption as $day): ?>
                            <tr>
                                <td><?= date('d/m/Y', strtotime($day['consumption_date'])) ?></td>
                                <td><strong><?= number_format($day['requests_count']) ?></strong></td>
                                <td><strong><?= number_format($day['tokens_consumed']) ?></strong></td>
                                <td><?= number_format($day['total_cost'], 2) ?>€</td>
                                <td><?= number_format($day['active_users']) ?></td>
                                <td>
                                    <?php if ($day['failed_requests'] > 0): ?>
                                        <span class="badge badge-danger"><?= $day['failed_requests'] ?></span>
                                    <?php else: ?>
                                        <span class="badge badge-success">0</span>
                                    <?php endif; ?>
                                </td>
                                <td><?= number_format($day['avg_processing_time']) ?>ms</td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Modal Attribution Manuelle -->
    <div id="manualAttributionModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">💰 Attribution Manuelle de Tokens</h3>
                <button type="button" class="close-btn" onclick="closeModal('manualAttributionModal')">&times;</button>
            </div>
            
            <form method="POST">
                <input type="hidden" name="action" value="manual_attribution">
                
                <div class="form-group">
                    <label class="form-label">👤 Utilisateur</label>
                    <select name="user_id" class="form-select" required>
                        <option value="">Sélectionner un utilisateur</option>
                        <?php foreach ($users_tokens as $user): ?>
                            <option value="<?= $user['id'] ?>"><?= htmlspecialchars($user['name']) ?> (<?= htmlspecialchars($user['email']) ?>)</option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">💰 Nombre de tokens</label>
                    <input type="number" name="tokens_amount" class="form-control" min="1" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">📦 Type d'attribution</label>
                    <select name="package_name" class="form-select">
                        <option value="admin_manual">🎁 Attribution manuelle</option>
                        <option value="bonus">🎉 Bonus</option>
                        <option value="compensation">🔧 Compensation</option>
                        <option value="promotion">🏷️ Promotion</option>
                        <option value="refund">🔙 Remboursement</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">📝 Notes (optionnel)</label>
                    <textarea name="notes" class="form-control" rows="3" placeholder="Motif de l'attribution..."></textarea>
                </div>
                
                <div class="action-buttons">
                    <button type="submit" class="btn btn-success">
                        ✅ Attribuer les tokens
                    </button>
                    <button type="button" onclick="closeModal('manualAttributionModal')" class="btn btn-secondary">
                        ❌ Annuler
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal Création Package -->
    <div id="createPackageModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">📦 Créer un Package Catalogue</h3>
                <button type="button" class="close-btn" onclick="closeModal('createPackageModal')">&times;</button>
            </div>
            
            <form method="POST">
                <input type="hidden" name="action" value="create_package_catalog">
                
                <div class="form-group">
                    <label class="form-label">🔑 Clé package (identifiant unique)</label>
                    <input type="text" name="package_key" class="form-control" required placeholder="ex: starter_1000">
                </div>
                
                <div class="form-group">
                    <label class="form-label">📦 Nom du package</label>
                    <input type="text" name="package_name" class="form-control" required placeholder="ex: Starter 1000 tokens">
                </div>
                
                <div class="form-group">
                    <label class="form-label">💰 Tokens inclus</label>
                    <input type="number" name="tokens_included" class="form-control" min="1" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">💶 Prix en euros</label>
                    <input type="number" name="price_eur" class="form-control" step="0.01" min="0" required>
                </div>
                
                <div class="form-group">
                    <label class="form-label">📝 Description</label>
                    <textarea name="description" class="form-control" rows="3" placeholder="Description du package..."></textarea>
                </div>
                
                <div class="form-group">
                    <label>
                        <input type="checkbox" name="is_popular"> ⭐ Marquer comme populaire
                    </label>
                </div>
                
                <div class="action-buttons">
                    <button type="submit" class="btn btn-success">
                        ✅ Créer le package
                    </button>
                    <button type="button" onclick="closeModal('createPackageModal')" class="btn btn-secondary">
                        ❌ Annuler
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Gestion des onglets
        function showTab(tabName) {
            // Masquer tous les contenus
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            
            // Retirer active de tous les onglets
            document.querySelectorAll('.tab').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Afficher le contenu sélectionné
            document.getElementById(tabName).classList.add('active');
            
            // Activer l'onglet
            event.target.classList.add('active');
        }

        // Gestion des modals
        function openManualAttributionModal() {
            document.getElementById('manualAttributionModal').style.display = 'flex';
        }

        function openCreatePackageModal() {
            document.getElementById('createPackageModal').style.display = 'flex';
        }

        function closeModal(modalId) {
            document.getElementById(modalId).style.display = 'none';
        }

        // Actions utilisateurs
        function addTokensToUser(userId, userName) {
            // Pré-remplir le modal d'attribution
            const select = document.querySelector('#manualAttributionModal select[name="user_id"]');
            select.value = userId;
            openManualAttributionModal();
        }

        function viewUserDetails(userId) {
            window.open(`users_management.php?action=edit&user_id=${userId}`, '_blank');
        }

        // Actions packages
        function editPackage(packageId) {
            alert('🚧 Fonctionnalité en développement : Édition package #' + packageId);
        }

        function togglePackageStatus(packageId) {
            if (confirm('Changer le statut de ce package ?')) {
                // Implémenter toggle status
                alert('🚧 Fonctionnalité en développement : Toggle package #' + packageId);
            }
        }

        // Actions quotas
        function openCreateQuotaModal() {
            alert('🚧 Fonctionnalité en développement : Création quota IA');
        }

        function editQuota(quotaId) {
            alert('🚧 Fonctionnalité en développement : Édition quota #' + quotaId);
        }

        function resetQuota(quotaId) {
            if (confirm('⚠️ Remettre à zéro ce quota IA ?')) {
                alert('🚧 Fonctionnalité en développement : Reset quota #' + quotaId);
            }
        }

        // Actions diverses
        function refreshData() {
            window.location.reload();
        }

        function exportTransactions() {
            alert('🚧 Fonctionnalité en développement : Export transactions');
        }

        // Fermeture modal en cliquant à l'extérieur
        window.onclick = function(event) {
            const modals = document.querySelectorAll('.modal');
            modals.forEach(modal => {
                if (event.target === modal) {
                    modal.style.display = 'none';
                }
            });
        }

        console.log('💰 SMG Studio v2.0 - Gestion Tokens Ready!');
        console.log('✅ Hub central tokens + quotas IA par Papy 77');
        console.log('🔧 Attribution manuelle + analytics temps réel');
    </script>
</body>
</html>