<?php
/**
 * ================================================================
 * 🔗 SOCIAL-ACCOUNTS.PHP - MULTI-USER ENHANCED avec CSS UNIFIÉ
 * ================================================================
 * 
 * 🎯 FONCTIONNALITÉS MULTI-USER :
 * - Isolation complète par utilisateur
 * - Gestion des rôles et permissions
 * - Quotas par utilisateur
 * - Audit trail complet
 * - Sécurité renforcée
 * - CSS UNIFIÉ SMG Studio v2.0
 * 
 * 💡 BY PAPY 77 - VERSION MULTI-USER COMPLÈTE + DESIGN HARMONISÉ !
 * ================================================================
 */

require_once 'auth_functions.php';
requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

// ================================================================
// 🔒 VÉRIFICATIONS SÉCURITÉ MULTI-USER
// ================================================================

// Vérifier les permissions utilisateur
if (!hasPermission($user, 'manage_social_accounts')) {
    die('❌ Vous n\'avez pas les permissions pour gérer les comptes sociaux');
}

// Récupérer les limites utilisateur
$user_limits = getUserLimits($user['id'], $pdo);

// Configuration des réseaux sociaux (inchangée)
$networks_config = [
    'facebook' => [
        'name' => 'Facebook',
        'icon' => '📘',
        'color' => '#1877f2',
        'description' => 'Pages et comptes business',
        'max_accounts' => $user_limits['facebook_accounts'] ?? 1,
        'fields' => [
            'access_token' => ['label' => 'Page Access Token', 'type' => 'password', 'required' => true],
            'page_id' => ['label' => 'Page ID', 'type' => 'text', 'required' => true],
            'app_id' => ['label' => 'App ID', 'type' => 'text', 'required' => false]
        ]
    ],
    'instagram' => [
        'name' => 'Instagram',
        'icon' => '📸',
        'color' => '#E4405F',
        'description' => 'Compte business/créateur',
        'max_accounts' => $user_limits['instagram_accounts'] ?? 1,
        'fields' => [
            'instagram_id' => ['label' => 'Instagram Business Account ID', 'type' => 'text', 'required' => true],
            'access_token' => ['label' => 'Facebook Page Access Token', 'type' => 'password', 'required' => true],
            'page_id' => ['label' => 'Facebook Page ID', 'type' => 'text', 'required' => true]
        ]
    ],
    'tiktok' => [
        'name' => 'TikTok',
        'icon' => '🎵',
        'color' => '#000000',
        'description' => 'TikTok for Business',
        'max_accounts' => $user_limits['tiktok_accounts'] ?? 1,
        'fields' => [
            'access_token' => ['label' => 'Access Token', 'type' => 'password', 'required' => true],
            'user_id' => ['label' => 'User ID', 'type' => 'text', 'required' => true],
            'client_key' => ['label' => 'Client Key', 'type' => 'text', 'required' => false]
        ]
    ],
    'linkedin' => [
        'name' => 'LinkedIn',
        'icon' => '💼',
        'color' => '#0A66C2',
        'description' => 'Pages et profils professionnels',
        'max_accounts' => $user_limits['linkedin_accounts'] ?? 1,
        'fields' => [
            'access_token' => ['label' => 'Access Token', 'type' => 'password', 'required' => true],
            'organization_id' => ['label' => 'Organization ID', 'type' => 'text', 'required' => true],
            'client_id' => ['label' => 'Client ID', 'type' => 'text', 'required' => false]
        ]
    ],
    'snapchat' => [
        'name' => 'Snapchat',
        'icon' => '👻',
        'color' => '#FFFC00',
        'description' => 'Snapchat Business',
        'max_accounts' => $user_limits['snapchat_accounts'] ?? 1,
        'fields' => [
            'access_token' => ['label' => 'Access Token', 'type' => 'password', 'required' => true],
            'ad_account_id' => ['label' => 'Ad Account ID', 'type' => 'text', 'required' => true],
            'client_id' => ['label' => 'Client ID', 'type' => 'text', 'required' => false]
        ]
    ]
];

// ================================================================
// 🗄️ CHARGEMENT COMPTES (ISOLATION PAR USER_ID)
// ================================================================

$connected_accounts = [];
$account_counts = [];

try {
    // Récupérer SEULEMENT les comptes de l'utilisateur connecté
    $stmt = $pdo->prepare("
        SELECT platform, username, platform_user_id, is_active, 
               created_at, updated_at, account_data, id
        FROM social_accounts 
        WHERE user_id = ? 
        ORDER BY platform, created_at DESC
    ");
    $stmt->execute([$user['id']]);
    $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($accounts as $account) {
        if (!isset($connected_accounts[$account['platform']])) {
            $connected_accounts[$account['platform']] = [];
            $account_counts[$account['platform']] = 0;
        }
        $connected_accounts[$account['platform']][] = $account;
        $account_counts[$account['platform']]++;
    }
    
} catch (PDOException $e) {
    error_log("Erreur chargement comptes sociaux (user_id: {$user['id']}): " . $e->getMessage());
}

// ================================================================
// 📊 STATISTIQUES UTILISATEUR
// ================================================================

$user_stats = [
    'total_accounts' => array_sum($account_counts),
    'active_accounts' => 0,
    'last_connection' => null
];

foreach ($connected_accounts as $platform_accounts) {
    foreach ($platform_accounts as $account) {
        if ($account['is_active']) {
            $user_stats['active_accounts']++;
        }
    }
}

// ================================================================
// ⚡ TRAITEMENT ACTIONS AVEC SÉCURITÉ RENFORCÉE
// ================================================================

$message = '';
$message_type = '';

if ($_POST) {
    $action = $_POST['action'] ?? '';
    
    // Vérifier le token CSRF
    if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf_token'] ?? '')) {
        $message = 'Token de sécurité invalide';
        $message_type = 'error';
    } else {
        switch ($action) {
            case 'connect':
                $result = connectSocialAccountSecure($_POST, $user, $pdo, $networks_config);
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'error';
                break;
                
            case 'disconnect':
                $result = disconnectSocialAccountSecure($_POST, $user, $pdo);
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'error';
                break;
                
            case 'test':
                $result = testSocialConnectionSecure($_POST, $user, $pdo);
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'error';
                break;
        }
        
        // Rechargement des comptes après action
        if ($message_type === 'success') {
            header("Location: " . $_SERVER['PHP_SELF'] . "?success=1");
            exit;
        }
    }
}

// Générer token CSRF pour les formulaires
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// ================================================================
// 🔒 FONCTIONS SÉCURISÉES MULTI-USER
// ================================================================

/**
 * Connexion sécurisée avec vérifications multi-user
 */
function connectSocialAccountSecure($data, $user, $pdo, $networks_config) {
    $platform = $data['platform'] ?? '';
    $credentials = $data['credentials'] ?? [];
    
    // Vérifications de base
    if (!$platform || !$credentials['access_token']) {
        return ['success' => false, 'message' => 'Données manquantes'];
    }
    
    if (!isset($networks_config[$platform])) {
        return ['success' => false, 'message' => 'Plateforme non supportée'];
    }
    
    // Vérifier les limites utilisateur
    $current_count = getCurrentAccountCount($user['id'], $platform, $pdo);
    $max_allowed = $networks_config[$platform]['max_accounts'];
    
    if ($current_count >= $max_allowed) {
        return ['success' => false, 'message' => "Limite atteinte: {$max_allowed} compte(s) maximum pour {$platform}"];
    }
    
    try {
        // Validation du token
        $validation = validateToken($platform, $credentials);
        
        if (!$validation['success']) {
            // Log de tentative échouée
            logSecurityEvent($user['id'], 'invalid_token_attempt', $platform, 
                           "Tentative de connexion avec token invalide", $pdo);
            return ['success' => false, 'message' => 'Token invalide: ' . $validation['error']];
        }
        
        // Vérifier si ce compte n'existe pas déjà
        if (accountAlreadyExists($user['id'], $platform, $validation['user_id'], $pdo)) {
            return ['success' => false, 'message' => 'Ce compte est déjà connecté'];
        }
        
        // Chiffrement sécurisé des credentials
        $encrypted_token = encryptTokenSecure($credentials['access_token'], $user['id']);
        $account_data = json_encode($credentials);
        
        // Sauvegarde avec audit
        $stmt = $pdo->prepare("
            INSERT INTO social_accounts 
            (user_id, platform, platform_user_id, username, access_token, 
             account_data, is_active, ip_address, user_agent)
            VALUES (?, ?, ?, ?, ?, ?, 1, ?, ?)
        ");
        
        $stmt->execute([
            $user['id'],
            $platform,
            $validation['user_id'] ?? 'unknown',
            $validation['username'] ?? 'user',
            $encrypted_token,
            $account_data,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ]);
        
        // Log de succès
        logSecurityEvent($user['id'], 'connect_social', $platform, 
                       "Connexion réussie: {$validation['username']}", $pdo);
        
        return ['success' => true, 'message' => ucfirst($platform) . ' connecté avec succès !'];
        
    } catch (Exception $e) {
        error_log("Erreur connexion $platform (user_id: {$user['id']}): " . $e->getMessage());
        logSecurityEvent($user['id'], 'connect_error', $platform, 
                       "Erreur de connexion: " . $e->getMessage(), $pdo);
        return ['success' => false, 'message' => 'Erreur de connexion: ' . $e->getMessage()];
    }
}

/**
 * Déconnexion sécurisée avec vérification de propriété
 */
function disconnectSocialAccountSecure($data, $user, $pdo) {
    $platform = $data['platform'] ?? '';
    $account_id = $data['account_id'] ?? null;
    
    if (!$platform) {
        return ['success' => false, 'message' => 'Plateforme manquante'];
    }
    
    try {
        // Vérifier que le compte appartient bien à l'utilisateur
        $stmt = $pdo->prepare("
            SELECT id, username FROM social_accounts 
            WHERE user_id = ? AND platform = ? AND (id = ? OR ? IS NULL)
            LIMIT 1
        ");
        $stmt->execute([$user['id'], $platform, $account_id, $account_id]);
        $account = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$account) {
            logSecurityEvent($user['id'], 'unauthorized_disconnect', $platform, 
                           "Tentative de déconnexion d'un compte non possédé", $pdo);
            return ['success' => false, 'message' => 'Compte non trouvé ou non autorisé'];
        }
        
        // Supprimer le compte
        $stmt = $pdo->prepare("DELETE FROM social_accounts WHERE id = ? AND user_id = ?");
        $stmt->execute([$account['id'], $user['id']]);
        
        logSecurityEvent($user['id'], 'disconnect_social', $platform, 
                       "Déconnexion: {$account['username']}", $pdo);
        
        return ['success' => true, 'message' => ucfirst($platform) . ' déconnecté'];
        
    } catch (PDOException $e) {
        error_log("Erreur déconnexion $platform (user_id: {$user['id']}): " . $e->getMessage());
        return ['success' => false, 'message' => 'Erreur de déconnexion'];
    }
}

/**
 * Test sécurisé avec vérification de propriété
 */
function testSocialConnectionSecure($data, $user, $pdo) {
    $platform = $data['platform'] ?? '';
    $account_id = $data['account_id'] ?? null;
    
    try {
        // Vérifier ownership
        $stmt = $pdo->prepare("
            SELECT access_token, username FROM social_accounts 
            WHERE user_id = ? AND platform = ? AND (id = ? OR ? IS NULL)
            LIMIT 1
        ");
        $stmt->execute([$user['id'], $platform, $account_id, $account_id]);
        $account = $stmt->fetch();
        
        if (!$account) {
            return ['success' => false, 'message' => 'Compte non autorisé'];
        }
        
        $decrypted_token = decryptTokenSecure($account['access_token'], $user['id']);
        $test_result = performAPITest($platform, $decrypted_token);
        
        // Log du test
        logSecurityEvent($user['id'], 'test_connection', $platform, 
                       "Test connexion: " . ($test_result['success'] ? 'Succès' : 'Échec'), $pdo);
        
        return $test_result;
        
    } catch (Exception $e) {
        error_log("Erreur test $platform (user_id: {$user['id']}): " . $e->getMessage());
        return ['success' => false, 'message' => 'Erreur de test'];
    }
}

// ================================================================
// 🛡️ FONCTIONS UTILITAIRES SÉCURITÉ
// ================================================================

/**
 * Vérifier les permissions utilisateur
 */
function hasPermission($user, $permission) {
    $permissions = [
        'user' => ['manage_social_accounts'],
        'manager' => ['manage_social_accounts', 'view_all_accounts'],
        'admin' => ['manage_social_accounts', 'view_all_accounts', 'manage_users']
    ];
    
    return in_array($permission, $permissions[$user['role']] ?? []);
}

/**
 * Récupérer les limites utilisateur
 */
function getUserLimits($user_id, $pdo) {
    try {
        $stmt = $pdo->prepare("
            SELECT platform, max_accounts FROM user_limits 
            WHERE user_id = ?
        ");
        $stmt->execute([$user_id]);
        $limits = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        // Limites par défaut
        $default_limits = [
            'facebook_accounts' => 2,
            'instagram_accounts' => 2,
            'tiktok_accounts' => 1,
            'linkedin_accounts' => 1,
            'snapchat_accounts' => 1
        ];
        
        return array_merge($default_limits, $limits);
        
    } catch (PDOException $e) {
        error_log("Erreur récupération limites user_id $user_id: " . $e->getMessage());
        return [
            'facebook_accounts' => 1,
            'instagram_accounts' => 1,
            'tiktok_accounts' => 1,
            'linkedin_accounts' => 1,
            'snapchat_accounts' => 1
        ];
    }
}

/**
 * Compter les comptes actuels
 */
function getCurrentAccountCount($user_id, $platform, $pdo) {
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM social_accounts 
            WHERE user_id = ? AND platform = ?
        ");
        $stmt->execute([$user_id, $platform]);
        return (int) $stmt->fetchColumn();
    } catch (PDOException $e) {
        return 0;
    }
}

/**
 * Vérifier si le compte existe déjà
 */
function accountAlreadyExists($user_id, $platform, $platform_user_id, $pdo) {
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM social_accounts 
            WHERE user_id = ? AND platform = ? AND platform_user_id = ?
        ");
        $stmt->execute([$user_id, $platform, $platform_user_id]);
        return $stmt->fetchColumn() > 0;
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * Chiffrement sécurisé avec clé utilisateur
 */
function encryptTokenSecure($token, $user_id) {
    $key = hash('sha256', 'SMG_SECRET_KEY_PAPY77' . $user_id);
    $iv = random_bytes(16);
    $encrypted = openssl_encrypt($token, 'AES-256-CBC', $key, 0, $iv);
    return base64_encode($iv . $encrypted);
}

/**
 * Déchiffrement sécurisé avec clé utilisateur
 */
function decryptTokenSecure($encrypted_token, $user_id) {
    $key = hash('sha256', 'SMG_SECRET_KEY_PAPY77' . $user_id);
    $data = base64_decode($encrypted_token);
    $iv = substr($data, 0, 16);
    $encrypted = substr($data, 16);
    return openssl_decrypt($encrypted, 'AES-256-CBC', $key, 0, $iv);
}

/**
 * Log des événements de sécurité
 */
function logSecurityEvent($user_id, $action, $platform, $description, $pdo) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO security_logs 
            (user_id, action, platform, description, ip_address, user_agent, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $user_id, 
            $action, 
            $platform, 
            $description, 
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ]);
    } catch (PDOException $e) {
        error_log("Erreur log sécurité: " . $e->getMessage());
    }
}

// ================================================================
// FONCTIONS API (simulées pour l'exemple)
// ================================================================

function validateToken($platform, $credentials) {
    // Simulation de validation de token
    return [
        'success' => true,
        'user_id' => 'user_' . random_int(1000, 9999),
        'username' => $platform . '_user_' . random_int(100, 999)
    ];
}

function performAPITest($platform, $token) {
    // Simulation de test API
    return [
        'success' => true,
        'message' => "Test de connexion $platform réussi"
    ];
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🔗 Comptes Sociaux - <?php echo htmlspecialchars($user['name'] ?? 'Utilisateur'); ?> - SMG</title>
    
    <style>
        /* =============================================================================
           SMG Studio v2.0 - CSS ULTRA CONTRASTE - Lisibilité Maximale
           Texte noir + Cadres gris foncé + Aucun effet
           ============================================================================= */

        :root {
            /* Couleurs principales */
            --primary-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            --secondary-gradient: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            --success-gradient: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
            --warning-gradient: linear-gradient(135deg, #ffd700 0%, #ff8c00 100%);
            --danger-gradient: linear-gradient(135deg, #ff6b6b 0%, #ee5a52 100%);
            
            /* Couleurs neutres */
            --bg-primary: #667eea;
            --bg-secondary: #764ba2;
            --bg-success: #48bb78;
            --bg-warning: #ffd700;
            --bg-danger: #ff6b6b;
            
            /* TEXTE ULTRA CONTRASTE */
            --text-primary: #000000;      /* NOIR PUR */
            --text-secondary: #2d3748;    /* GRIS TRÈS FONCÉ */
            --text-dark: #000000;         /* NOIR PUR */
            --text-muted: #4a5568;        /* GRIS FONCÉ */
            --text-white: #ffffff;        /* BLANC pour boutons */
            
            /* BACKGROUNDS CONTRASTÉS */
            --glass-bg: #f7fafc;          /* BLANC CASSÉ */
            --glass-bg-light: #ffffff;    /* BLANC PUR */
            --glass-bg-strong: #ffffff;   /* BLANC PUR */
            --glass-bg-dark: #2d3748;     /* GRIS FONCÉ */
            
            /* BORDURES CONTRASTÉES */
            --glass-border: #2d3748;      /* GRIS FONCÉ */
            --glass-border-strong: #1a202c; /* GRIS TRÈS FONCÉ */
            --border-light: #e2e8f0;      /* GRIS CLAIR */
            
            /* OMBRES SUPPRIMÉES */
            --shadow-light: none;
            --shadow-medium: none;
            --shadow-strong: none;
            --shadow-text: none;
            
            /* Rayons et espacements inchangés */
            --radius-small: 8px;
            --radius-medium: 12px;
            --radius-large: 16px;
            --radius-xl: 20px;
            
            --spacing-xs: 8px;
            --spacing-sm: 12px;
            --spacing-md: 16px;
            --spacing-lg: 24px;
            --spacing-xl: 32px;
            --spacing-2xl: 48px;
            
            --transition-fast: all 0.2s ease;
            --transition-normal: all 0.3s ease;
            --transition-slow: all 0.5s ease;
        }

        /* Reset et base */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: var(--primary-gradient);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
            overflow-x: hidden;
            
            /* SUPPRESSION TOUS EFFETS TEXTE */
            text-shadow: none;
            -webkit-font-smoothing: auto;
            -moz-osx-font-smoothing: auto;
        }

        /* Container principal */
        .smg-container {
            max-width: 1600px;
            margin: 0 auto;
            padding: var(--spacing-lg);
        }

        /* =============================================================================
           HEADERS ULTRA CONTRASTÉS
           ============================================================================= */

        .smg-header {
            background: var(--glass-bg-strong);
            border-radius: var(--radius-xl);
            padding: var(--spacing-xl);
            margin-bottom: var(--spacing-xl);
            border: 2px solid var(--glass-border);
            position: relative;
            
            /* CONTRASTE MAXIMAL */
            background: #ffffff;
            color: var(--text-primary);
            box-shadow: none;
            backdrop-filter: none;
        }

        .smg-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: var(--primary-gradient);
        }

        .smg-header-top {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: var(--spacing-lg);
            width: 100%;
        }

        .smg-back-btn {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-xs);
            color: var(--bg-primary);
            text-decoration: none;
            font-weight: 600;
            padding: var(--spacing-sm) var(--spacing-md);
            border-radius: var(--radius-small);
            transition: var(--transition-normal);
            background: var(--glass-bg);
            border: 2px solid var(--glass-border);
            
            /* SUPPRESSION EFFETS */
            text-shadow: none;
            box-shadow: none;
        }

        .smg-back-btn:hover {
            background: var(--border-light);
            transform: none;
            color: var(--bg-secondary);
        }

        .smg-user-info {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
            background: var(--glass-bg);
            padding: var(--spacing-sm) var(--spacing-md);
            border-radius: var(--radius-medium);
            border: 2px solid var(--glass-border);
        }

        .smg-user-avatar {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            background: var(--bg-primary);
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: 14px;
            color: var(--text-white);
            text-shadow: none;
        }

        .smg-user-details {
            font-size: 12px;
        }

        .smg-user-name {
            font-weight: 600;
            color: var(--text-primary);
            text-shadow: none;
        }

        .smg-user-role {
            color: var(--text-muted);
            text-shadow: none;
        }

        .smg-header-title {
            font-size: 32px;
            font-weight: 700;
            background: var(--primary-gradient);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: var(--spacing-sm);
            text-shadow: none;
        }

        .smg-header-subtitle {
            color: var(--text-muted);
            font-size: 16px;
            text-shadow: none;
        }

        /* =============================================================================
           PANELS ULTRA CONTRASTÉS
           ============================================================================= */

        .smg-panel {
            background: #ffffff;
            border-radius: var(--radius-large);
            padding: var(--spacing-xl);
            border: 2px solid var(--glass-border);
            margin-bottom: var(--spacing-xl);
            color: var(--text-primary);
            
            /* SUPPRESSION EFFETS */
            backdrop-filter: none;
            box-shadow: none;
        }

        .smg-panel-header {
            color: var(--bg-primary);
            margin-bottom: var(--spacing-xl);
            font-size: 20px;
            border-bottom: 2px solid var(--bg-primary);
            padding-bottom: var(--spacing-sm);
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            text-shadow: none;
        }

        /* =============================================================================
           STATISTIQUES UTILISATEUR
           ============================================================================= */

        .user-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: var(--spacing-lg);
            margin: var(--spacing-xl) 0;
        }

        .stat-card {
            background: #ffffff;
            border: 2px solid var(--glass-border);
            border-radius: var(--radius-medium);
            padding: var(--spacing-lg);
            text-align: center;
            transition: var(--transition-normal);
        }

        .stat-card:hover {
            border-color: var(--bg-primary);
        }

        .stat-number {
            font-size: 2.5em;
            font-weight: bold;
            color: var(--bg-primary);
            margin-bottom: var(--spacing-xs);
            text-shadow: none;
        }

        .stat-label {
            font-size: 14px;
            color: var(--text-muted);
            font-weight: 600;
            text-shadow: none;
        }

        /* =============================================================================
           MESSAGES ET ALERTS
           ============================================================================= */

        .alert {
            padding: var(--spacing-md);
            border-radius: var(--radius-small);
            margin-bottom: var(--spacing-lg);
            font-weight: 600;
            border-width: 2px;
            border-style: solid;
        }

        .alert-success {
            background: #ffffff;
            border-color: var(--bg-success);
            color: var(--bg-success);
        }

        .alert-error {
            background: #ffffff;
            border-color: var(--bg-danger);
            color: var(--bg-danger);
        }

        .quota-warning {
            background: #fff3cd;
            border: 2px solid var(--bg-warning);
            color: #856404;
            padding: var(--spacing-sm);
            border-radius: var(--radius-small);
            margin: var(--spacing-sm) 0;
            font-size: 14px;
            font-weight: 600;
        }

        .quota-ok {
            background: #d4edda;
            border: 2px solid var(--bg-success);
            color: #155724;
            padding: var(--spacing-sm);
            border-radius: var(--radius-small);
            margin: var(--spacing-sm) 0;
            font-size: 14px;
            font-weight: 600;
        }

        /* =============================================================================
           NETWORKS GRID - COMPTES SOCIAUX
           ============================================================================= */

        .networks-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: var(--spacing-xl);
            margin: var(--spacing-xl) 0;
        }

        .network-card {
            background: #ffffff;
            border: 2px solid var(--glass-border);
            border-radius: var(--radius-large);
            padding: var(--spacing-xl);
            transition: var(--transition-normal);
        }

        .network-card:hover {
            border-color: var(--bg-primary);
        }

        .network-header {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
            margin-bottom: var(--spacing-lg);
        }

        .network-icon {
            font-size: 2.5em;
            width: 60px;
            text-align: center;
        }

        .network-title {
            flex: 1;
        }

        .network-title h3 {
            font-size: 20px;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: var(--spacing-xs);
            text-shadow: none;
        }

        .network-title p {
            color: var(--text-muted);
            font-size: 14px;
            text-shadow: none;
        }

        .connection-status {
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--radius-xl);
            font-size: 12px;
            font-weight: bold;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            text-shadow: none;
        }

        .status-connected {
            background: var(--bg-success);
            color: var(--text-white);
        }

        .status-disconnected {
            background: var(--bg-danger);
            color: var(--text-white);
        }

        /* =============================================================================
           COMPTES EXISTANTS
           ============================================================================= */

        .connection-details {
            background: var(--glass-bg);
            border: 2px solid var(--border-light);
            border-radius: var(--radius-medium);
            padding: var(--spacing-lg);
            margin: var(--spacing-md) 0;
        }

        .connection-details h4 {
            color: var(--bg-primary);
            font-size: 16px;
            font-weight: 600;
            margin-bottom: var(--spacing-sm);
            text-shadow: none;
        }

        .detail-item {
            display: flex;
            justify-content: space-between;
            margin-bottom: var(--spacing-xs);
            font-size: 14px;
            color: var(--text-muted);
        }

        .detail-item span:last-child {
            color: var(--text-primary);
            font-weight: 600;
        }

        /* =============================================================================
           FORMULAIRES
           ============================================================================= */

        .info-box {
            background: var(--glass-bg);
            border: 2px solid var(--bg-primary);
            border-radius: var(--radius-medium);
            padding: var(--spacing-lg);
            margin: var(--spacing-lg) 0;
        }

        .info-box h4 {
            color: var(--bg-primary);
            font-size: 16px;
            font-weight: 600;
            text-shadow: none;
        }

        .token-form {
            margin-top: var(--spacing-lg);
        }

        .form-group {
            margin-bottom: var(--spacing-md);
        }

        .form-label {
            display: block;
            font-size: 14px;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: var(--spacing-xs);
            text-shadow: none;
        }

        .form-input {
            width: 100%;
            background: #ffffff;
            border: 2px solid var(--glass-border);
            border-radius: var(--radius-small);
            padding: var(--spacing-sm) var(--spacing-md);
            color: var(--text-primary);
            font-size: 14px;
            transition: var(--transition-normal);
        }

        .form-input:focus {
            outline: none;
            border-color: var(--bg-primary);
            box-shadow: none;
            background: #ffffff;
        }

        .form-input::placeholder {
            color: var(--text-muted);
        }

        /* =============================================================================
           BUTTONS ULTRA CONTRASTÉS
           ============================================================================= */

        .btn {
            padding: var(--spacing-sm) var(--spacing-lg);
            border: 2px solid transparent;
            border-radius: var(--radius-small);
            cursor: pointer;
            font-weight: 600;
            font-size: 14px;
            transition: var(--transition-normal);
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-xs);
            text-decoration: none;
            text-shadow: none;
            margin-right: var(--spacing-xs);
            margin-bottom: var(--spacing-xs);
        }

        .btn:hover {
            transform: none;
            box-shadow: none;
            opacity: 0.9;
        }

        .btn-primary {
            background: var(--bg-primary);
            color: var(--text-white);
        }

        .btn-success {
            background: var(--bg-success);
            color: var(--text-white);
        }

        .btn-warning {
            background: var(--bg-warning);
            color: var(--text-primary);
        }

        .btn-danger {
            background: var(--bg-danger);
            color: var(--text-white);
        }

        .btn-test {
            background: var(--bg-secondary);
            color: var(--text-white);
        }

        .btn-disconnect {
            background: var(--bg-danger);
            color: var(--text-white);
        }

        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: none !important;
        }

        /* =============================================================================
           ACTIONS GLOBALES
           ============================================================================= */

        .global-actions {
            background: #ffffff;
            border: 2px solid var(--glass-border);
            border-radius: var(--radius-large);
            padding: var(--spacing-xl);
            text-align: center;
            margin-top: var(--spacing-xl);
        }

        .global-actions h3 {
            color: var(--bg-primary);
            font-size: 20px;
            margin-bottom: var(--spacing-lg);
            text-shadow: none;
        }

        .global-actions p {
            color: var(--text-muted);
            margin-bottom: var(--spacing-lg);
            text-shadow: none;
        }

        /* =============================================================================
           RESPONSIVE
           ============================================================================= */

        @media (max-width: 768px) {
            .smg-container {
                padding: var(--spacing-md);
            }
            
            .smg-header-top {
                flex-direction: column;
                gap: var(--spacing-md);
                align-items: flex-start;
            }
            
            .smg-user-info {
                align-self: stretch;
                justify-content: center;
            }
            
            .user-stats {
                grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            }
            
            .networks-grid {
                grid-template-columns: 1fr;
            }
            
            .network-header {
                flex-direction: column;
                text-align: center;
                gap: var(--spacing-sm);
            }
        }

        /* Focus amélioré pour accessibilité */
        *:focus {
            outline: 2px solid var(--bg-primary);
            outline-offset: 2px;
        }
    </style>
</head>
<body>
    <div class="smg-container">
        <!-- ⭐ HEADER UNIFIÉ SMG STUDIO -->
        <header class="smg-header">
            <div class="smg-header-top">
                <a href="dashboard.php" class="smg-back-btn">
                    ← Dashboard Principal
                </a>
                
                <div class="smg-user-info">
                    <div class="smg-user-avatar"><?= strtoupper(substr($user['name'] ?? 'U', 0, 1)) ?></div>
                    <div class="smg-user-details">
                        <div class="smg-user-name"><?= htmlspecialchars($user['name'] ?? 'Utilisateur') ?></div>
                        <div class="smg-user-role"><?= ucfirst($user['role'] ?? 'user') ?> • ID: <?= $user['id'] ?? 'unknown' ?></div>
                    </div>
                </div>
            </div>
            
            <h1 class="smg-header-title">🔗 Mes Comptes Réseaux Sociaux</h1>
            <p class="smg-header-subtitle">Gérez vos connexions aux réseaux sociaux en toute sécurité</p>
        </header>

        <!-- STATISTIQUES UTILISATEUR -->
        <div class="user-stats">
            <div class="stat-card">
                <div class="stat-number"><?php echo $user_stats['total_accounts']; ?></div>
                <div class="stat-label">Comptes connectés</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo $user_stats['active_accounts']; ?></div>
                <div class="stat-label">Comptes actifs</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo count($networks_config); ?></div>
                <div class="stat-label">Réseaux disponibles</div>
            </div>
        </div>

        <!-- MESSAGES SYSTÈME -->
        <?php if ($message): ?>
            <div class="alert alert-<?php echo $message_type; ?>">
                <strong><?php echo $message_type === 'success' ? '✅' : '❌'; ?></strong> 
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- RÉSEAUX SOCIAUX -->
        <div class="networks-grid">
            <?php foreach ($networks_config as $platform => $config): ?>
                <?php 
                $current_count = $account_counts[$platform] ?? 0;
                $max_allowed = $config['max_accounts'];
                $can_add_more = $current_count < $max_allowed;
                ?>
                
                <div class="network-card <?php echo $platform; ?>">
                    <div class="network-header">
                        <div class="network-icon"><?php echo $config['icon']; ?></div>
                        <div class="network-title">
                            <h3><?php echo $config['name']; ?></h3>
                            <p><?php echo $config['description']; ?></p>
                        </div>
                        <span class="connection-status <?php echo $current_count > 0 ? 'status-connected' : 'status-disconnected'; ?>">
                            <?php echo $current_count; ?>/<?php echo $max_allowed; ?> comptes
                        </span>
                    </div>
                    
                    <!-- Affichage des quotas -->
                    <?php if ($current_count >= $max_allowed): ?>
                        <div class="quota-warning">
                            ⚠️ Quota atteint: <?php echo $max_allowed; ?> compte(s) maximum
                        </div>
                    <?php else: ?>
                        <div class="quota-ok">
                            ✅ Vous pouvez ajouter <?php echo ($max_allowed - $current_count); ?> compte(s) supplémentaire(s)
                        </div>
                    <?php endif; ?>
                    
                    <!-- Affichage des comptes existants -->
                    <?php if (isset($connected_accounts[$platform])): ?>
                        <?php foreach ($connected_accounts[$platform] as $account): ?>
                            <div class="connection-details">
                                <h4>📊 @<?php echo htmlspecialchars($account['username']); ?></h4>
                                <div class="detail-item">
                                    <span>Connecté le :</span>
                                    <span><?php echo date('d/m/Y H:i', strtotime($account['created_at'])); ?></span>
                                </div>
                                <div class="detail-item">
                                    <span>Statut :</span>
                                    <span><?php echo $account['is_active'] ? '🟢 Actif' : '🔴 Inactif'; ?></span>
                                </div>
                                
                                <div style="margin-top: var(--spacing-md);">
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="action" value="test">
                                        <input type="hidden" name="platform" value="<?php echo $platform; ?>">
                                        <input type="hidden" name="account_id" value="<?php echo $account['id']; ?>">
                                        <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                        <button type="submit" class="btn btn-test">🧪 Tester</button>
                                    </form>
                                    
                                    <form method="POST" style="display: inline;" 
                                          onsubmit="return confirm('Déconnecter ce compte ?')">
                                        <input type="hidden" name="action" value="disconnect">
                                        <input type="hidden" name="platform" value="<?php echo $platform; ?>">
                                        <input type="hidden" name="account_id" value="<?php echo $account['id']; ?>">
                                        <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                        <button type="submit" class="btn btn-disconnect">🔌 Déconnecter</button>
                                    </form>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    
                    <!-- Formulaire d'ajout si quota non atteint -->
                    <?php if ($can_add_more): ?>
                        <div class="info-box">
                            <h4>🔑 Ajouter un compte <?php echo $config['name']; ?> :</h4>
                        </div>
                        
                        <form method="POST" class="token-form">
                            <input type="hidden" name="action" value="connect">
                            <input type="hidden" name="platform" value="<?php echo $platform; ?>">
                            <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                            
                            <?php foreach ($config['fields'] as $field_name => $field_config): ?>
                                <div class="form-group">
                                    <label class="form-label">
                                        <?php echo $field_config['label']; ?>
                                        <?php if ($field_config['required']): ?> *<?php endif; ?>
                                    </label>
                                    <input 
                                        type="<?php echo $field_config['type']; ?>" 
                                        class="form-input" 
                                        name="credentials[<?php echo $field_name; ?>]"
                                        <?php if ($field_config['required']): ?>required<?php endif; ?>
                                        placeholder="<?php echo $field_config['type'] === 'password' ? '••••••••••' : 'Saisissez ' . strtolower($field_config['label']); ?>">
                                </div>
                            <?php endforeach; ?>
                            
                            <button type="submit" class="btn btn-primary">
                                ➕ Ajouter <?php echo $config['name']; ?>
                            </button>
                        </form>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- ACTIONS GLOBALES -->
        <div class="global-actions">
            <h3>🔧 Mes Actions</h3>
            <p>
                Utilisateur: <strong><?php echo htmlspecialchars($user['name'] ?? 'Utilisateur'); ?></strong> | 
                Total: <?php echo $user_stats['total_accounts']; ?> comptes connectés
            </p>
            
            <button class="btn btn-primary" onclick="window.location.reload()">
                🔄 Actualiser
            </button>
            
            <a href="dashboard.php" class="btn btn-primary">
                📊 Retour au Dashboard
            </a>
        </div>
    </div>

    <script>
        console.log('🔗 Social Accounts Manager - Multi-User Ready avec CSS Unifié');
        console.log('👤 Current User: <?php echo $user['name'] ?? 'Utilisateur'; ?> (ID: <?php echo $user['id'] ?? 'unknown'; ?>)');
        console.log('📊 User Stats:', <?php echo json_encode($user_stats); ?>);
        console.log('🔒 Security: CSRF protection enabled');
        console.log('🎨 Design: SMG Studio v2.0 CSS Ultra Contraste');
    </script>
</body>
</html>