<?php
/**
 * ================================================================
 * 🔗 SOCIAL-ACCOUNTS.PHP - GESTION COMPTES RÉSEAUX SOCIAUX
 * ================================================================
 * 
 * 🎯 FONCTIONNALITÉS :
 * - Interface complète gestion comptes sociaux
 * - Connexion/déconnexion tous réseaux
 * - Test en temps réel des connexions
 * - Sauvegarde sécurisée tokens
 * - Intégration base MySQL SMG
 * 
 * 💡 BY PAPY 77 - INTERFACE SOCIAL COMPLETE !
 * ================================================================
 */

require_once 'auth_functions.php';
requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

// Configuration des réseaux sociaux
$networks_config = [
    'facebook' => [
        'name' => 'Facebook',
        'icon' => '📘',
        'color' => '#1877f2',
        'description' => 'Pages et comptes business',
        'fields' => [
            'access_token' => ['label' => 'Page Access Token', 'type' => 'password', 'required' => true],
            'page_id' => ['label' => 'Page ID', 'type' => 'text', 'required' => true],
            'app_id' => ['label' => 'App ID', 'type' => 'text', 'required' => false]
        ]
    ],
    'instagram' => [
        'name' => 'Instagram',
        'icon' => '📸',
        'color' => '#E4405F',
        'description' => 'Compte business/créateur',
        'fields' => [
            'instagram_id' => ['label' => 'Instagram Business Account ID', 'type' => 'text', 'required' => true],
            'access_token' => ['label' => 'Facebook Page Access Token', 'type' => 'password', 'required' => true],
            'page_id' => ['label' => 'Facebook Page ID', 'type' => 'text', 'required' => true]
        ]
    ],
    'tiktok' => [
        'name' => 'TikTok',
        'icon' => '🎵',
        'color' => '#000000',
        'description' => 'TikTok for Business',
        'fields' => [
            'access_token' => ['label' => 'Access Token', 'type' => 'password', 'required' => true],
            'user_id' => ['label' => 'User ID', 'type' => 'text', 'required' => true],
            'client_key' => ['label' => 'Client Key', 'type' => 'text', 'required' => false]
        ]
    ],
    'linkedin' => [
        'name' => 'LinkedIn',
        'icon' => '💼',
        'color' => '#0A66C2',
        'description' => 'Pages et profils professionnels',
        'fields' => [
            'access_token' => ['label' => 'Access Token', 'type' => 'password', 'required' => true],
            'organization_id' => ['label' => 'Organization ID', 'type' => 'text', 'required' => true],
            'client_id' => ['label' => 'Client ID', 'type' => 'text', 'required' => false]
        ]
    ],
    'snapchat' => [
        'name' => 'Snapchat',
        'icon' => '👻',
        'color' => '#FFFC00',
        'description' => 'Snapchat Business',
        'fields' => [
            'access_token' => ['label' => 'Access Token', 'type' => 'password', 'required' => true],
            'ad_account_id' => ['label' => 'Ad Account ID', 'type' => 'text', 'required' => true],
            'client_id' => ['label' => 'Client ID', 'type' => 'text', 'required' => false]
        ]
    ]
];

// Chargement des comptes connectés
$connected_accounts = [];
try {
    $stmt = $pdo->prepare("
        SELECT platform, username, platform_user_id, is_active, 
               created_at, updated_at, account_data 
        FROM social_accounts 
        WHERE user_id = ? 
        ORDER BY platform
    ");
    $stmt->execute([$user['id']]);
    $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($accounts as $account) {
        $connected_accounts[$account['platform']] = $account;
    }
} catch (PDOException $e) {
    error_log("Erreur chargement comptes sociaux: " . $e->getMessage());
}

// Traitement des actions POST
$message = '';
$message_type = '';

if ($_POST) {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'connect':
            $result = connectSocialAccount($_POST, $user['id'], $pdo);
            $message = $result['message'];
            $message_type = $result['success'] ? 'success' : 'error';
            break;
            
        case 'disconnect':
            $result = disconnectSocialAccount($_POST['platform'], $user['id'], $pdo);
            $message = $result['message'];
            $message_type = $result['success'] ? 'success' : 'error';
            break;
            
        case 'test':
            $result = testSocialConnection($_POST['platform'], $user['id'], $pdo);
            $message = $result['message'];
            $message_type = $result['success'] ? 'success' : 'error';
            break;
    }
    
    // Rechargement des comptes après action
    if ($message_type === 'success') {
        header("Location: " . $_SERVER['PHP_SELF'] . "?success=1");
        exit;
    }
}

/**
 * Connexion d'un compte social
 */
function connectSocialAccount($data, $user_id, $pdo) {
    $platform = $data['platform'] ?? '';
    $credentials = $data['credentials'] ?? [];
    
    if (!$platform || !$credentials['access_token']) {
        return ['success' => false, 'message' => 'Données manquantes'];
    }
    
    try {
        // Validation du token
        $validation = validateToken($platform, $credentials);
        
        if (!$validation['success']) {
            return ['success' => false, 'message' => 'Token invalide: ' . $validation['error']];
        }
        
        // Chiffrement des credentials
        $encrypted_token = encryptToken($credentials['access_token']);
        $account_data = json_encode($credentials);
        
        // Sauvegarde en base
        $stmt = $pdo->prepare("
            INSERT INTO social_accounts 
            (user_id, platform, platform_user_id, username, access_token, account_data, is_active)
            VALUES (?, ?, ?, ?, ?, ?, 1)
            ON DUPLICATE KEY UPDATE
            platform_user_id = VALUES(platform_user_id),
            username = VALUES(username),
            access_token = VALUES(access_token),
            account_data = VALUES(account_data),
            is_active = 1,
            updated_at = CURRENT_TIMESTAMP
        ");
        
        $stmt->execute([
            $user_id,
            $platform,
            $validation['user_id'] ?? 'unknown',
            $validation['username'] ?? 'user',
            $encrypted_token,
            $account_data
        ]);
        
        // Log de l'activité
        logActivity($user_id, 'connect_social', 'social_account', null, 
                   "Connexion {$platform}", $pdo);
        
        return ['success' => true, 'message' => ucfirst($platform) . ' connecté avec succès !'];
        
    } catch (Exception $e) {
        error_log("Erreur connexion $platform: " . $e->getMessage());
        return ['success' => false, 'message' => 'Erreur de connexion: ' . $e->getMessage()];
    }
}

/**
 * Déconnexion d'un compte social
 */
function disconnectSocialAccount($platform, $user_id, $pdo) {
    try {
        $stmt = $pdo->prepare("DELETE FROM social_accounts WHERE user_id = ? AND platform = ?");
        $stmt->execute([$user_id, $platform]);
        
        logActivity($user_id, 'disconnect_social', 'social_account', null, 
                   "Déconnexion {$platform}", $pdo);
        
        return ['success' => true, 'message' => ucfirst($platform) . ' déconnecté'];
        
    } catch (PDOException $e) {
        error_log("Erreur déconnexion $platform: " . $e->getMessage());
        return ['success' => false, 'message' => 'Erreur de déconnexion'];
    }
}

/**
 * Test de connexion
 */
function testSocialConnection($platform, $user_id, $pdo) {
    try {
        $stmt = $pdo->prepare("SELECT access_token FROM social_accounts WHERE user_id = ? AND platform = ?");
        $stmt->execute([$user_id, $platform]);
        $account = $stmt->fetch();
        
        if (!$account) {
            return ['success' => false, 'message' => 'Compte non connecté'];
        }
        
        $decrypted_token = decryptToken($account['access_token']);
        $test_result = performAPITest($platform, $decrypted_token);
        
        return $test_result;
        
    } catch (Exception $e) {
        error_log("Erreur test $platform: " . $e->getMessage());
        return ['success' => false, 'message' => 'Erreur de test'];
    }
}

/**
 * Validation des tokens par API
 */
function validateToken($platform, $credentials) {
    switch ($platform) {
        case 'facebook':
            return validateFacebookToken($credentials['access_token'], $credentials['page_id']);
        case 'instagram':
            return validateInstagramToken($credentials['access_token'], $credentials['instagram_id']);
        case 'tiktok':
            return validateTikTokToken($credentials['access_token'], $credentials['user_id']);
        case 'linkedin':
            return validateLinkedInToken($credentials['access_token']);
        case 'snapchat':
            return validateSnapchatToken($credentials['access_token']);
        default:
            return ['success' => false, 'error' => 'Plateforme non supportée'];
    }
}

/**
 * Validation Facebook
 */
function validateFacebookToken($token, $page_id) {
    $url = "https://graph.facebook.com/{$page_id}?fields=name,access_token&access_token={$token}";
    
    $context = stream_context_create([
        'http' => [
            'timeout' => 10,
            'ignore_errors' => true
        ]
    ]);
    
    $response = @file_get_contents($url, false, $context);
    
    if ($response === false) {
        return ['success' => false, 'error' => 'Impossible de contacter Facebook'];
    }
    
    $data = json_decode($response, true);
    
    if (isset($data['error'])) {
        return ['success' => false, 'error' => $data['error']['message']];
    }
    
    if (isset($data['name'])) {
        return [
            'success' => true,
            'user_id' => $page_id,
            'username' => $data['name']
        ];
    }
    
    return ['success' => false, 'error' => 'Réponse invalide'];
}

/**
 * Validation Instagram
 */
function validateInstagramToken($token, $instagram_id) {
    $url = "https://graph.instagram.com/{$instagram_id}?fields=id,username&access_token={$token}";
    
    $response = @file_get_contents($url);
    
    if ($response === false) {
        return ['success' => false, 'error' => 'Impossible de contacter Instagram'];
    }
    
    $data = json_decode($response, true);
    
    if (isset($data['error'])) {
        return ['success' => false, 'error' => $data['error']['message']];
    }
    
    if (isset($data['username'])) {
        return [
            'success' => true,
            'user_id' => $data['id'],
            'username' => $data['username']
        ];
    }
    
    return ['success' => false, 'error' => 'Réponse invalide'];
}

/**
 * Validation TikTok (simulation)
 */
function validateTikTokToken($token, $user_id) {
    // Simulation - en production, utilisez l'API TikTok
    if (strlen($token) > 20 && strlen($user_id) > 10) {
        return [
            'success' => true,
            'user_id' => $user_id,
            'username' => 'tiktok_user'
        ];
    }
    
    return ['success' => false, 'error' => 'Format de token incorrect'];
}

/**
 * Validation LinkedIn (simulation)
 */
function validateLinkedInToken($token) {
    // Simulation - en production, utilisez l'API LinkedIn
    if (strlen($token) > 20) {
        return [
            'success' => true,
            'user_id' => 'linkedin_id',
            'username' => 'linkedin_user'
        ];
    }
    
    return ['success' => false, 'error' => 'Format de token incorrect'];
}

/**
 * Validation Snapchat (simulation)
 */
function validateSnapchatToken($token) {
    // Simulation - en production, utilisez l'API Snapchat
    if (strlen($token) > 20) {
        return [
            'success' => true,
            'user_id' => 'snapchat_id',
            'username' => 'snapchat_user'
        ];
    }
    
    return ['success' => false, 'error' => 'Format de token incorrect'];
}

/**
 * Test API en direct
 */
function performAPITest($platform, $token) {
    // Simulation de test - en production, faire un vrai call API
    if ($token) {
        return ['success' => true, 'message' => "Test {$platform} réussi !"];
    }
    
    return ['success' => false, 'message' => "Test {$platform} échoué"];
}

/**
 * Chiffrement simple des tokens
 */
function encryptToken($token) {
    $key = 'SMG_SECRET_KEY_PAPY77'; // En production, utilisez une vraie clé
    return base64_encode($token . '|' . $key);
}

/**
 * Déchiffrement des tokens
 */
function decryptToken($encrypted_token) {
    $decoded = base64_decode($encrypted_token);
    $parts = explode('|', $decoded);
    return $parts[0] ?? '';
}

/**
 * Log des activités
 */
function logActivity($user_id, $action, $entity_type, $entity_id, $description, $pdo) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs (user_id, action, entity_type, entity_id, description, ip_address) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$user_id, $action, $entity_type, $entity_id, $description, $_SERVER['REMOTE_ADDR']]);
    } catch (PDOException $e) {
        error_log("Erreur log activité: " . $e->getMessage());
    }
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🔗 Gestion des Comptes Réseaux Sociaux - SMG</title>
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 16px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 30px;
            text-align: center;
            position: relative;
        }
        
        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            font-weight: 300;
        }
        
        .header p {
            opacity: 0.9;
            font-size: 1.1em;
        }
        
        .header .back-btn {
            position: absolute;
            left: 30px;
            top: 50%;
            transform: translateY(-50%);
            background: rgba(255, 255, 255, 0.2);
            color: white;
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .header .back-btn:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: translateY(-50%) translateX(-2px);
        }
        
        .content {
            padding: 40px;
        }
        
        .networks-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
            gap: 30px;
            margin-bottom: 40px;
        }
        
        .network-card {
            background: white;
            border: 2px solid #f0f0f0;
            border-radius: 12px;
            padding: 25px;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .network-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 30px rgba(0,0,0,0.1);
            border-color: var(--network-color);
        }
        
        .network-card.facebook { --network-color: #1877f2; }
        .network-card.instagram { --network-color: #E4405F; }
        .network-card.tiktok { --network-color: #000000; }
        .network-card.linkedin { --network-color: #0A66C2; }
        .network-card.snapchat { --network-color: #FFFC00; }
        
        .network-header {
            display: flex;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .network-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 15px;
            font-size: 24px;
            color: white;
        }
        
        .facebook .network-icon { background: #1877f2; }
        .instagram .network-icon { background: linear-gradient(45deg, #f09433 0%,#e6683c 25%,#dc2743 50%,#cc2366 75%,#bc1888 100%); }
        .tiktok .network-icon { background: #000000; }
        .linkedin .network-icon { background: #0A66C2; }
        .snapchat .network-icon { background: #FFFC00; color: #000; }
        
        .network-title {
            flex: 1;
        }
        
        .network-title h3 {
            font-size: 1.4em;
            color: #333;
            margin-bottom: 5px;
        }
        
        .network-title p {
            color: #666;
            font-size: 0.9em;
        }
        
        .connection-status {
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 0.8em;
            font-weight: bold;
            text-transform: uppercase;
        }
        
        .status-connected {
            background: #d4edda;
            color: #155724;
        }
        
        .status-disconnected {
            background: #f8d7da;
            color: #721c24;
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        .form-label {
            display: block;
            margin-bottom: 5px;
            font-weight: 600;
            color: #333;
        }
        
        .form-input {
            width: 100%;
            padding: 12px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s;
        }
        
        .form-input:focus {
            outline: none;
            border-color: var(--network-color);
        }
        
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 12px;
            letter-spacing: 0.5px;
            transition: all 0.3s;
            margin-right: 10px;
            margin-bottom: 10px;
        }
        
        .btn-primary {
            background: var(--network-color);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        
        .btn-test {
            background: #28a745;
            color: white;
        }
        
        .btn-disconnect {
            background: #dc3545;
            color: white;
        }
        
        .info-box {
            background: #f8f9fa;
            border-left: 4px solid var(--network-color);
            padding: 15px;
            margin: 15px 0;
            border-radius: 0 8px 8px 0;
        }
        
        .connection-details {
            background: #e9ecef;
            padding: 15px;
            border-radius: 8px;
            margin-top: 15px;
        }
        
        .detail-item {
            display: flex;
            justify-content: space-between;
            margin: 5px 0;
            font-size: 0.9em;
        }
        
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin: 15px 0;
        }
        
        .alert-success {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
        }
        
        .alert-error {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
        }
        
        .global-actions {
            background: #f8f9fa;
            padding: 30px;
            border-top: 1px solid #dee2e6;
            text-align: center;
        }
        
        .hidden {
            display: none;
        }
        
        @media (max-width: 768px) {
            .header .back-btn {
                position: static;
                transform: none;
                margin-bottom: 20px;
                display: inline-block;
            }
            
            .networks-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <a href="dashboard.php" class="back-btn">← Dashboard</a>
            <h1>🔗 Gestion des Comptes Réseaux</h1>
            <p>Connectez vos comptes pour publier automatiquement sur tous vos réseaux sociaux</p>
        </div>
        
        <div class="content">
            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?>">
                    <strong><?php echo $message_type === 'success' ? '✅' : '❌'; ?></strong> 
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <?php if (isset($_GET['success'])): ?>
                <div class="alert alert-success">
                    <strong>✅</strong> Action effectuée avec succès !
                </div>
            <?php endif; ?>
            
            <div class="networks-grid">
                <?php foreach ($networks_config as $platform => $config): ?>
                    <?php $is_connected = isset($connected_accounts[$platform]); ?>
                    
                    <div class="network-card <?php echo $platform; ?>">
                        <div class="network-header">
                            <div class="network-icon"><?php echo $config['icon']; ?></div>
                            <div class="network-title">
                                <h3><?php echo $config['name']; ?></h3>
                                <p><?php echo $config['description']; ?></p>
                            </div>
                            <span class="connection-status <?php echo $is_connected ? 'status-connected' : 'status-disconnected'; ?>">
                                <?php echo $is_connected ? 'Connecté' : 'Déconnecté'; ?>
                            </span>
                        </div>
                        
                        <?php if ($is_connected): ?>
                            <!-- Compte connecté -->
                            <div class="connection-details">
                                <h4>📊 Détails de connexion :</h4>
                                <div class="detail-item">
                                    <span>Nom d'utilisateur :</span>
                                    <span>@<?php echo htmlspecialchars($connected_accounts[$platform]['username']); ?></span>
                                </div>
                                <div class="detail-item">
                                    <span>Connecté le :</span>
                                    <span><?php echo date('d/m/Y H:i', strtotime($connected_accounts[$platform]['created_at'])); ?></span>
                                </div>
                                <div class="detail-item">
                                    <span>Dernière mise à jour :</span>
                                    <span><?php echo date('d/m/Y H:i', strtotime($connected_accounts[$platform]['updated_at'])); ?></span>
                                </div>
                                <div class="detail-item">
                                    <span>Statut :</span>
                                    <span><?php echo $connected_accounts[$platform]['is_active'] ? '🟢 Actif' : '🔴 Inactif'; ?></span>
                                </div>
                            </div>
                            
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="test">
                                <input type="hidden" name="platform" value="<?php echo $platform; ?>">
                                <button type="submit" class="btn btn-test">🧪 Tester</button>
                            </form>
                            
                            <form method="POST" style="display: inline;" 
                                  onsubmit="return confirm('Êtes-vous sûr de vouloir déconnecter ce compte ?')">
                                <input type="hidden" name="action" value="disconnect">
                                <input type="hidden" name="platform" value="<?php echo $platform; ?>">
                                <button type="submit" class="btn btn-disconnect">🔌 Déconnecter</button>
                            </form>
                            
                        <?php else: ?>
                            <!-- Formulaire de connexion -->
                            <div class="info-box">
                                <h4>🔑 Informations requises :</h4>
                                <ul style="margin: 10px 0 0 20px; font-size: 0.9em;">
                                    <?php foreach ($config['fields'] as $field_name => $field_config): ?>
                                        <li><strong><?php echo $field_config['label']; ?></strong></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                            
                            <form method="POST" class="token-form">
                                <input type="hidden" name="action" value="connect">
                                <input type="hidden" name="platform" value="<?php echo $platform; ?>">
                                
                                <?php foreach ($config['fields'] as $field_name => $field_config): ?>
                                    <div class="form-group">
                                        <label class="form-label">
                                            <?php echo $field_config['label']; ?>
                                            <?php if ($field_config['required']): ?> *<?php endif; ?>
                                        </label>
                                        <input 
                                            type="<?php echo $field_config['type']; ?>" 
                                            class="form-input" 
                                            name="credentials[<?php echo $field_name; ?>]"
                                            <?php if ($field_config['required']): ?>required<?php endif; ?>
                                            placeholder="<?php echo $field_config['type'] === 'password' ? '••••••••••' : 'Saisissez ' . strtolower($field_config['label']); ?>">
                                    </div>
                                <?php endforeach; ?>
                                
                                <button type="submit" class="btn btn-primary">
                                    💾 Connecter <?php echo $config['name']; ?>
                                </button>
                            </form>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        
        <div class="global-actions">
            <h3>🔧 Actions Globales</h3>
            <p style="margin-bottom: 20px; color: #666;">
                <?php echo count($connected_accounts); ?> compte(s) connecté(s) sur <?php echo count($networks_config); ?> réseaux disponibles
            </p>
            
            <form method="POST" style="display: inline;">
                <input type="hidden" name="action" value="test_all">
                <button type="submit" class="btn btn-test">🧪 Tester Toutes les Connexions</button>
            </form>
            
            <button class="btn btn-primary" onclick="window.location.reload()">
                🔄 Actualiser la Page
            </button>
            
            <a href="dashboard.php" class="btn btn-primary">
                📊 Retour au Dashboard
            </a>
        </div>
    </div>

    <script>
        // Auto-refresh toutes les 30 secondes si des tests sont en cours
        let autoRefresh = false;
        
        // Gestion des formulaires
        document.querySelectorAll('form').forEach(form => {
            form.addEventListener('submit', function(e) {
                const button = this.querySelector('button[type="submit"]');
                if (button) {
                    button.disabled = true;
                    button.innerHTML = button.innerHTML.replace(/🧪|💾|🔌/, '⏳');
                    
                    // Re-enable après 10 secondes au cas où
                    setTimeout(() => {
                        button.disabled = false;
                    }, 10000);
                }
            });
        });
        
        // Validation en temps réel
        document.querySelectorAll('input[required]').forEach(input => {
            input.addEventListener('input', function() {
                const form = this.closest('form');
                const submitBtn = form.querySelector('button[type="submit"]');
                const requiredInputs = form.querySelectorAll('input[required]');
                
                let allFilled = true;
                requiredInputs.forEach(field => {
                    if (!field.value.trim()) {
                        allFilled = false;
                    }
                });
                
                if (submitBtn) {
                    submitBtn.disabled = !allFilled;
                    submitBtn.style.opacity = allFilled ? '1' : '0.5';
                }
            });
        });
        
        // Initialiser la validation
        document.querySelectorAll('form').forEach(form => {
            const firstInput = form.querySelector('input[required]');
            if (firstInput) {
                firstInput.dispatchEvent(new Event('input'));
            }
        });
        
        console.log('🔗 Social Accounts Manager Ready - PHP Version');
        console.log('👤 User: <?php echo $user['name']; ?>');
        console.log('📊 Connected accounts: <?php echo count($connected_accounts); ?>');
    </script>
</body>
</html>