-- ================================================================
-- 🔧 CORRECTIONS SQL SMG STUDIO v2.0 - PAPY 77 EDITION
-- ================================================================
-- 📋 Suppression références client_id + Structure user-only
-- 📥 Tables download + programmation complètes
-- 🌐 Plateformes sociales + logs transmission
-- ================================================================

-- 1️⃣ SUPPRESSION DES CONTRAINTES CLIENT_ID
-- ================================================================

-- Supprimer les contraintes foreign key liées à client_id
ALTER TABLE posts DROP FOREIGN KEY IF EXISTS fk_posts_client;
ALTER TABLE user_workspaces DROP FOREIGN KEY IF EXISTS fk_workspaces_client;

-- Supprimer les colonnes client_id
ALTER TABLE posts DROP COLUMN IF EXISTS client_id;
ALTER TABLE user_workspaces DROP COLUMN IF EXISTS client_id;

-- 2️⃣ TABLE POSTS - STRUCTURE USER-ONLY
-- ================================================================

CREATE TABLE IF NOT EXISTS `posts` (
  `id` int NOT NULL AUTO_INCREMENT,
  `user_id` int NOT NULL,
  `titre` varchar(255) COLLATE utf8mb4_unicode_ci NOT NULL,
  `contenu` text COLLATE utf8mb4_unicode_ci,
  `hashtags` text COLLATE utf8mb4_unicode_ci,
  `statut` enum('draft','scheduled','published','archived') COLLATE utf8mb4_unicode_ci DEFAULT 'draft',
  `type` enum('text','image','video','carousel','story') COLLATE utf8mb4_unicode_ci DEFAULT 'text',
  `priority` int DEFAULT '1',
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  `published_at` timestamp NULL DEFAULT NULL,
  PRIMARY KEY (`id`),
  KEY `idx_posts_user_id` (`user_id`),
  KEY `idx_posts_statut` (`statut`),
  KEY `idx_posts_created_at` (`created_at`),
  CONSTRAINT `fk_posts_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 3️⃣ RÉSEAUX SOCIAUX ET CONFIGURATIONS
-- ================================================================

CREATE TABLE IF NOT EXISTS `social_platforms` (
  `id` int NOT NULL AUTO_INCREMENT,
  `name` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
  `display_name` varchar(100) COLLATE utf8mb4_unicode_ci NOT NULL,
  `icon` varchar(10) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
  `is_active` tinyint(1) DEFAULT '1',
  `api_endpoint` varchar(255) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
  `max_chars` int DEFAULT '0',
  `supports_images` tinyint(1) DEFAULT '1',
  `supports_videos` tinyint(1) DEFAULT '1',
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `name` (`name`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Données plateformes sociales
INSERT INTO `social_platforms` (`name`, `display_name`, `icon`, `max_chars`, `supports_images`, `supports_videos`) VALUES
('instagram', 'Instagram', '📸', 2200, 1, 1),
('facebook', 'Facebook', '📘', 63206, 1, 1),
('twitter', 'Twitter/X', '🐦', 280, 1, 1),
('linkedin', 'LinkedIn', '💼', 3000, 1, 1),
('tiktok', 'TikTok', '🎵', 2200, 1, 1),
('snapchat', 'Snapchat', '👻', 250, 1, 1),
('youtube', 'YouTube', '📺', 5000, 1, 1)
ON DUPLICATE KEY UPDATE 
display_name = VALUES(display_name),
icon = VALUES(icon),
max_chars = VALUES(max_chars);

-- 4️⃣ ASSOCIATION POSTS-RÉSEAUX
-- ================================================================

CREATE TABLE IF NOT EXISTS `post_networks` (
  `id` int NOT NULL AUTO_INCREMENT,
  `post_id` int NOT NULL,
  `network` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
  `is_active` tinyint(1) DEFAULT '1',
  `custom_content` text COLLATE utf8mb4_unicode_ci,
  `custom_hashtags` text COLLATE utf8mb4_unicode_ci,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_post_network` (`post_id`,`network`),
  KEY `idx_post_networks_network` (`network`),
  CONSTRAINT `fk_post_networks_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_post_networks_platform` FOREIGN KEY (`network`) REFERENCES `social_platforms` (`name`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 5️⃣ MÉDIAS POSTS
-- ================================================================

CREATE TABLE IF NOT EXISTS `post_medias` (
  `id` int NOT NULL AUTO_INCREMENT,
  `post_id` int NOT NULL,
  `media_path` varchar(500) COLLATE utf8mb4_unicode_ci NOT NULL,
  `media_type` enum('image','video','audio','document') COLLATE utf8mb4_unicode_ci NOT NULL,
  `media_size` bigint DEFAULT '0',
  `mime_type` varchar(100) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
  `display_order` int DEFAULT '1',
  `alt_text` varchar(255) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
  `thumbnail_path` varchar(500) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_post_medias_post_id` (`post_id`),
  KEY `idx_post_medias_type` (`media_type`),
  CONSTRAINT `fk_post_medias_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 6️⃣ PROGRAMMATION DIFFUSION
-- ================================================================

CREATE TABLE IF NOT EXISTS `scheduled_posts` (
  `id` int NOT NULL AUTO_INCREMENT,
  `user_id` int NOT NULL,
  `post_id` int NOT NULL,
  `network` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
  `scheduled_datetime` timestamp NOT NULL,
  `status` enum('pending','processing','sent','failed','cancelled') COLLATE utf8mb4_unicode_ci DEFAULT 'pending',
  `priority` int DEFAULT '1',
  `retry_count` int DEFAULT '0',
  `max_retries` int DEFAULT '3',
  `error_message` text COLLATE utf8mb4_unicode_ci,
  `external_post_id` varchar(255) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
  `sent_at` timestamp NULL DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_post_network_schedule` (`post_id`,`network`,`scheduled_datetime`),
  KEY `idx_scheduled_posts_user` (`user_id`),
  KEY `idx_scheduled_posts_status` (`status`),
  KEY `idx_scheduled_posts_datetime` (`scheduled_datetime`),
  KEY `idx_scheduled_posts_network` (`network`),
  CONSTRAINT `fk_scheduled_posts_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_scheduled_posts_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_scheduled_posts_network` FOREIGN KEY (`network`) REFERENCES `social_platforms` (`name`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 7️⃣ LOGS TRANSMISSION
-- ================================================================

CREATE TABLE IF NOT EXISTS `transmission_logs` (
  `id` int NOT NULL AUTO_INCREMENT,
  `scheduled_post_id` int NOT NULL,
  `network` varchar(50) COLLATE utf8mb4_unicode_ci NOT NULL,
  `status` enum('success','failed','retry','cancelled') COLLATE utf8mb4_unicode_ci NOT NULL,
  `attempt_number` int DEFAULT '1',
  `response_data` json DEFAULT NULL,
  `error_message` text COLLATE utf8mb4_unicode_ci,
  `execution_time_ms` int DEFAULT '0',
  `external_post_id` varchar(255) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
  `transmitted_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_transmission_logs_scheduled` (`scheduled_post_id`),
  KEY `idx_transmission_logs_network` (`network`),
  KEY `idx_transmission_logs_status` (`status`),
  KEY `idx_transmission_logs_date` (`transmitted_at`),
  CONSTRAINT `fk_transmission_logs_scheduled` FOREIGN KEY (`scheduled_post_id`) REFERENCES `scheduled_posts` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 8️⃣ SYSTÈME DOWNLOADS USER
-- ================================================================

CREATE TABLE IF NOT EXISTS `user_downloads` (
  `id` int NOT NULL AUTO_INCREMENT,
  `user_id` int NOT NULL,
  `post_id` int DEFAULT NULL,
  `download_type` enum('post','media','final','zip') COLLATE utf8mb4_unicode_ci DEFAULT 'zip',
  `file_path` varchar(500) COLLATE utf8mb4_unicode_ci NOT NULL,
  `file_size` bigint DEFAULT '0',
  `download_count` int DEFAULT '0',
  `expires_at` timestamp NULL DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `last_downloaded` timestamp NULL DEFAULT NULL,
  PRIMARY KEY (`id`),
  KEY `idx_user_downloads_user` (`user_id`),
  KEY `idx_user_downloads_post` (`post_id`),
  KEY `idx_user_downloads_expires` (`expires_at`),
  CONSTRAINT `fk_user_downloads_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_user_downloads_post` FOREIGN KEY (`post_id`) REFERENCES `posts` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 9️⃣ WORKSPACE USER (sans client_id)
-- ================================================================

CREATE TABLE IF NOT EXISTS `user_workspaces` (
  `id` int NOT NULL AUTO_INCREMENT,
  `user_id` int NOT NULL,
  `workspace_name` varchar(100) COLLATE utf8mb4_unicode_ci NOT NULL,
  `workspace_path` varchar(500) COLLATE utf8mb4_unicode_ci NOT NULL,
  `storage_used` bigint DEFAULT '0',
  `storage_limit` bigint DEFAULT '1073741824',
  `is_active` tinyint(1) DEFAULT '1',
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_user_workspace` (`user_id`,`workspace_name`),
  KEY `idx_user_workspaces_user` (`user_id`),
  CONSTRAINT `fk_user_workspaces_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 🔟 NETTOYAGE ET OPTIMISATION
-- ================================================================

-- Supprimer les tables obsolètes clients si elles existent
DROP TABLE IF EXISTS `client_posts`;
DROP TABLE IF EXISTS `clients`;

-- Index de performance
CREATE INDEX IF NOT EXISTS `idx_posts_user_status` ON `posts` (`user_id`, `statut`);
CREATE INDEX IF NOT EXISTS `idx_scheduled_posts_processing` ON `scheduled_posts` (`status`, `scheduled_datetime`);
CREATE INDEX IF NOT EXISTS `idx_downloads_cleanup` ON `user_downloads` (`expires_at`, `created_at`);

-- ================================================================
-- ✅ DONNÉES DE TEST POUR VALIDATION
-- ================================================================

-- Post de test pour user_id = 4 (admin actif)
INSERT INTO `posts` (`user_id`, `titre`, `contenu`, `hashtags`, `statut`, `type`) VALUES
(4, '🎯 Test Post SMG Studio', 'Premier post de test pour valider le système SMG Studio v2.0 - Phase 1 Dashboard User', '#SMGStudio #Test #SocialMedia #Audiovizu', 'draft', 'text')
ON DUPLICATE KEY UPDATE titre = VALUES(titre);

-- Association avec réseaux sociaux
INSERT INTO `post_networks` (`post_id`, `network`, `is_active`) VALUES
((SELECT id FROM posts WHERE user_id = 4 AND titre LIKE '%Test Post SMG%' LIMIT 1), 'instagram', 1),
((SELECT id FROM posts WHERE user_id = 4 AND titre LIKE '%Test Post SMG%' LIMIT 1), 'facebook', 1),
((SELECT id FROM posts WHERE user_id = 4 AND titre LIKE '%Test Post SMG%' LIMIT 1), 'linkedin', 1)
ON DUPLICATE KEY UPDATE is_active = VALUES(is_active);

-- ================================================================
-- 🎯 PROCÉDURES DE MAINTENANCE
-- ================================================================

DELIMITER //

-- Nettoyage automatique des fichiers temporaires expirés
CREATE PROCEDURE IF NOT EXISTS CleanupExpiredDownloads()
BEGIN
    DECLARE done INT DEFAULT FALSE;
    DECLARE file_path VARCHAR(500);
    DECLARE download_cursor CURSOR FOR 
        SELECT file_path FROM user_downloads 
        WHERE expires_at < NOW() AND expires_at IS NOT NULL;
    DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = TRUE;
    
    OPEN download_cursor;
    read_loop: LOOP
        FETCH download_cursor INTO file_path;
        IF done THEN
            LEAVE read_loop;
        END IF;
        
        -- Log pour debugging
        INSERT INTO system_logs (log_type, message) VALUES 
        ('cleanup', CONCAT('Fichier expiré supprimé: ', file_path));
        
    END LOOP;
    CLOSE download_cursor;
    
    -- Supprimer les enregistrements expirés
    DELETE FROM user_downloads WHERE expires_at < NOW() AND expires_at IS NOT NULL;
END //

-- Statistiques utilisateur
CREATE PROCEDURE IF NOT EXISTS GetUserStats(IN p_user_id INT)
BEGIN
    SELECT 
        COUNT(DISTINCT p.id) as total_posts,
        COUNT(DISTINCT CASE WHEN p.statut = 'draft' THEN p.id END) as draft_posts,
        COUNT(DISTINCT CASE WHEN p.statut = 'scheduled' THEN p.id END) as scheduled_posts,
        COUNT(DISTINCT CASE WHEN p.statut = 'published' THEN p.id END) as published_posts,
        COUNT(DISTINCT pn.network) as networks_count,
        COUNT(DISTINCT sp.id) as pending_schedules,
        COUNT(DISTINCT pm.id) as total_medias,
        COALESCE(SUM(pm.media_size), 0) as total_storage_used
    FROM posts p
    LEFT JOIN post_networks pn ON p.id = pn.post_id AND pn.is_active = 1
    LEFT JOIN scheduled_posts sp ON p.id = sp.post_id AND sp.status = 'pending'
    LEFT JOIN post_medias pm ON p.id = pm.post_id
    WHERE p.user_id = p_user_id;
END //

DELIMITER ;

-- ================================================================
-- 📊 VUES POUR DASHBOARD
-- ================================================================

-- Vue posts avec statistiques
CREATE OR REPLACE VIEW view_posts_stats AS
SELECT 
    p.*,
    COUNT(DISTINCT pn.network) as networks_count,
    COUNT(DISTINCT pm.id) as media_count,
    COUNT(DISTINCT sp.id) as scheduled_count,
    GROUP_CONCAT(DISTINCT pn.network ORDER BY pn.network) as networks_list,
    u.name as user_name
FROM posts p
LEFT JOIN post_networks pn ON p.id = pn.post_id AND pn.is_active = 1
LEFT JOIN post_medias pm ON p.id = pm.post_id
LEFT JOIN scheduled_posts sp ON p.id = sp.post_id AND sp.status IN ('pending', 'processing')
LEFT JOIN users u ON p.user_id = u.id
GROUP BY p.id;

-- Vue transmissions avec détails
CREATE OR REPLACE VIEW view_transmission_details AS
SELECT 
    tl.*,
    p.titre as post_title,
    p.contenu as post_content,
    sp.scheduled_datetime,
    sp.status as schedule_status,
    u.name as user_name,
    plt.display_name as network_display_name,
    plt.icon as network_icon
FROM transmission_logs tl
JOIN scheduled_posts sp ON tl.scheduled_post_id = sp.id
JOIN posts p ON sp.post_id = p.id
JOIN users u ON sp.user_id = u.id
LEFT JOIN social_platforms plt ON tl.network = plt.name;

-- ================================================================
-- ✅ VALIDATION STRUCTURE
-- ================================================================

-- Vérifier que les tables principales existent
SELECT 
    TABLE_NAME,
    ENGINE,
    TABLE_ROWS,
    CREATE_TIME
FROM information_schema.TABLES 
WHERE TABLE_SCHEMA = 'laloedata' 
AND TABLE_NAME IN ('posts', 'post_networks', 'scheduled_posts', 'transmission_logs', 'user_downloads', 'social_platforms')
ORDER BY TABLE_NAME;

-- Vérifier les contraintes foreign key
SELECT 
    CONSTRAINT_NAME,
    TABLE_NAME,
    COLUMN_NAME,
    REFERENCED_TABLE_NAME,
    REFERENCED_COLUMN_NAME
FROM information_schema.KEY_COLUMN_USAGE 
WHERE TABLE_SCHEMA = 'laloedata' 
AND REFERENCED_TABLE_NAME IS NOT NULL
ORDER BY TABLE_NAME, CONSTRAINT_NAME;