<?php
/**
 * ================================================================
 * 🔗 SOCIAL-ACCOUNTS.PHP - MULTI-USER ENHANCED
 * ================================================================
 * 
 * 🎯 FONCTIONNALITÉS MULTI-USER :
 * - Isolation complète par utilisateur
 * - Gestion des rôles et permissions
 * - Quotas par utilisateur
 * - Audit trail complet
 * - Sécurité renforcée
 * 
 * 💡 BY PAPY 77 - VERSION MULTI-USER COMPLÈTE !
 * ================================================================
 */

require_once 'auth_functions.php';
requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

// ================================================================
// 🔒 VÉRIFICATIONS SÉCURITÉ MULTI-USER
// ================================================================

// Vérifier les permissions utilisateur
if (!hasPermission($user, 'manage_social_accounts')) {
    die('❌ Vous n\'avez pas les permissions pour gérer les comptes sociaux');
}

// Récupérer les limites utilisateur
$user_limits = getUserLimits($user['id'], $pdo);

// Configuration des réseaux sociaux (inchangée)
$networks_config = [
    'facebook' => [
        'name' => 'Facebook',
        'icon' => '📘',
        'color' => '#1877f2',
        'description' => 'Pages et comptes business',
        'max_accounts' => $user_limits['facebook_accounts'] ?? 1,
        'fields' => [
            'access_token' => ['label' => 'Page Access Token', 'type' => 'password', 'required' => true],
            'page_id' => ['label' => 'Page ID', 'type' => 'text', 'required' => true],
            'app_id' => ['label' => 'App ID', 'type' => 'text', 'required' => false]
        ]
    ],
    'instagram' => [
        'name' => 'Instagram',
        'icon' => '📸',
        'color' => '#E4405F',
        'description' => 'Compte business/créateur',
        'max_accounts' => $user_limits['instagram_accounts'] ?? 1,
        'fields' => [
            'instagram_id' => ['label' => 'Instagram Business Account ID', 'type' => 'text', 'required' => true],
            'access_token' => ['label' => 'Facebook Page Access Token', 'type' => 'password', 'required' => true],
            'page_id' => ['label' => 'Facebook Page ID', 'type' => 'text', 'required' => true]
        ]
    ],
    'tiktok' => [
        'name' => 'TikTok',
        'icon' => '🎵',
        'color' => '#000000',
        'description' => 'TikTok for Business',
        'max_accounts' => $user_limits['tiktok_accounts'] ?? 1,
        'fields' => [
            'access_token' => ['label' => 'Access Token', 'type' => 'password', 'required' => true],
            'user_id' => ['label' => 'User ID', 'type' => 'text', 'required' => true],
            'client_key' => ['label' => 'Client Key', 'type' => 'text', 'required' => false]
        ]
    ],
    'linkedin' => [
        'name' => 'LinkedIn',
        'icon' => '💼',
        'color' => '#0A66C2',
        'description' => 'Pages et profils professionnels',
        'max_accounts' => $user_limits['linkedin_accounts'] ?? 1,
        'fields' => [
            'access_token' => ['label' => 'Access Token', 'type' => 'password', 'required' => true],
            'organization_id' => ['label' => 'Organization ID', 'type' => 'text', 'required' => true],
            'client_id' => ['label' => 'Client ID', 'type' => 'text', 'required' => false]
        ]
    ],
    'snapchat' => [
        'name' => 'Snapchat',
        'icon' => '👻',
        'color' => '#FFFC00',
        'description' => 'Snapchat Business',
        'max_accounts' => $user_limits['snapchat_accounts'] ?? 1,
        'fields' => [
            'access_token' => ['label' => 'Access Token', 'type' => 'password', 'required' => true],
            'ad_account_id' => ['label' => 'Ad Account ID', 'type' => 'text', 'required' => true],
            'client_id' => ['label' => 'Client ID', 'type' => 'text', 'required' => false]
        ]
    ]
];

// ================================================================
// 🗄️ CHARGEMENT COMPTES (ISOLATION PAR USER_ID)
// ================================================================

$connected_accounts = [];
$account_counts = [];

try {
    // Récupérer SEULEMENT les comptes de l'utilisateur connecté
    $stmt = $pdo->prepare("
        SELECT platform, username, platform_user_id, is_active, 
               created_at, updated_at, account_data, id
        FROM social_accounts 
        WHERE user_id = ? 
        ORDER BY platform, created_at DESC
    ");
    $stmt->execute([$user['id']]);
    $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($accounts as $account) {
        if (!isset($connected_accounts[$account['platform']])) {
            $connected_accounts[$account['platform']] = [];
            $account_counts[$account['platform']] = 0;
        }
        $connected_accounts[$account['platform']][] = $account;
        $account_counts[$account['platform']]++;
    }
    
} catch (PDOException $e) {
    error_log("Erreur chargement comptes sociaux (user_id: {$user['id']}): " . $e->getMessage());
}

// ================================================================
// 📊 STATISTIQUES UTILISATEUR
// ================================================================

$user_stats = [
    'total_accounts' => array_sum($account_counts),
    'active_accounts' => 0,
    'last_connection' => null
];

foreach ($connected_accounts as $platform_accounts) {
    foreach ($platform_accounts as $account) {
        if ($account['is_active']) {
            $user_stats['active_accounts']++;
        }
    }
}

// ================================================================
// ⚡ TRAITEMENT ACTIONS AVEC SÉCURITÉ RENFORCÉE
// ================================================================

$message = '';
$message_type = '';

if ($_POST) {
    $action = $_POST['action'] ?? '';
    
    // Vérifier le token CSRF
    if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf_token'] ?? '')) {
        $message = 'Token de sécurité invalide';
        $message_type = 'error';
    } else {
        switch ($action) {
            case 'connect':
                $result = connectSocialAccountSecure($_POST, $user, $pdo, $networks_config);
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'error';
                break;
                
            case 'disconnect':
                $result = disconnectSocialAccountSecure($_POST, $user, $pdo);
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'error';
                break;
                
            case 'test':
                $result = testSocialConnectionSecure($_POST, $user, $pdo);
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'error';
                break;
        }
        
        // Rechargement des comptes après action
        if ($message_type === 'success') {
            header("Location: " . $_SERVER['PHP_SELF'] . "?success=1");
            exit;
        }
    }
}

// Générer token CSRF pour les formulaires
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// ================================================================
// 🔒 FONCTIONS SÉCURISÉES MULTI-USER
// ================================================================

/**
 * Connexion sécurisée avec vérifications multi-user
 */
function connectSocialAccountSecure($data, $user, $pdo, $networks_config) {
    $platform = $data['platform'] ?? '';
    $credentials = $data['credentials'] ?? [];
    
    // Vérifications de base
    if (!$platform || !$credentials['access_token']) {
        return ['success' => false, 'message' => 'Données manquantes'];
    }
    
    if (!isset($networks_config[$platform])) {
        return ['success' => false, 'message' => 'Plateforme non supportée'];
    }
    
    // Vérifier les limites utilisateur
    $current_count = getCurrentAccountCount($user['id'], $platform, $pdo);
    $max_allowed = $networks_config[$platform]['max_accounts'];
    
    if ($current_count >= $max_allowed) {
        return ['success' => false, 'message' => "Limite atteinte: {$max_allowed} compte(s) maximum pour {$platform}"];
    }
    
    try {
        // Validation du token
        $validation = validateToken($platform, $credentials);
        
        if (!$validation['success']) {
            // Log de tentative échouée
            logSecurityEvent($user['id'], 'invalid_token_attempt', $platform, 
                           "Tentative de connexion avec token invalide", $pdo);
            return ['success' => false, 'message' => 'Token invalide: ' . $validation['error']];
        }
        
        // Vérifier si ce compte n'existe pas déjà
        if (accountAlreadyExists($user['id'], $platform, $validation['user_id'], $pdo)) {
            return ['success' => false, 'message' => 'Ce compte est déjà connecté'];
        }
        
        // Chiffrement sécurisé des credentials
        $encrypted_token = encryptTokenSecure($credentials['access_token'], $user['id']);
        $account_data = json_encode($credentials);
        
        // Sauvegarde avec audit
        $stmt = $pdo->prepare("
            INSERT INTO social_accounts 
            (user_id, platform, platform_user_id, username, access_token, 
             account_data, is_active, ip_address, user_agent)
            VALUES (?, ?, ?, ?, ?, ?, 1, ?, ?)
        ");
        
        $stmt->execute([
            $user['id'],
            $platform,
            $validation['user_id'] ?? 'unknown',
            $validation['username'] ?? 'user',
            $encrypted_token,
            $account_data,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ]);
        
        // Log de succès
        logSecurityEvent($user['id'], 'connect_social', $platform, 
                       "Connexion réussie: {$validation['username']}", $pdo);
        
        return ['success' => true, 'message' => ucfirst($platform) . ' connecté avec succès !'];
        
    } catch (Exception $e) {
        error_log("Erreur connexion $platform (user_id: {$user['id']}): " . $e->getMessage());
        logSecurityEvent($user['id'], 'connect_error', $platform, 
                       "Erreur de connexion: " . $e->getMessage(), $pdo);
        return ['success' => false, 'message' => 'Erreur de connexion: ' . $e->getMessage()];
    }
}

/**
 * Déconnexion sécurisée avec vérification de propriété
 */
function disconnectSocialAccountSecure($data, $user, $pdo) {
    $platform = $data['platform'] ?? '';
    $account_id = $data['account_id'] ?? null;
    
    if (!$platform) {
        return ['success' => false, 'message' => 'Plateforme manquante'];
    }
    
    try {
        // Vérifier que le compte appartient bien à l'utilisateur
        $stmt = $pdo->prepare("
            SELECT id, username FROM social_accounts 
            WHERE user_id = ? AND platform = ? AND (id = ? OR ? IS NULL)
            LIMIT 1
        ");
        $stmt->execute([$user['id'], $platform, $account_id, $account_id]);
        $account = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$account) {
            logSecurityEvent($user['id'], 'unauthorized_disconnect', $platform, 
                           "Tentative de déconnexion d'un compte non possédé", $pdo);
            return ['success' => false, 'message' => 'Compte non trouvé ou non autorisé'];
        }
        
        // Supprimer le compte
        $stmt = $pdo->prepare("DELETE FROM social_accounts WHERE id = ? AND user_id = ?");
        $stmt->execute([$account['id'], $user['id']]);
        
        logSecurityEvent($user['id'], 'disconnect_social', $platform, 
                       "Déconnexion: {$account['username']}", $pdo);
        
        return ['success' => true, 'message' => ucfirst($platform) . ' déconnecté'];
        
    } catch (PDOException $e) {
        error_log("Erreur déconnexion $platform (user_id: {$user['id']}): " . $e->getMessage());
        return ['success' => false, 'message' => 'Erreur de déconnexion'];
    }
}

/**
 * Test sécurisé avec vérification de propriété
 */
function testSocialConnectionSecure($data, $user, $pdo) {
    $platform = $data['platform'] ?? '';
    $account_id = $data['account_id'] ?? null;
    
    try {
        // Vérifier ownership
        $stmt = $pdo->prepare("
            SELECT access_token, username FROM social_accounts 
            WHERE user_id = ? AND platform = ? AND (id = ? OR ? IS NULL)
            LIMIT 1
        ");
        $stmt->execute([$user['id'], $platform, $account_id, $account_id]);
        $account = $stmt->fetch();
        
        if (!$account) {
            return ['success' => false, 'message' => 'Compte non autorisé'];
        }
        
        $decrypted_token = decryptTokenSecure($account['access_token'], $user['id']);
        $test_result = performAPITest($platform, $decrypted_token);
        
        // Log du test
        logSecurityEvent($user['id'], 'test_connection', $platform, 
                       "Test connexion: " . ($test_result['success'] ? 'Succès' : 'Échec'), $pdo);
        
        return $test_result;
        
    } catch (Exception $e) {
        error_log("Erreur test $platform (user_id: {$user['id']}): " . $e->getMessage());
        return ['success' => false, 'message' => 'Erreur de test'];
    }
}

// ================================================================
// 🛡️ FONCTIONS UTILITAIRES SÉCURITÉ
// ================================================================

/**
 * Vérifier les permissions utilisateur
 */
function hasPermission($user, $permission) {
    $permissions = [
        'user' => ['manage_social_accounts'],
        'manager' => ['manage_social_accounts', 'view_all_accounts'],
        'admin' => ['manage_social_accounts', 'view_all_accounts', 'manage_users']
    ];
    
    return in_array($permission, $permissions[$user['role']] ?? []);
}

/**
 * Récupérer les limites utilisateur
 */
function getUserLimits($user_id, $pdo) {
    try {
        $stmt = $pdo->prepare("
            SELECT platform, max_accounts FROM user_limits 
            WHERE user_id = ?
        ");
        $stmt->execute([$user_id]);
        $limits = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        // Limites par défaut
        $default_limits = [
            'facebook_accounts' => 2,
            'instagram_accounts' => 2,
            'tiktok_accounts' => 1,
            'linkedin_accounts' => 1,
            'snapchat_accounts' => 1
        ];
        
        return array_merge($default_limits, $limits);
        
    } catch (PDOException $e) {
        error_log("Erreur récupération limites user_id $user_id: " . $e->getMessage());
        return [
            'facebook_accounts' => 1,
            'instagram_accounts' => 1,
            'tiktok_accounts' => 1,
            'linkedin_accounts' => 1,
            'snapchat_accounts' => 1
        ];
    }
}

/**
 * Compter les comptes actuels
 */
function getCurrentAccountCount($user_id, $platform, $pdo) {
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM social_accounts 
            WHERE user_id = ? AND platform = ?
        ");
        $stmt->execute([$user_id, $platform]);
        return (int) $stmt->fetchColumn();
    } catch (PDOException $e) {
        return 0;
    }
}

/**
 * Vérifier si le compte existe déjà
 */
function accountAlreadyExists($user_id, $platform, $platform_user_id, $pdo) {
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM social_accounts 
            WHERE user_id = ? AND platform = ? AND platform_user_id = ?
        ");
        $stmt->execute([$user_id, $platform, $platform_user_id]);
        return $stmt->fetchColumn() > 0;
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * Chiffrement sécurisé avec clé utilisateur
 */
function encryptTokenSecure($token, $user_id) {
    $key = hash('sha256', 'SMG_SECRET_KEY_PAPY77' . $user_id);
    $iv = random_bytes(16);
    $encrypted = openssl_encrypt($token, 'AES-256-CBC', $key, 0, $iv);
    return base64_encode($iv . $encrypted);
}

/**
 * Déchiffrement sécurisé avec clé utilisateur
 */
function decryptTokenSecure($encrypted_token, $user_id) {
    $key = hash('sha256', 'SMG_SECRET_KEY_PAPY77' . $user_id);
    $data = base64_decode($encrypted_token);
    $iv = substr($data, 0, 16);
    $encrypted = substr($data, 16);
    return openssl_decrypt($encrypted, 'AES-256-CBC', $key, 0, $iv);
}

/**
 * Log des événements de sécurité
 */
function logSecurityEvent($user_id, $action, $platform, $description, $pdo) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO security_logs 
            (user_id, action, platform, description, ip_address, user_agent, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $user_id, 
            $action, 
            $platform, 
            $description, 
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ]);
    } catch (PDOException $e) {
        error_log("Erreur log sécurité: " . $e->getMessage());
    }
}

// Reprendre toutes les autres fonctions existantes (validateToken, performAPITest, etc.)
// ... [code existant] ...

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🔗 Comptes Sociaux - <?php echo htmlspecialchars($user['name']); ?> - SMG</title>
    <style>
        /* Reprendre le CSS existant + ajouts pour multi-user */
        
        .user-info {
            background: rgba(255, 255, 255, 0.1);
            color: white;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .user-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin: 20px 0;
        }
        
        .stat-card {
            background: white;
            padding: 15px;
            border-radius: 8px;
            text-align: center;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .stat-number {
            font-size: 2em;
            font-weight: bold;
            color: #667eea;
        }
        
        .stat-label {
            font-size: 0.9em;
            color: #666;
            margin-top: 5px;
        }
        
        .quota-warning {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            color: #856404;
            padding: 10px;
            border-radius: 6px;
            margin: 10px 0;
            font-size: 0.9em;
        }
        
        .quota-ok {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
            padding: 10px;
            border-radius: 6px;
            margin: 10px 0;
            font-size: 0.9em;
        }
        
        /* CSS existant conservé */
        /* ... */
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <a href="dashboard.php" class="back-btn">← Dashboard</a>
            <h1>🔗 Mes Comptes Réseaux Sociaux</h1>
            <p>Gérez vos connexions aux réseaux sociaux en toute sécurité</p>
            
            <div class="user-info">
                <strong>👤 <?php echo htmlspecialchars($user['name']); ?></strong> 
                (<?php echo ucfirst($user['role']); ?>)
            </div>
        </div>
        
        <div class="content">
            <!-- Statistiques utilisateur -->
            <div class="user-stats">
                <div class="stat-card">
                    <div class="stat-number"><?php echo $user_stats['total_accounts']; ?></div>
                    <div class="stat-label">Comptes connectés</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo $user_stats['active_accounts']; ?></div>
                    <div class="stat-label">Comptes actifs</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo count($networks_config); ?></div>
                    <div class="stat-label">Réseaux disponibles</div>
                </div>
            </div>
            
            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?>">
                    <strong><?php echo $message_type === 'success' ? '✅' : '❌'; ?></strong> 
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <div class="networks-grid">
                <?php foreach ($networks_config as $platform => $config): ?>
                    <?php 
                    $current_count = $account_counts[$platform] ?? 0;
                    $max_allowed = $config['max_accounts'];
                    $can_add_more = $current_count < $max_allowed;
                    ?>
                    
                    <div class="network-card <?php echo $platform; ?>">
                        <div class="network-header">
                            <div class="network-icon"><?php echo $config['icon']; ?></div>
                            <div class="network-title">
                                <h3><?php echo $config['name']; ?></h3>
                                <p><?php echo $config['description']; ?></p>
                            </div>
                            <span class="connection-status <?php echo $current_count > 0 ? 'status-connected' : 'status-disconnected'; ?>">
                                <?php echo $current_count; ?>/<?php echo $max_allowed; ?> comptes
                            </span>
                        </div>
                        
                        <!-- Affichage des quotas -->
                        <?php if ($current_count >= $max_allowed): ?>
                            <div class="quota-warning">
                                ⚠️ Quota atteint: <?php echo $max_allowed; ?> compte(s) maximum
                            </div>
                        <?php else: ?>
                            <div class="quota-ok">
                                ✅ Vous pouvez ajouter <?php echo ($max_allowed - $current_count); ?> compte(s) supplémentaire(s)
                            </div>
                        <?php endif; ?>
                        
                        <!-- Affichage des comptes existants -->
                        <?php if (isset($connected_accounts[$platform])): ?>
                            <?php foreach ($connected_accounts[$platform] as $account): ?>
                                <div class="connection-details">
                                    <h4>📊 @<?php echo htmlspecialchars($account['username']); ?></h4>
                                    <div class="detail-item">
                                        <span>Connecté le :</span>
                                        <span><?php echo date('d/m/Y H:i', strtotime($account['created_at'])); ?></span>
                                    </div>
                                    <div class="detail-item">
                                        <span>Statut :</span>
                                        <span><?php echo $account['is_active'] ? '🟢 Actif' : '🔴 Inactif'; ?></span>
                                    </div>
                                    
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="action" value="test">
                                        <input type="hidden" name="platform" value="<?php echo $platform; ?>">
                                        <input type="hidden" name="account_id" value="<?php echo $account['id']; ?>">
                                        <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                        <button type="submit" class="btn btn-test">🧪 Tester</button>
                                    </form>
                                    
                                    <form method="POST" style="display: inline;" 
                                          onsubmit="return confirm('Déconnecter ce compte ?')">
                                        <input type="hidden" name="action" value="disconnect">
                                        <input type="hidden" name="platform" value="<?php echo $platform; ?>">
                                        <input type="hidden" name="account_id" value="<?php echo $account['id']; ?>">
                                        <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                        <button type="submit" class="btn btn-disconnect">🔌 Déconnecter</button>
                                    </form>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                        
                        <!-- Formulaire d'ajout si quota non atteint -->
                        <?php if ($can_add_more): ?>
                            <div class="info-box">
                                <h4>🔑 Ajouter un compte <?php echo $config['name']; ?> :</h4>
                            </div>
                            
                            <form method="POST" class="token-form">
                                <input type="hidden" name="action" value="connect">
                                <input type="hidden" name="platform" value="<?php echo $platform; ?>">
                                <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                
                                <?php foreach ($config['fields'] as $field_name => $field_config): ?>
                                    <div class="form-group">
                                        <label class="form-label">
                                            <?php echo $field_config['label']; ?>
                                            <?php if ($field_config['required']): ?> *<?php endif; ?>
                                        </label>
                                        <input 
                                            type="<?php echo $field_config['type']; ?>" 
                                            class="form-input" 
                                            name="credentials[<?php echo $field_name; ?>]"
                                            <?php if ($field_config['required']): ?>required<?php endif; ?>
                                            placeholder="<?php echo $field_config['type'] === 'password' ? '••••••••••' : 'Saisissez ' . strtolower($field_config['label']); ?>">
                                    </div>
                                <?php endforeach; ?>
                                
                                <button type="submit" class="btn btn-primary">
                                    ➕ Ajouter <?php echo $config['name']; ?>
                                </button>
                            </form>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        
        <div class="global-actions">
            <h3>🔧 Mes Actions</h3>
            <p style="margin-bottom: 20px; color: #666;">
                Utilisateur: <strong><?php echo htmlspecialchars($user['name']); ?></strong> | 
                Total: <?php echo $user_stats['total_accounts']; ?> comptes connectés
            </p>
            
            <button class="btn btn-primary" onclick="window.location.reload()">
                🔄 Actualiser
            </button>
            
            <a href="dashboard.php" class="btn btn-primary">
                📊 Retour au Dashboard
            </a>
        </div>
    </div>

    <script>
        console.log('🔗 Social Accounts Manager - Multi-User Ready');
        console.log('👤 Current User: <?php echo $user['name']; ?> (ID: <?php echo $user['id']; ?>)');
        console.log('📊 User Stats:', <?php echo json_encode($user_stats); ?>);
        console.log('🔒 Security: CSRF protection enabled');
    </script>
</body>
</html>