<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SMG Studio - Clipmaker Pro</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #0a0a0a;
            color: #ffffff;
            overflow: hidden;
        }

        .container {
            width: 100vw;
            height: 100vh;
            display: flex;
            flex-direction: column;
        }

        /* Header */
        .header {
            background: linear-gradient(135deg, #1a1a1a 0%, #2d2d2d 100%);
            padding: 15px 30px;
            border-bottom: 1px solid #333;
            box-shadow: 0 2px 20px rgba(0,0,0,0.4);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .clip-title {
            font-size: 24px;
            font-weight: bold;
            background: linear-gradient(45deg, #ff6b6b, #4ecdc4);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        /* Boutons header */
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }

        .header-btn {
            padding: 8px 16px;
            border: none;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .header-btn.back { 
            background: #666; 
            color: white; 
        }

        .header-btn.quit { 
            background: linear-gradient(45deg, #ff6b6b, #ff9ff3); 
            color: white; 
        }

        .header-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        }

        /* Zone principale */
        .main-content {
            flex: 1;
            display: grid;
            grid-template-columns: 280px 1fr;
            overflow: hidden;
        }

        /* Sidebar gauche */
        .sidebar {
            background: #1a1a1a;
            border-right: 1px solid #333;
            overflow-y: auto;
            display: flex;
            flex-direction: column;
        }

        .section-header {
            padding: 15px 20px;
            background: #222;
            border-bottom: 1px solid #333;
            font-weight: 600;
            color: #4ecdc4;
            font-size: 14px;
        }

        /* Zone de preview et extraction */
        .preview-zone {
            flex: 1;
            display: flex;
            flex-direction: column;
        }

        /* Contrôles vidéo personnalisés */
        .custom-video-controls {
            position: absolute;
            bottom: 15px;
            left: 50%;
            transform: translateX(-50%);
            display: flex;
            align-items: center;
            gap: 8px;
            background: rgba(0,0,0,0.8);
            padding: 8px 15px;
            border-radius: 25px;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255,255,255,0.1);
        }

        .video-btn {
            width: 36px;
            height: 36px;
            border: none;
            border-radius: 50%;
            background: rgba(255,255,255,0.15);
            color: white;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            transition: all 0.3s ease;
            backdrop-filter: blur(5px);
        }

        .video-btn:hover {
            background: #4ecdc4;
            transform: scale(1.1);
            box-shadow: 0 4px 12px rgba(78, 205, 196, 0.3);
        }

        .video-btn.play-pause {
            width: 44px;
            height: 44px;
            font-size: 16px;
            background: linear-gradient(45deg, #4ecdc4, #45b7d1);
        }

        .volume-control {
            display: flex;
            align-items: center;
            gap: 6px;
            margin-left: 8px;
        }

        .volume-slider {
            width: 60px;
            height: 4px;
            background: rgba(255,255,255,0.2);
            border-radius: 2px;
            outline: none;
            cursor: pointer;
        }

        .volume-slider::-webkit-slider-thumb {
            appearance: none;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: #4ecdc4;
            cursor: pointer;
            box-shadow: 0 2px 6px rgba(0,0,0,0.3);
        }

        .time-info {
            margin-left: 10px;
            font-size: 11px;
            color: rgba(255,255,255,0.8);
            font-family: monospace;
            min-width: 80px;
            text-align: center;
        }

        .video-container {
            flex: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            background: #111;
            position: relative;
        }

        .video-preview {
            max-width: 90%;
            max-height: 90%;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.5);
        }

        /* Timeline d'extraction */
        .extraction-timeline {
            height: 120px;
            background: #1a1a1a;
            border-top: 1px solid #333;
            padding: 10px 20px;
        }

        .extraction-controls {
            display: flex;
            gap: 10px;
            margin-bottom: 10px;
        }

        .control-btn {
            padding: 6px 12px;
            border: none;
            border-radius: 15px;
            font-size: 11px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .control-btn.in { background: #4ecdc4; color: white; }
        .control-btn.out { background: #ff6b6b; color: white; }
        .control-btn.extract { background: linear-gradient(45deg, #feca57, #ff9ff3); color: white; }
        .control-btn.clear { background: #666; color: white; }

        .control-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(0,0,0,0.3);
        }

        .timeline-ruler {
            height: 40px;
            background: #2d2d2d;
            border-radius: 4px;
            position: relative;
            cursor: pointer;
        }

        .playhead {
            position: absolute;
            top: 0;
            bottom: 0;
            width: 2px;
            background: #ff6b6b;
            z-index: 100;
            box-shadow: 0 0 8px rgba(255, 107, 107, 0.8);
        }

        .marker {
            position: absolute;
            top: 0;
            bottom: 0;
            width: 2px;
            z-index: 90;
        }
        .marker.in { background: #4ecdc4; }
        .marker.out { background: #ff6b6b; }

        .selection-zone {
            position: absolute;
            top: 0;
            bottom: 0;
            background: rgba(78, 205, 196, 0.2);
            border: 1px solid #4ecdc4;
            z-index: 50;
        }

        /* Timeline horizontale principale */
        .horizontal-timeline {
            height: 200px;
            background: #1a1a1a;
            border-top: 2px solid #333;
            display: flex;
            flex-direction: column;
        }

        .timeline-header {
            padding: 10px 20px;
            background: #222;
            border-bottom: 1px solid #333;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .timeline-title {
            font-weight: 600;
            color: #4ecdc4;
            font-size: 14px;
        }

        .timeline-controls {
            display: flex;
            gap: 8px;
        }

        .timeline-btn {
            padding: 6px 12px;
            border: none;
            border-radius: 15px;
            font-size: 11px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .timeline-btn.preview { background: #4ecdc4; color: white; }
        .timeline-btn.save { background: #feca57; color: white; }
        .timeline-btn.load { background: #45b7d1; color: white; }
        .timeline-btn.transitions { background: linear-gradient(45deg, #ff9ff3, #feca57); color: white; }
        .timeline-btn.export { background: linear-gradient(45deg, #ff6b6b, #4ecdc4); color: white; }

        .timeline-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 3px 10px rgba(0,0,0,0.3);
        }

        .timeline-track {
            flex: 1;
            margin: 10px 20px;
            background: #2d2d2d;
            border-radius: 6px;
            position: relative;
            min-height: 80px;
            border: 2px dashed #444;
            transition: all 0.3s ease;
        }

        .timeline-track.drag-over {
            border-color: #4ecdc4;
            background: rgba(78, 205, 196, 0.1);
        }

        .timeline-empty {
            display: flex;
            align-items: center;
            justify-content: center;
            height: 100%;
            color: #666;
            font-size: 12px;
        }

        .sequence-clip {
            position: absolute;
            height: 60px;
            top: 10px;
            background: linear-gradient(135deg, #4ecdc4, #45b7d1);
            border-radius: 4px;
            display: flex;
            align-items: center;
            padding: 0 8px;
            font-size: 10px;
            color: white;
            font-weight: 600;
            cursor: pointer;
            border: 2px solid transparent;
            transition: all 0.3s ease;
            user-select: none;
            box-shadow: 0 2px 8px rgba(0,0,0,0.3);
        }

        .sequence-clip:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            border-color: rgba(255,255,255,0.3);
        }

        .sequence-clip.selected {
            border-color: #ff6b6b;
            background: linear-gradient(135deg, #ff6b6b, #ff9ff3);
        }

        .sequence-clip.dragging {
            opacity: 0.7;
            transform: rotate(3deg) scale(1.05);
            z-index: 1000;
        }

        .transition-zone {
            position: absolute;
            width: 20px;
            height: 30px;
            top: 25px;
            background: rgba(254, 202, 87, 0.8);
            border-radius: 10px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 8px;
            color: white;
            font-weight: bold;
            transition: all 0.3s ease;
            border: 1px solid #feca57;
        }

        .transition-zone:hover {
            background: linear-gradient(45deg, #feca57, #ff9ff3);
            transform: scale(1.1);
        }

        .transition-zone.has-transition {
            background: linear-gradient(45deg, #feca57, #ff9ff3);
            border-color: #ff9ff3;
        }

        /* Sidebar content */
        .clip-management {
            padding: 15px;
            border-bottom: 1px solid #333;
        }

        .input-group {
            margin-bottom: 10px;
        }

        .input-group label {
            display: block;
            color: #4ecdc4;
            font-size: 11px;
            margin-bottom: 4px;
            text-transform: uppercase;
        }

        .input-group input {
            width: 100%;
            padding: 8px;
            background: #2d2d2d;
            border: 1px solid #444;
            border-radius: 6px;
            color: white;
            font-size: 14px;
            font-weight: 500;
        }

        .input-group input:focus {
            outline: none;
            border-color: #4ecdc4;
            box-shadow: 0 0 8px rgba(78, 205, 196, 0.3);
        }

        .action-buttons {
            display: flex;
            gap: 8px;
            margin-top: 15px;
        }

        .action-btn {
            flex: 1;
            padding: 10px;
            border: none;
            border-radius: 6px;
            font-size: 12px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .action-btn.save { 
            background: linear-gradient(45deg, #4ecdc4, #45b7d1); 
            color: white; 
        }

        .action-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        }

        .upload-area {
            padding: 15px;
            border-bottom: 1px solid #333;
        }

        .upload-btn {
            width: 100%;
            padding: 12px;
            background: linear-gradient(45deg, #4ecdc4, #45b7d1);
            border: none;
            border-radius: 6px;
            color: white;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .upload-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(78, 205, 196, 0.3);
        }

        .rush-info {
            padding: 15px;
            display: none;
        }

        .rush-card {
            background: #2d2d2d;
            border-radius: 6px;
            padding: 10px;
        }

        .rush-name {
            font-weight: 600;
            margin-bottom: 4px;
            font-size: 12px;
        }

        .rush-duration {
            font-size: 10px;
            color: #888;
            margin-bottom: 8px;
        }

        .remove-rush-btn {
            width: 100%;
            padding: 6px;
            background: #ff6b6b;
            border: none;
            border-radius: 4px;
            color: white;
            font-size: 10px;
            cursor: pointer;
        }

        .status-info {
            padding: 15px;
            font-size: 11px;
            color: #666;
        }

        .status-info div {
            margin-bottom: 4px;
        }

        /* Panel transitions */
        .transitions-panel {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            width: 400px;
            max-height: 500px;
            background: #1a1a1a;
            border: 1px solid #333;
            border-radius: 8px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.5);
            z-index: 2000;
            display: none;
        }

        .transitions-header {
            padding: 15px 20px;
            border-bottom: 1px solid #333;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .transitions-header h3 {
            color: #4ecdc4;
            font-size: 16px;
        }

        .close-btn {
            background: none;
            border: none;
            color: #888;
            font-size: 18px;
            cursor: pointer;
        }

        .close-btn:hover {
            color: #ff6b6b;
        }

        .transitions-list {
            max-height: 300px;
            overflow-y: auto;
            padding: 10px;
        }

        .transition-item {
            background: #2d2d2d;
            border-radius: 6px;
            padding: 10px;
            margin-bottom: 6px;
            cursor: pointer;
            transition: all 0.3s ease;
            border-left: 3px solid #feca57;
        }

        .transition-item:hover {
            background: #333;
            transform: translateX(3px);
        }

        .transition-item.selected {
            background: rgba(254, 202, 87, 0.2);
            border-left-color: #ff9ff3;
        }

        .transition-name {
            font-weight: 600;
            margin-bottom: 3px;
            color: #feca57;
            font-size: 12px;
        }

        .transition-info {
            font-size: 10px;
            color: #888;
            display: flex;
            justify-content: space-between;
        }

        .duration-controls {
            padding: 15px;
        }

        .duration-slider {
            margin: 10px 0;
        }

        .duration-slider label {
            color: #888;
            font-size: 11px;
            display: block;
            margin-bottom: 5px;
        }

        .duration-value {
            color: #4ecdc4;
            font-weight: bold;
        }

        .apply-transition-btn {
            background: linear-gradient(45deg, #feca57, #ff9ff3);
            border: none;
            padding: 10px 20px;
            border-radius: 20px;
            color: white;
            font-weight: bold;
            cursor: pointer;
            width: 100%;
        }

        /* Notifications */
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            background: linear-gradient(45deg, #4ecdc4, #45b7d1);
            color: white;
            padding: 12px 20px;
            border-radius: 6px;
            font-weight: 500;
            z-index: 10000;
            box-shadow: 0 5px 15px rgba(0,0,0,0.3);
            transform: translateX(100%);
            transition: transform 0.3s ease;
        }

        .notification.show {
            transform: translateX(0);
        }

        /* Modal de sortie */
        .exit-modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.8);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 10000;
        }

        .exit-modal-content {
            background: #1a1a1a;
            padding: 30px;
            border-radius: 12px;
            border: 1px solid #333;
            text-align: center;
            max-width: 400px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.5);
        }

        .exit-modal h3 {
            color: #4ecdc4;
            margin-bottom: 15px;
            font-size: 18px;
        }

        .exit-modal p {
            color: #ccc;
            margin-bottom: 25px;
            line-height: 1.4;
        }

        .exit-modal-buttons {
            display: flex;
            gap: 15px;
            justify-content: center;
        }

        .exit-btn {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .exit-btn.save { 
            background: linear-gradient(45deg, #4ecdc4, #45b7d1); 
            color: white; 
        }

        .exit-btn.abandon { 
            background: #ff6b6b; 
            color: white; 
        }

        .exit-btn.cancel { 
            background: #666; 
            color: white; 
        }

        .exit-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        }

        /* Responsive */
        @media (max-width: 1200px) {
            .main-content {
                grid-template-columns: 250px 1fr;
            }
        }

        @media (max-width: 800px) {
            .main-content {
                grid-template-columns: 1fr;
                grid-template-rows: 1fr 180px;
            }
            
            .sidebar {
                order: 2;
                flex-direction: row;
                overflow-x: auto;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Header -->
        <header class="header">
            <div class="clip-title">
                🎬 SMG Studio - Clipmaker Pro
            </div>
            <div class="header-actions">
                <a href="javascript:history.back()" class="header-btn back">
                    ← Retour Workflow
                </a>
                <button class="header-btn quit" onclick="showExitModal()">
                    🚪 Quitter
                </button>
            </div>
        </header>

        <!-- Zone principale -->
        <div class="main-content">
            <!-- Sidebar gauche -->
            <aside class="sidebar">
                <!-- Gestion du clip -->
                <div class="section-header">📋 Gestion Clip</div>
                <div class="clip-management">
                    <div class="input-group">
                        <label>Nom du Clip</label>
                        <input type="text" id="clipNameInput" value="" placeholder="Nom du clip...">
                    </div>
                    <div class="action-buttons">
                        <button class="action-btn save" onclick="saveClip()">💾 Sauvegarder</button>
                    </div>
                </div>

                <!-- Upload rush -->
                <div class="section-header">📹 Rush Vidéo</div>
                <div class="upload-area">
                    <button class="upload-btn" onclick="uploadRush()">
                        ⬆️ Charger Rush Vidéo
                    </button>
                    <div style="font-size: 10px; color: #666; margin-top: 6px; text-align: center;">
                        MP4, MOV, AVI, WEBM
                    </div>
                </div>

                <!-- Info rush -->
                <div id="rushInfo" class="rush-info">
                    <div class="rush-card">
                        <div class="rush-name" id="rushName">-</div>
                        <div class="rush-duration" id="rushDuration">-</div>
                        <button class="remove-rush-btn" onclick="clearRush()">
                            🗑️ Supprimer Rush
                        </button>
                    </div>
                </div>

                <!-- Statut -->
                <div class="section-header">📊 Statut</div>
                <div class="status-info">
                    <div>🎬 <span id="clipStatus">Nouveau clip</span></div>
                    <div>⏱️ <span id="totalDuration">00:00:00</span></div>
                    <div>✂️ <span id="sequenceCount">0 séquences</span></div>
                </div>
            </aside>

            <!-- Zone preview et extraction -->
            <div class="preview-zone">
                <!-- Preview vidéo -->
                <div class="video-container">
                    <video class="video-preview" id="videoPreview" preload="metadata">
                        <source src="" type="video/mp4">
                        Votre navigateur ne supporte pas la vidéo.
                    </video>
                    
                    <!-- Contrôles personnalisés élégants -->
                    <div class="custom-video-controls" id="customControls">
                        <button class="video-btn play-pause" onclick="togglePlayback()" title="Play/Pause (Espace)">
                            <span id="playIcon">▶️</span>
                        </button>
                        <button class="video-btn" onclick="seekRelative(-5)" title="Reculer 5s">
                            ⏪
                        </button>
                        <button class="video-btn" onclick="seekRelative(5)" title="Avancer 5s">
                            ⏩
                        </button>
                        <button class="video-btn" onclick="goToStart()" title="Début">
                            ⏮️
                        </button>
                        <button class="video-btn" onclick="goToEnd()" title="Fin">
                            ⏭️
                        </button>
                        
                        <div class="volume-control">
                            <button class="video-btn volume-btn" onclick="toggleMute()" title="Volume">
                                <span id="volumeIcon">🔊</span>
                            </button>
                            <input type="range" class="volume-slider" id="volumeSlider" min="0" max="1" step="0.1" value="1" oninput="setVolume(this.value)">
                        </div>
                        
                        <div class="time-info">
                            <span id="videoCurrentTime">00:00</span> / <span id="videoDuration">00:00</span>
                        </div>
                    </div>
                </div>

                <!-- Timeline d'extraction -->
                <div class="extraction-timeline">
                    <div class="extraction-controls">
                        <button class="control-btn in" onclick="setInPoint()" title="Point IN (I)">📍 IN</button>
                        <button class="control-btn out" onclick="setOutPoint()" title="Point OUT (O)">📍 OUT</button>
                        <button class="control-btn extract" onclick="extractSequence()" title="Extraire (Enter)">✂️ Extraire</button>
                        <button class="control-btn clear" onclick="clearMarkers()" title="Effacer (Esc)">🗑️ Clear</button>
                        
                        <div style="margin-left: auto; display: flex; gap: 10px; font-size: 11px; color: #888;">
                            <span>Position: <span id="currentTime">00:00:00</span></span>
                            <span>IN: <span id="inTime">--:--:--</span></span>
                            <span>OUT: <span id="outTime">--:--:--</span></span>
                            <span>Durée: <span id="durationTime">00:00:00</span></span>
                        </div>
                    </div>
                    
                    <div class="timeline-ruler" id="timelineRuler" onclick="seekToPosition(event)">
                        <div class="playhead" id="playhead"></div>
                        <div class="selection-zone" id="selectionZone"></div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Timeline horizontale -->
        <div class="horizontal-timeline">
            <div class="timeline-header">
                <div class="timeline-title">🎬 Timeline du Clip</div>
                <div class="timeline-controls">
                    <button class="timeline-btn preview" onclick="previewTimeline()" title="Preview complète">
                        ▶️ Preview
                    </button>
                    <button class="timeline-btn save" onclick="saveTimeline()" title="Sauvegarder">
                        💾 Sauvegarder
                    </button>
                    <button class="timeline-btn load" onclick="loadTimeline()" title="Charger">
                        📂 Charger
                    </button>
                    <button class="timeline-btn transitions" onclick="showTransitionsPanel()" title="Gérer transitions">
                        🎭 Transitions
                    </button>
                    <button class="timeline-btn export" onclick="exportTimeline()" title="Exporter">
                        🚀 Export
                    </button>
                </div>
            </div>
            
            <div class="timeline-track" id="timelineTrack" ondrop="dropSequence(event)" ondragover="allowDrop(event)">
                <div class="timeline-empty" id="timelineEmpty">
                    Glissez vos séquences ici pour créer votre timeline
                </div>
            </div>
        </div>
    </div>

    <!-- Panel transitions -->
    <div class="transitions-panel" id="transitionsPanel">
        <div class="transitions-header">
            <h3>🎭 Choisir une Transition</h3>
            <button class="close-btn" onclick="hideTransitionsPanel()">✕</button>
        </div>
        
        <div class="transitions-list" id="transitionsList"></div>
        
        <div class="duration-controls">
            <div class="duration-slider">
                <label>Durée: <span class="duration-value" id="durationValue">1.0s</span></label>
                <input type="range" id="durationSlider" min="0.1" max="3.0" step="0.1" value="1.0" oninput="updateDurationValue()">
            </div>
            <button class="apply-transition-btn" onclick="applyTransition()">
                ✨ Appliquer Transition
            </button>
        </div>
    </div>

    <!-- Modal de sortie -->
    <div class="exit-modal" id="exitModal">
        <div class="exit-modal-content">
            <h3>🚪 Quitter Clipmaker</h3>
            <p>Voulez-vous sauvegarder vos modifications avant de quitter ?</p>
            <div class="exit-modal-buttons">
                <button class="exit-btn save" onclick="saveAndExit()">💾 Sauvegarder et Quitter</button>
                <button class="exit-btn abandon" onclick="abandonAndExit()">🗑️ Quitter sans Sauvegarder</button>
                <button class="exit-btn cancel" onclick="hideExitModal()">❌ Annuler</button>
            </div>
        </div>
    </div>

    <script>
        // Variables globales
        let currentClipName = "";
        let originalClipName = "";
        let currentUser = "";
        let currentRush = null;
        let timelineSequences = [];
        let clipTransitions = [];
        let availableTransitions = [];
        let isPlaying = false;
        let currentTime = 0;
        let inPoint = null;
        let outPoint = null;
        let rushDuration = 0;
        let pixelsPerSecond = 10;
        let selectedSequenceIndex = -1;
        let selectedTransitionIndex = -1;
        let sequenceIdCounter = 1;
        let hasUnsavedChanges = false;

        // Initialisation avec paramètres URL
        document.addEventListener('DOMContentLoaded', function() {
            initializeFromURL();
            setupKeyboardControls();
            loadAvailableTransitions();
            updateDisplay();
            showNotification('🎬 SMG Studio Clipmaker prêt!');
        });

        // Récupérer les paramètres URL
        function initializeFromURL() {
            const urlParams = new URLSearchParams(window.location.search);
            currentUser = urlParams.get('user') || 'user_001';
            currentClipName = urlParams.get('project') || 'Mon Nouveau Clip';
            originalClipName = currentClipName;
            
            // Mettre à jour l'interface
            document.getElementById('clipNameInput').value = currentClipName;
            
            console.log('🎬 Clipmaker initialisé:', { user: currentUser, project: currentClipName });
        }

        // Détecter les changements
        function markAsChanged() {
            hasUnsavedChanges = true;
            updateClipStatus('modified');
        }

        // Surveiller les changements du nom
        document.getElementById('clipNameInput').addEventListener('input', function() {
            const newName = this.value.trim();
            if (newName !== currentClipName) {
                currentClipName = newName;
                markAsChanged();
            }
        });

        // Modal de sortie
        function showExitModal() {
            document.getElementById('exitModal').style.display = 'flex';
        }

        function hideExitModal() {
            document.getElementById('exitModal').style.display = 'none';
        }

        function saveAndExit() {
            saveClip();
            setTimeout(() => {
                redirectToWorkflow();
            }, 1000);
        }

        function abandonAndExit() {
            redirectToWorkflow();
        }

        function redirectToWorkflow() {
            // Rediriger vers le workflow manager avec le nouveau nom de projet
            const workflowUrl = `workflow_manager.php?user=${encodeURIComponent(currentUser)}`;
            window.location.href = workflowUrl;
        }

        // Sauvegarder le clip
        function saveClip() {
            if (!currentClipName.trim()) {
                showNotification('⚠️ Nom de clip requis');
                return;
            }

            // Simulation de sauvegarde
            const clipData = {
                name: currentClipName,
                originalName: originalClipName,
                user: currentUser,
                sequences: timelineSequences,
                transitions: clipTransitions,
                totalDuration: timelineSequences.reduce((sum, seq) => sum + seq.duration, 0),
                savedAt: new Date().toISOString()
            };

            console.log('💾 Sauvegarde clip:', clipData);
            
            // Marquer comme sauvegardé
            hasUnsavedChanges = false;
            updateClipStatus('saved');
            showNotification(`💾 Clip "${currentClipName}" sauvegardé`);

            // TODO: Envoyer au serveur
            // fetch('/api/clipmaker/save', { method: 'POST', body: JSON.stringify(clipData) })
        }

        // Contrôles clavier
        function setupKeyboardControls() {
            document.addEventListener('keydown', (e) => {
                if (e.target.tagName === 'INPUT') return;
                
                switch(e.key.toLowerCase()) {
                    case 'i':
                        e.preventDefault();
                        setInPoint();
                        break;
                    case 'o':
                        e.preventDefault();
                        setOutPoint();
                        break;
                    case ' ':
                        e.preventDefault();
                        togglePlayback();
                        break;
                    case 'enter':
                        if (inPoint !== null && outPoint !== null) {
                            e.preventDefault();
                            extractSequence();
                        }
                        break;
                    case 'escape':
                        clearMarkers();
                        break;
                    case 'arrowleft':
                        e.preventDefault();
                        seekRelative(-1);
                        break;
                    case 'arrowright':
                        e.preventDefault();
                        seekRelative(1);
                        break;
                }
            });
        }

        // Upload rush
        function uploadRush() {
            const input = document.createElement('input');
            input.type = 'file';
            input.accept = 'video/*';
            
            input.onchange = function(e) {
                const file = e.target.files[0];
                if (!file) return;
                
                if (currentRush && currentRush.url) {
                    URL.revokeObjectURL(currentRush.url);
                }
                
                currentRush = {
                    id: Date.now(),
                    name: file.name,
                    url: URL.createObjectURL(file),
                    file: file
                };
                
                loadRushToPreview();
                markAsChanged();
                showNotification(`📹 Rush "${file.name}" chargé`);
            };
            
            input.click();
        }

        function loadRushToPreview() {
            if (!currentRush) return;
            
            const video = document.getElementById('videoPreview');
            video.src = currentRush.url;
            
            video.onloadedmetadata = () => {
                rushDuration = video.duration;
                pixelsPerSecond = 600 / rushDuration;
                generateTimelineRuler();
                showRushInfo();
                updateVideoTimeDisplay();
            };
            
            video.ontimeupdate = () => {
                currentTime = video.currentTime;
                updatePlayhead();
                updateTimeDisplays();
                updateVideoTimeDisplay();
            };

            video.onplay = () => {
                isPlaying = true;
                document.getElementById('playIcon').textContent = '⏸️';
            };

            video.onpause = () => {
                isPlaying = false;
                document.getElementById('playIcon').textContent = '▶️';
            };
            
            clearMarkers();
        }

        function updateVideoTimeDisplay() {
            document.getElementById('videoCurrentTime').textContent = formatTimeShort(currentTime);
            document.getElementById('videoDuration').textContent = formatTimeShort(rushDuration);
        }

        function formatTimeShort(seconds) {
            const m = Math.floor(seconds / 60);
            const s = Math.floor(seconds % 60);
            return `${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`;
        }

        function showRushInfo() {
            document.getElementById('rushInfo').style.display = 'block';
            document.getElementById('rushName').textContent = currentRush.name;
            document.getElementById('rushDuration').textContent = `${formatTime(rushDuration)} • ${Math.round(rushDuration)}s`;
        }

        function clearRush() {
            if (currentRush && currentRush.url) {
                URL.revokeObjectURL(currentRush.url);
            }
            currentRush = null;
            document.getElementById('videoPreview').src = '';
            document.getElementById('rushInfo').style.display = 'none';
            clearMarkers();
            markAsChanged();
            showNotification('🗑️ Rush supprimé');
        }

        // Contrôles de lecture
        function togglePlayback() {
            const video = document.getElementById('videoPreview');
            const playIcon = document.getElementById('playIcon');
            
            if (!video.src || video.src === window.location.href) {
                showNotification('⚠️ Aucun rush chargé');
                return;
            }
            
            if (isPlaying) {
                video.pause();
                isPlaying = false;
                playIcon.textContent = '▶️';
            } else {
                video.play();
                isPlaying = true;
                playIcon.textContent = '⏸️';
            }
        }

        function goToStart() {
            const video = document.getElementById('videoPreview');
            if (!video.src) return;
            currentTime = 0;
            video.currentTime = 0;
        }

        function goToEnd() {
            const video = document.getElementById('videoPreview');
            if (!video.src) return;
            currentTime = rushDuration;
            video.currentTime = rushDuration;
        }

        function toggleMute() {
            const video = document.getElementById('videoPreview');
            const volumeIcon = document.getElementById('volumeIcon');
            const volumeSlider = document.getElementById('volumeSlider');
            
            if (video.muted) {
                video.muted = false;
                volumeIcon.textContent = '🔊';
                volumeSlider.value = video.volume;
            } else {
                video.muted = true;
                volumeIcon.textContent = '🔇';
            }
        }

        function setVolume(value) {
            const video = document.getElementById('videoPreview');
            const volumeIcon = document.getElementById('volumeIcon');
            
            video.volume = value;
            video.muted = false;
            
            if (value == 0) {
                volumeIcon.textContent = '🔇';
            } else if (value < 0.5) {
                volumeIcon.textContent = '🔉';
            } else {
                volumeIcon.textContent = '🔊';
            }
        }

        function seekToPosition(event) {
            if (!currentRush) return;
            
            const ruler = document.getElementById('timelineRuler');
            const rect = ruler.getBoundingClientRect();
            const clickX = event.clientX - rect.left;
            const newTime = Math.max(0, Math.min(rushDuration, clickX / pixelsPerSecond));
            
            currentTime = newTime;
            document.getElementById('videoPreview').currentTime = newTime;
        }

        function seekRelative(seconds) {
            if (!currentRush) return;
            
            const newTime = Math.max(0, Math.min(rushDuration, currentTime + seconds));
            currentTime = newTime;
            document.getElementById('videoPreview').currentTime = newTime;
        }

        // Markers et extraction
        function setInPoint() {
            if (!currentRush) {
                showNotification('⚠️ Chargez d\'abord un rush');
                return;
            }
            inPoint = currentTime;
            updateMarkers();
            updateTimeDisplays();
            markAsChanged();
            showNotification(`📍 Point IN: ${formatTime(inPoint)}`);
        }

        function setOutPoint() {
            if (!currentRush) {
                showNotification('⚠️ Chargez d\'abord un rush');
                return;
            }
            outPoint = currentTime;
            updateMarkers();
            updateTimeDisplays();
            markAsChanged();
            showNotification(`📍 Point OUT: ${formatTime(outPoint)}`);
        }

        function clearMarkers() {
            inPoint = null;
            outPoint = null;
            updateMarkers();
            updateTimeDisplays();
        }

        function extractSequence() {
            if (!currentRush) {
                showNotification('⚠️ Chargez d\'abord un rush');
                return;
            }
            
            if (inPoint === null || outPoint === null) {
                showNotification('⚠️ Définissez les points IN et OUT');
                return;
            }
            
            if (inPoint >= outPoint) {
                showNotification('⚠️ Le point OUT doit être après le point IN');
                return;
            }
            
            const duration = outPoint - inPoint;
            const sequenceName = prompt(`Nom de la séquence (${formatTime(duration)}):`, `Seq_${sequenceIdCounter}`);
            
            if (sequenceName && sequenceName.trim()) {
                const sequence = {
                    id: sequenceIdCounter++,
                    name: sequenceName.trim(),
                    rushName: currentRush.name,
                    rushId: currentRush.id,
                    inPoint: inPoint,
                    outPoint: outPoint,
                    duration: duration,
                    order: timelineSequences.length
                };
                
                createDraggableSequence(sequence);
                markAsChanged();
                clearMarkers();
                showNotification(`✂️ Séquence "${sequenceName}" créée`);
            }
        }

        function createDraggableSequence(sequence) {
            const tempElement = document.createElement('div');
            tempElement.style.cssText = `
                position: fixed;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%);
                background: linear-gradient(135deg, #4ecdc4, #45b7d1);
                color: white;
                padding: 8px 12px;
                border-radius: 4px;
                font-size: 11px;
                font-weight: 600;
                z-index: 1000;
                box-shadow: 0 4px 15px rgba(0,0,0,0.5);
                cursor: grab;
            `;
            tempElement.textContent = sequence.name;
            tempElement.draggable = true;
            
            tempElement.ondragstart = (e) => {
                e.dataTransfer.setData('sequence', JSON.stringify(sequence));
                tempElement.style.opacity = '0.5';
            };
            
            tempElement.ondragend = () => {
                tempElement.remove();
            };
            
            document.body.appendChild(tempElement);
            
            setTimeout(() => {
                tempElement.style.animation = 'bounce 0.5s ease';
            }, 100);
        }

        // Timeline horizontale
        function allowDrop(event) {
            event.preventDefault();
            const track = document.getElementById('timelineTrack');
            track.classList.add('drag-over');
        }

        function dropSequence(event) {
            event.preventDefault();
            const track = document.getElementById('timelineTrack');
            track.classList.remove('drag-over');
            
            const sequenceData = event.dataTransfer.getData('sequence');
            if (sequenceData) {
                const sequence = JSON.parse(sequenceData);
                addSequenceToTimeline(sequence);
            }
        }

        function addSequenceToTimeline(sequence) {
            timelineSequences.push(sequence);
            markAsChanged();
            
            setTimeout(() => {
                renderTimeline();
                updateDisplay();
            }, 100);
            
            showNotification(`🎬 "${sequence.name}" ajoutée à la timeline`);
        }

        function renderTimeline() {
            const track = document.getElementById('timelineTrack');
            const empty = document.getElementById('timelineEmpty');
            
            Array.from(track.children).forEach(child => {
                if (child.id !== 'timelineEmpty') {
                    child.remove();
                }
            });
            
            if (timelineSequences.length === 0) {
                empty.style.display = 'flex';
                return;
            }
            
            empty.style.display = 'none';
            
            const totalDuration = timelineSequences.reduce((sum, seq) => sum + seq.duration, 0);
            const trackWidth = track.offsetWidth - 60;
            const transitionWidth = 25;
            const totalTransitionsWidth = (timelineSequences.length - 1) * transitionWidth;
            const availableWidth = trackWidth - totalTransitionsWidth;
            
            let currentPosition = 20;
            
            timelineSequences.forEach((sequence, index) => {
                const clipWidth = Math.max(40, (sequence.duration / totalDuration) * availableWidth);
                
                const clip = document.createElement('div');
                clip.className = 'sequence-clip';
                clip.style.left = currentPosition + 'px';
                clip.style.width = clipWidth + 'px';
                clip.textContent = sequence.name;
                clip.title = `${sequence.name} (${formatTime(sequence.duration)})`;
                clip.dataset.index = index;
                
                clip.onclick = () => selectSequence(index);
                clip.ondblclick = () => previewSequence(sequence);
                clip.draggable = true;
                
                clip.ondragstart = (e) => {
                    e.dataTransfer.setData('reorder', index);
                    clip.classList.add('dragging');
                };
                
                clip.ondragend = () => {
                    clip.classList.remove('dragging');
                };
                
                clip.ondragover = (e) => {
                    e.preventDefault();
                };
                
                clip.ondrop = (e) => {
                    e.preventDefault();
                    const draggedIndex = parseInt(e.dataTransfer.getData('reorder'));
                    if (draggedIndex !== index && !isNaN(draggedIndex)) {
                        reorderSequences(draggedIndex, index);
                    }
                };
                
                track.appendChild(clip);
                
                if (index < timelineSequences.length - 1) {
                    const transition = document.createElement('div');
                    transition.className = 'transition-zone';
                    transition.style.left = (currentPosition + clipWidth + 2) + 'px';
                    transition.textContent = '+';
                    transition.title = 'Ajouter transition';
                    transition.dataset.fromIndex = index;
                    
                    const existingTransition = clipTransitions.find(t => t.fromIndex === index);
                    if (existingTransition) {
                        transition.classList.add('has-transition');
                        transition.textContent = 'T';
                        transition.title = `${existingTransition.name} (${existingTransition.duration}s)`;
                    }
                    
                    transition.onclick = (e) => {
                        e.stopPropagation();
                        selectedTransitionIndex = index;
                        showTransitionsPanel();
                    };
                    
                    track.appendChild(transition);
                }
                
                currentPosition += clipWidth + transitionWidth;
            });
        }

        function selectSequence(index) {
            selectedSequenceIndex = index;
            
            document.querySelectorAll('.sequence-clip').forEach((clip, i) => {
                clip.classList.toggle('selected', i === index);
            });
            
            showNotification(`📌 Séquence "${timelineSequences[index].name}" sélectionnée`);
        }

        function previewSequence(sequence) {
            if (!currentRush || currentRush.id !== sequence.rushId) {
                showNotification('⚠️ Rush source non disponible');
                return;
            }
            
            const video = document.getElementById('videoPreview');
            video.currentTime = sequence.inPoint;
            video.play();
            
            const stopHandler = () => {
                if (video.currentTime >= sequence.outPoint) {
                    video.pause();
                    video.removeEventListener('timeupdate', stopHandler);
                }
            };
            
            video.addEventListener('timeupdate', stopHandler);
            showNotification(`▶️ Preview "${sequence.name}"`);
        }

        function reorderSequences(fromIndex, toIndex) {
            if (fromIndex === toIndex) return;
            
            const movedSequence = timelineSequences.splice(fromIndex, 1)[0];
            timelineSequences.splice(toIndex, 0, movedSequence);
            
            timelineSequences.forEach((seq, index) => {
                seq.order = index;
            });
            
            clipTransitions.forEach(t => {
                if (t.fromIndex === fromIndex) {
                    t.fromIndex = toIndex;
                } else if (fromIndex < toIndex) {
                    if (t.fromIndex > fromIndex && t.fromIndex <= toIndex) {
                        t.fromIndex--;
                    }
                } else {
                    if (t.fromIndex >= toIndex && t.fromIndex < fromIndex) {
                        t.fromIndex++;
                    }
                }
            });
            
            markAsChanged();
            renderTimeline();
            showNotification(`🔄 Séquence réorganisée`);
        }

        // Gestion des transitions
        function loadAvailableTransitions() {
            availableTransitions = [
                { id: 1, name: 'Fondu Enchaîné', type: 'fade', description: 'Transition classique en fondu', duration_min: 0.5, duration_max: 3.0 },
                { id: 2, name: 'Coupe Franche', type: 'cut', description: 'Changement immédiat', duration_min: 0.0, duration_max: 0.0 },
                { id: 3, name: 'Glissement Gauche', type: 'slide_left', description: 'Glissement vers la gauche', duration_min: 0.5, duration_max: 2.0 },
                { id: 4, name: 'Glissement Droite', type: 'slide_right', description: 'Glissement vers la droite', duration_min: 0.5, duration_max: 2.0 },
                { id: 5, name: 'Zoom In', type: 'zoom_in', description: 'Zoom avant', duration_min: 0.8, duration_max: 2.5 },
                { id: 6, name: 'Zoom Out', type: 'zoom_out', description: 'Zoom arrière', duration_min: 0.8, duration_max: 2.5 },
                { id: 7, name: 'Rotation', type: 'rotate', description: 'Rotation 3D', duration_min: 1.0, duration_max: 3.0 },
                { id: 8, name: 'Flou Gaussien', type: 'blur', description: 'Transition avec flou', duration_min: 0.5, duration_max: 2.0 }
            ];
        }

        function showTransitionsPanel() {
            if (selectedTransitionIndex === -1) {
                showNotification('⚠️ Sélectionnez d\'abord une zone de transition');
                return;
            }
            
            document.getElementById('transitionsPanel').style.display = 'block';
            renderTransitionsList();
        }

        function hideTransitionsPanel() {
            document.getElementById('transitionsPanel').style.display = 'none';
            selectedTransitionIndex = -1;
        }

        function renderTransitionsList() {
            const list = document.getElementById('transitionsList');
            list.innerHTML = '';
            
            availableTransitions.forEach((transition, index) => {
                const item = document.createElement('div');
                item.className = 'transition-item';
                item.onclick = () => selectTransition(index);
                
                item.innerHTML = `
                    <div class="transition-name">${transition.name}</div>
                    <div class="transition-info">
                        <span>${transition.description}</span>
                        <span>${transition.duration_min}s - ${transition.duration_max}s</span>
                    </div>
                `;
                
                list.appendChild(item);
            });
        }

        function selectTransition(index) {
            document.querySelectorAll('.transition-item').forEach((item, i) => {
                item.classList.toggle('selected', i === index);
            });
            
            const transition = availableTransitions[index];
            const slider = document.getElementById('durationSlider');
            slider.min = transition.duration_min;
            slider.max = transition.duration_max;
            slider.value = Math.min(1.0, transition.duration_max);
            updateDurationValue();
        }

        function updateDurationValue() {
            const slider = document.getElementById('durationSlider');
            document.getElementById('durationValue').textContent = slider.value + 's';
        }

        function applyTransition() {
            const selectedItem = document.querySelector('.transition-item.selected');
            if (!selectedItem) {
                showNotification('⚠️ Sélectionnez une transition');
                return;
            }
            
            const transitionIndex = Array.from(document.querySelectorAll('.transition-item')).indexOf(selectedItem);
            const transition = availableTransitions[transitionIndex];
            const duration = parseFloat(document.getElementById('durationSlider').value);
            
            clipTransitions = clipTransitions.filter(t => t.fromIndex !== selectedTransitionIndex);
            
            clipTransitions.push({
                fromIndex: selectedTransitionIndex,
                id: transition.id,
                name: transition.name,
                type: transition.type,
                duration: duration
            });
            
            markAsChanged();
            renderTimeline();
            hideTransitionsPanel();
            showNotification(`🎭 Transition "${transition.name}" appliquée (${duration}s)`);
        }

        // Actions timeline
        function previewTimeline() {
            if (timelineSequences.length === 0) {
                showNotification('⚠️ Timeline vide');
                return;
            }
            
            const totalDuration = timelineSequences.reduce((sum, seq) => sum + seq.duration, 0);
            showNotification(`▶️ Preview timeline (${formatTime(totalDuration)})`);
            
            alert(`🎬 Preview Timeline\n\nDurée totale: ${formatTime(totalDuration)}\nSéquences: ${timelineSequences.length}\nTransitions: ${clipTransitions.length}\n\n(Preview complet en développement)`);
        }

        function saveTimeline() {
            saveClip();
        }

        function loadTimeline() {
            const saved = localStorage.getItem('clipmaker_timeline');
            if (!saved) {
                showNotification('⚠️ Aucune timeline sauvegardée');
                return;
            }
            
            try {
                const timelineData = JSON.parse(saved);
                timelineSequences = timelineData.sequences;
                clipTransitions = timelineData.transitions;
                
                currentClipName = timelineData.name;
                document.getElementById('clipNameInput').value = currentClipName;
                
                renderTimeline();
                updateDisplay();
                updateClipStatus('loaded');
                showNotification(`📂 Timeline "${timelineData.name}" chargée`);
            } catch (error) {
                showNotification('❌ Erreur de chargement');
            }
        }

        function exportTimeline() {
            if (!currentClipName.trim()) {
                showNotification('⚠️ Donnez un nom au clip');
                return;
            }
            
            if (timelineSequences.length === 0) {
                showNotification('⚠️ Timeline vide');
                return;
            }
            
            const exportData = {
                clipName: currentClipName,
                user: currentUser,
                sequences: timelineSequences,
                transitions: clipTransitions,
                totalDuration: timelineSequences.reduce((sum, seq) => sum + seq.duration, 0),
                exportedAt: new Date().toISOString()
            };
            
            console.log('🚀 Export Data:', exportData);
            
            showNotification(`🚀 Export "${currentClipName}" démarré...`);
            
            setTimeout(() => {
                markAsChanged();
                showNotification('✅ Export terminé! Prêt pour post-production audio.');
            }, 2000);
        }

        // Utilitaires
        function formatTime(seconds) {
            const h = Math.floor(seconds / 3600);
            const m = Math.floor((seconds % 3600) / 60);
            const s = Math.floor(seconds % 60);
            return `${h.toString().padStart(2, '0')}:${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`;
        }

        function updateMarkers() {
            const ruler = document.getElementById('timelineRuler');
            
            ruler.querySelectorAll('.marker').forEach(m => m.remove());
            
            if (inPoint !== null) {
                const inMarker = document.createElement('div');
                inMarker.className = 'marker in';
                inMarker.style.left = (inPoint * pixelsPerSecond) + 'px';
                ruler.appendChild(inMarker);
            }
            
            if (outPoint !== null) {
                const outMarker = document.createElement('div');
                outMarker.className = 'marker out';
                outMarker.style.left = (outPoint * pixelsPerSecond) + 'px';
                ruler.appendChild(outMarker);
            }
            
            updateSelectionZone();
        }

        function updateSelectionZone() {
            const zone = document.getElementById('selectionZone');
            
            if (inPoint !== null && outPoint !== null) {
                zone.style.display = 'block';
                zone.style.left = (inPoint * pixelsPerSecond) + 'px';