<?php
/**
 * ================================================================
 * ⚙️ CLIPMAKER SETTINGS - MODULE CONFIGURATION
 * ================================================================
 * 
 * 🎯 FONCTIONNALITÉS :
 * - Configuration FFmpeg et paths
 * - Paramètres export par défaut
 * - Gestion stockage et quotas
 * - Configuration IA/Services
 * - Préférences utilisateur
 * - Backup/Restore settings
 * 
 * 🔥 BY PAPY 77 - CONTROL CENTER !
 * ================================================================
 */

require_once '../auth_functions.php';
requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

$action = $_GET['action'] ?? 'dashboard';
$section = $_GET['section'] ?? 'general';

// Configuration par défaut
$default_settings = [
    // Général
    'clipmaker_default_resolution' => '1920x1080',
    'clipmaker_default_fps' => '30',
    'clipmaker_default_quality' => 'medium',
    'clipmaker_default_format' => 'mp4',
    
    // Stockage
    'clipmaker_storage_path' => '/uploads/clipmaker/',
    'clipmaker_temp_path' => '/tmp/clipmaker/',
    'clipmaker_max_file_size' => '524288000', // 500MB
    'clipmaker_max_project_duration' => '3600', // 1 heure
    'clipmaker_auto_cleanup_days' => '30',
    
    // FFmpeg
    'clipmaker_ffmpeg_path' => 'ffmpeg',
    'clipmaker_ffprobe_path' => 'ffprobe',
    'clipmaker_ffmpeg_threads' => '4',
    'clipmaker_max_concurrent_exports' => '2',
    
    // Export
    'clipmaker_export_quality_high' => '{"crf": 18, "preset": "medium"}',
    'clipmaker_export_quality_medium' => '{"crf": 23, "preset": "fast"}',
    'clipmaker_export_quality_low' => '{"crf": 28, "preset": "faster"}',
    
    // Features
    'clipmaker_ai_enabled' => '1',
    'clipmaker_transitions_enabled' => '1',
    'clipmaker_templates_enabled' => '1',
    'clipmaker_collaboration_enabled' => '0',
    
    // Formats autorisés
    'clipmaker_allowed_formats' => '["mp4","webm","mov","avi","mkv","jpg","jpeg","png","webp","mp3","wav","ogg"]',
    
    // Interface
    'clipmaker_theme' => 'default',
    'clipmaker_auto_save' => '1',
    'clipmaker_auto_save_interval' => '30', // secondes
    'clipmaker_show_tooltips' => '1'
];

// Traitement sauvegarde settings
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'save_settings') {
    header('Content-Type: application/json');
    
    try {
        $section = $_POST['section'] ?? 'general';
        $settings = $_POST['settings'] ?? [];
        
        if (empty($settings)) {
            throw new Exception("Aucun paramètre à sauvegarder");
        }
        
        // Validation selon section
        $validated_settings = validateSettings($section, $settings);
        
        // Sauvegarde en base
        foreach ($validated_settings as $key => $value) {
            $stmt = $pdo->prepare("
                INSERT INTO configuration (config_key, config_value) 
                VALUES (?, ?) 
                ON DUPLICATE KEY UPDATE config_value = VALUES(config_value)
            ");
            $stmt->execute([$key, $value]);
        }
        
        echo json_encode([
            'success' => true,
            'message' => 'Paramètres sauvegardés avec succès',
            'section' => $section
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Test FFmpeg
if ($action === 'test_ffmpeg') {
    header('Content-Type: application/json');
    
    try {
        $ffmpeg_path = $_POST['ffmpeg_path'] ?? 'ffmpeg';
        
        // Test commande FFmpeg
        $command = escapeshellcmd($ffmpeg_path) . ' -version 2>&1';
        $output = shell_exec($command);
        
        if (strpos($output, 'ffmpeg version') !== false) {
            // Parse version
            preg_match('/ffmpeg version ([^\s]+)/', $output, $matches);
            $version = $matches[1] ?? 'inconnue';
            
            echo json_encode([
                'success' => true,
                'message' => 'FFmpeg détecté avec succès',
                'version' => $version,
                'output' => substr($output, 0, 500)
            ]);
        } else {
            throw new Exception("FFmpeg introuvable ou non fonctionnel");
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Nettoyage fichiers temporaires
if ($action === 'cleanup_temp') {
    header('Content-Type: application/json');
    
    try {
        $temp_path = getCurrentSetting('clipmaker_temp_path', '/tmp/clipmaker/');
        $full_temp_path = $temp_path;
        
        $deleted_files = 0;
        $freed_space = 0;
        
        if (is_dir($full_temp_path)) {
            $files = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($full_temp_path, RecursiveDirectoryIterator::SKIP_DOTS),
                RecursiveIteratorIterator::CHILD_FIRST
            );
            
            foreach ($files as $fileinfo) {
                if ($fileinfo->isFile()) {
                    $freed_space += $fileinfo->getSize();
                    unlink($fileinfo->getRealPath());
                    $deleted_files++;
                } elseif ($fileinfo->isDir()) {
                    rmdir($fileinfo->getRealPath());
                }
            }
        }
        
        echo json_encode([
            'success' => true,
            'message' => "Nettoyage terminé",
            'deleted_files' => $deleted_files,
            'freed_space' => formatFileSize($freed_space)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Export configuration
if ($action === 'export_config') {
    try {
        $stmt = $pdo->query("SELECT * FROM configuration WHERE config_key LIKE 'clipmaker_%'");
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        $config_export = [
            'export_date' => date('Y-m-d H:i:s'),
            'user_id' => $user['id'],
            'user_name' => $user['name'],
            'settings' => $settings
        ];
        
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="clipmaker_config_' . date('Y-m-d') . '.json"');
        echo json_encode($config_export, JSON_PRETTY_PRINT);
        exit;
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
        exit;
    }
}

// Import configuration
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'import_config') {
    header('Content-Type: application/json');
    
    try {
        if (!isset($_FILES['config_file'])) {
            throw new Exception("Aucun fichier sélectionné");
        }
        
        $file_content = file_get_contents($_FILES['config_file']['tmp_name']);
        $config_data = json_decode($file_content, true);
        
        if (!$config_data || !isset($config_data['settings'])) {
            throw new Exception("Fichier de configuration invalide");
        }
        
        $imported_count = 0;
        foreach ($config_data['settings'] as $key => $value) {
            if (strpos($key, 'clipmaker_') === 0) {
                $stmt = $pdo->prepare("
                    INSERT INTO configuration (config_key, config_value) 
                    VALUES (?, ?) 
                    ON DUPLICATE KEY UPDATE config_value = VALUES(config_value)
                ");
                $stmt->execute([$key, $value]);
                $imported_count++;
            }
        }
        
        echo json_encode([
            'success' => true,
            'message' => "$imported_count paramètres importés avec succès"
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Récupération settings actuels
try {
    $stmt = $pdo->query("SELECT config_key, config_value FROM configuration WHERE config_key LIKE 'clipmaker_%'");
    $current_settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    // Merge avec defaults
    $settings = array_merge($default_settings, $current_settings);
    
    // Stats système
    $system_stats = getSystemStats();
    
} catch (PDOException $e) {
    error_log("Erreur récupération settings: " . $e->getMessage());
    $settings = $default_settings;
    $system_stats = [];
}

// Fonctions utilitaires
function validateSettings($section, $input_settings) {
    $validated = [];
    
    switch ($section) {
        case 'general':
            $allowed_keys = [
                'clipmaker_default_resolution',
                'clipmaker_default_fps', 
                'clipmaker_default_quality',
                'clipmaker_default_format'
            ];
            
            foreach ($allowed_keys as $key) {
                if (isset($input_settings[$key])) {
                    $validated[$key] = sanitizeSettingValue($key, $input_settings[$key]);
                }
            }
            break;
            
        case 'storage':
            $allowed_keys = [
                'clipmaker_storage_path',
                'clipmaker_temp_path',
                'clipmaker_max_file_size',
                'clipmaker_max_project_duration',
                'clipmaker_auto_cleanup_days'
            ];
            
            foreach ($allowed_keys as $key) {
                if (isset($input_settings[$key])) {
                    $validated[$key] = sanitizeSettingValue($key, $input_settings[$key]);
                }
            }
            break;
            
        case 'ffmpeg':
            $allowed_keys = [
                'clipmaker_ffmpeg_path',
                'clipmaker_ffprobe_path',
                'clipmaker_ffmpeg_threads',
                'clipmaker_max_concurrent_exports'
            ];
            
            foreach ($allowed_keys as $key) {
                if (isset($input_settings[$key])) {
                    $validated[$key] = sanitizeSettingValue($key, $input_settings[$key]);
                }
            }
            break;
            
        case 'features':
            $allowed_keys = [
                'clipmaker_ai_enabled',
                'clipmaker_transitions_enabled',
                'clipmaker_templates_enabled',
                'clipmaker_collaboration_enabled'
            ];
            
            foreach ($allowed_keys as $key) {
                if (isset($input_settings[$key])) {
                    $validated[$key] = $input_settings[$key] ? '1' : '0';
                }
            }
            break;
    }
    
    return $validated;
}

function sanitizeSettingValue($key, $value) {
    switch ($key) {
        case 'clipmaker_default_fps':
        case 'clipmaker_ffmpeg_threads':
        case 'clipmaker_max_concurrent_exports':
        case 'clipmaker_auto_cleanup_days':
            return (string)max(1, intval($value));
            
        case 'clipmaker_max_file_size':
        case 'clipmaker_max_project_duration':
            return (string)max(0, intval($value));
            
        case 'clipmaker_ffmpeg_path':
        case 'clipmaker_ffprobe_path':
            return trim(strip_tags($value));
            
        case 'clipmaker_storage_path':
        case 'clipmaker_temp_path':
            $path = trim(strip_tags($value));
            return rtrim($path, '/') . '/';
            
        default:
            return trim(strip_tags($value));
    }
}

function getCurrentSetting($key, $default = null) {
    global $settings;
    return $settings[$key] ?? $default;
}

function getSystemStats() {
    $stats = [];
    
    try {
        // Espace disque
        $storage_path = getCurrentSetting('clipmaker_storage_path', '/uploads/clipmaker/');
        $full_path = $_SERVER['DOCUMENT_ROOT'] . $storage_path;
        
        if (is_dir($full_path)) {
            $stats['storage_used'] = getDirSize($full_path);
            $stats['storage_free'] = disk_free_space($full_path);
            $stats['storage_total'] = disk_total_space($full_path);
        }
        
        // Temp path
        $temp_path = getCurrentSetting('clipmaker_temp_path', '/tmp/clipmaker/');
        if (is_dir($temp_path)) {
            $stats['temp_used'] = getDirSize($temp_path);
        }
        
        // Infos système
        $stats['php_version'] = PHP_VERSION;
        $stats['memory_limit'] = ini_get('memory_limit');
        $stats['upload_max_filesize'] = ini_get('upload_max_filesize');
        $stats['post_max_size'] = ini_get('post_max_size');
        
        // FFmpeg version
        $ffmpeg_path = getCurrentSetting('clipmaker_ffmpeg_path', 'ffmpeg');
        $ffmpeg_output = shell_exec($ffmpeg_path . ' -version 2>&1');
        if ($ffmpeg_output && strpos($ffmpeg_output, 'ffmpeg version') !== false) {
            preg_match('/ffmpeg version ([^\s]+)/', $ffmpeg_output, $matches);
            $stats['ffmpeg_version'] = $matches[1] ?? 'détecté';
            $stats['ffmpeg_available'] = true;
        } else {
            $stats['ffmpeg_available'] = false;
        }
        
    } catch (Exception $e) {
        error_log("Erreur stats système: " . $e->getMessage());
    }
    
    return $stats;
}

function getDirSize($directory) {
    $size = 0;
    if (is_dir($directory)) {
        foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS)) as $file) {
            $size += $file->getSize();
        }
    }
    return $size;
}

function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    }
    return $bytes . ' B';
}

?>
<div class="settings-module">
    <!-- ================================================================
     * ⚙️ HEADER MODULE SETTINGS
     * ================================================================ -->
    <div class="module-header">
        <div class="header-content">
            <h2>⚙️ Configuration Clipmaker</h2>
            <p>Paramétrez votre studio vidéo selon vos besoins</p>
        </div>
        <div class="header-actions">
            <button class="btn btn-secondary" onclick="exportConfig()">
                <span>📤</span>
                <span>Exporter Config</span>
            </button>
            <button class="btn btn-primary" onclick="showImportModal()">
                <span>📥</span>
                <span>Importer Config</span>
            </button>
        </div>
    </div>

    <!-- ================================================================
     * 📊 SYSTÈME STATUS
     * ================================================================ -->
    <div class="system-status">
        <h3>📊 État du Système</h3>
        
        <div class="status-grid">
            <div class="status-card">
                <div class="status-icon">💾</div>
                <div class="status-info">
                    <div class="status-label">Stockage utilisé</div>
                    <div class="status-value">
                        <?php echo isset($system_stats['storage_used']) ? formatFileSize($system_stats['storage_used']) : 'N/A'; ?>
                    </div>
                    <?php if (isset($system_stats['storage_total'])): ?>
                        <div class="status-detail">
                            / <?php echo formatFileSize($system_stats['storage_total']); ?> total
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="status-card">
                <div class="status-icon">🔧</div>
                <div class="status-info">
                    <div class="status-label">FFmpeg</div>
                    <div class="status-value <?php echo ($system_stats['ffmpeg_available'] ?? false) ? 'success' : 'error'; ?>">
                        <?php echo ($system_stats['ffmpeg_available'] ?? false) ? '✅ Disponible' : '❌ Indisponible'; ?>
                    </div>
                    <?php if (isset($system_stats['ffmpeg_version'])): ?>
                        <div class="status-detail">v<?php echo $system_stats['ffmpeg_version']; ?></div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="status-card">
                <div class="status-icon">🐘</div>
                <div class="status-info">
                    <div class="status-label">PHP</div>
                    <div class="status-value"><?php echo $system_stats['php_version'] ?? 'N/A'; ?></div>
                    <div class="status-detail">
                        Mémoire: <?php echo $system_stats['memory_limit'] ?? 'N/A'; ?>
                    </div>
                </div>
            </div>
            
            <div class="status-card">
                <div class="status-icon">🗂️</div>
                <div class="status-info">
                    <div class="status-label">Fichiers temp</div>
                    <div class="status-value">
                        <?php echo isset($system_stats['temp_used']) ? formatFileSize($system_stats['temp_used']) : '0 B'; ?>
                    </div>
                    <button class="btn btn-sm btn-warning" onclick="cleanupTempFiles()">
                        🧹 Nettoyer
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- ================================================================
     * 🎛️ NAVIGATION SECTIONS
     * ================================================================ -->
    <div class="settings-nav">
        <button class="nav-btn <?php echo $section === 'general' ? 'active' : ''; ?>" 
                onclick="switchSection('general')">
            <span>🎯</span>
            <span>Général</span>
        </button>
        
        <button class="nav-btn <?php echo $section === 'storage' ? 'active' : ''; ?>" 
                onclick="switchSection('storage')">
            <span>💾</span>
            <span>Stockage</span>
        </button>
        
        <button class="nav-btn <?php echo $section === 'ffmpeg' ? 'active' : ''; ?>" 
                onclick="switchSection('ffmpeg')">
            <span>🔧</span>
            <span>FFmpeg</span>
        </button>
        
        <button class="nav-btn <?php echo $section === 'export' ? 'active' : ''; ?>" 
                onclick="switchSection('export')">
            <span>🎥</span>
            <span>Export</span>
        </button>
        
        <button class="nav-btn <?php echo $section === 'features' ? 'active' : ''; ?>" 
                onclick="switchSection('features')">
            <span>✨</span>
            <span>Fonctionnalités</span>
        </button>
        
        <button class="nav-btn <?php echo $section === 'interface' ? 'active' : ''; ?>" 
                onclick="switchSection('interface')">
            <span>🎨</span>
            <span>Interface</span>
        </button>
    </div>

    <!-- ================================================================
     * 📋 SECTIONS DE CONFIGURATION
     * ================================================================ -->
    <div class="settings-content">
        
        <!-- Section Général -->
        <div id="section-general" class="settings-section <?php echo $section === 'general' ? 'active' : ''; ?>">
            <div class="section-header">
                <h3>🎯 Paramètres Généraux</h3>
                <p>Configuration par défaut des nouveaux projets</p>
            </div>
            
            <form class="settings-form" data-section="general">
                <div class="form-grid">
                    <div class="form-group">
                        <label for="default_resolution">Résolution par défaut</label>
                        <select id="default_resolution" name="clipmaker_default_resolution">
                            <option value="1920x1080" <?php echo $settings['clipmaker_default_resolution'] === '1920x1080' ? 'selected' : ''; ?>>1920x1080 (Full HD)</option>
                            <option value="1280x720" <?php echo $settings['clipmaker_default_resolution'] === '1280x720' ? 'selected' : ''; ?>>1280x720 (HD)</option>
                            <option value="1080x1080" <?php echo $settings['clipmaker_default_resolution'] === '1080x1080' ? 'selected' : ''; ?>>1080x1080 (Carré)</option>
                            <option value="1080x1920" <?php echo $settings['clipmaker_default_resolution'] === '1080x1920' ? 'selected' : ''; ?>>1080x1920 (Portrait)</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="default_fps">FPS par défaut</label>
                        <select id="default_fps" name="clipmaker_default_fps">
                            <option value="24" <?php echo $settings['clipmaker_default_fps'] === '24' ? 'selected' : ''; ?>>24 FPS</option>
                            <option value="25" <?php echo $settings['clipmaker_default_fps'] === '25' ? 'selected' : ''; ?>>25 FPS</option>
                            <option value="30" <?php echo $settings['clipmaker_default_fps'] === '30' ? 'selected' : ''; ?>>30 FPS</option>
                            <option value="60" <?php echo $settings['clipmaker_default_fps'] === '60' ? 'selected' : ''; ?>>60 FPS</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="default_quality">Qualité par défaut</label>
                        <select id="default_quality" name="clipmaker_default_quality">
                            <option value="high" <?php echo $settings['clipmaker_default_quality'] === 'high' ? 'selected' : ''; ?>>Haute qualité</option>
                            <option value="medium" <?php echo $settings['clipmaker_default_quality'] === 'medium' ? 'selected' : ''; ?>>Qualité standard</option>
                            <option value="low" <?php echo $settings['clipmaker_default_quality'] === 'low' ? 'selected' : ''; ?>>Qualité web</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="default_format">Format par défaut</label>
                        <select id="default_format" name="clipmaker_default_format">
                            <option value="mp4" <?php echo $settings['clipmaker_default_format'] === 'mp4' ? 'selected' : ''; ?>>MP4</option>
                            <option value="webm" <?php echo $settings['clipmaker_default_format'] === 'webm' ? 'selected' : ''; ?>>WebM</option>
                            <option value="mov" <?php echo $settings['clipmaker_default_format'] === 'mov' ? 'selected' : ''; ?>>MOV</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">💾 Sauvegarder</button>
                </div>
            </form>
        </div>

        <!-- Section Stockage -->
        <div id="section-storage" class="settings-section <?php echo $section === 'storage' ? 'active' : ''; ?>">
            <div class="section-header">
                <h3>💾 Gestion du Stockage</h3>
                <p>Configuration des chemins et limites de stockage</p>
            </div>
            
            <form class="settings-form" data-section="storage">
                <div class="form-grid">
                    <div class="form-group">
                        <label for="storage_path">Dossier de stockage</label>
                        <input type="text" id="storage_path" name="clipmaker_storage_path" 
                               value="<?php echo htmlspecialchars($settings['clipmaker_storage_path']); ?>"
                               placeholder="/uploads/clipmaker/">
                        <small>Chemin relatif depuis la racine web</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="temp_path">Dossier temporaire</label>
                        <input type="text" id="temp_path" name="clipmaker_temp_path" 
                               value="<?php echo htmlspecialchars($settings['clipmaker_temp_path']); ?>"
                               placeholder="/tmp/clipmaker/">
                        <small>Chemin pour fichiers temporaires</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="max_file_size">Taille max par fichier (bytes)</label>
                        <input type="number" id="max_file_size" name="clipmaker_max_file_size" 
                               value="<?php echo $settings['clipmaker_max_file_size']; ?>"
                               min="1048576">
                        <small>Actuellement: <?php echo formatFileSize($settings['clipmaker_max_file_size']); ?></small>
                    </div>
                    
                    <div class="form-group">
                        <label for="max_duration">Durée max projet (secondes)</label>
                        <input type="number" id="max_duration" name="clipmaker_max_project_duration" 
                               value="<?php echo $settings['clipmaker_max_project_duration']; ?>"
                               min="60">
                        <small>Actuellement: <?php echo gmdate("H:i:s", $settings['clipmaker_max_project_duration']); ?></small>
                    </div>
                    
                    <div class="form-group">
                        <label for="cleanup_days">Auto-nettoyage (jours)</label>
                        <input type="number" id="cleanup_days" name="clipmaker_auto_cleanup_days" 
                               value="<?php echo $settings['clipmaker_auto_cleanup_days']; ?>"
                               min="1">
                        <small>Suppression automatique des fichiers temporaires</small>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">💾 Sauvegarder</button>
                </div>
            </form>
        </div>

        <!-- Section FFmpeg -->
        <div id="section-ffmpeg" class="settings-section <?php echo $section === 'ffmpeg' ? 'active' : ''; ?>">
            <div class="section-header">
                <h3>🔧 Configuration FFmpeg</h3>
                <p>Paramètres de traitement vidéo et audio</p>
            </div>
            
            <form class="settings-form" data-section="ffmpeg">
                <div class="form-grid">
                    <div class="form-group">
                        <label for="ffmpeg_path">Chemin FFmpeg</label>
                        <div class="input-group">
                            <input type="text" id="ffmpeg_path" name="clipmaker_ffmpeg_path" 
                                   value="<?php echo htmlspecialchars($settings['clipmaker_ffmpeg_path']); ?>"
                                   placeholder="ffmpeg">
                            <button type="button" class="btn btn-secondary" onclick="testFFmpeg()">
                                🧪 Tester
                            </button>
                        </div>
                        <small>Commande ou chemin complet vers FFmpeg</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="ffprobe_path">Chemin FFprobe</label>
                        <input type="text" id="ffprobe_path" name="clipmaker_ffprobe_path" 
                               value="<?php echo htmlspecialchars($settings['clipmaker_ffprobe_path']); ?>"
                               placeholder="ffprobe">
                        <small>Commande ou chemin complet vers FFprobe</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="ffmpeg_threads">Nombre de threads</label>
                        <input type="number" id="ffmpeg_threads" name="clipmaker_ffmpeg_threads" 
                               value="<?php echo $settings['clipmaker_ffmpeg_threads']; ?>"
                               min="1" max="16">
                        <small>Threads CPU utilisés par FFmpeg</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="max_concurrent">Exports simultanés max</label>
                        <input type="number" id="max_concurrent" name="clipmaker_max_concurrent_exports" 
                               value="<?php echo $settings['clipmaker_max_concurrent_exports']; ?>"
                               min="1" max="10">
                        <small>Nombre d'exports en parallèle</small>
                    </div>
                </div>
                
                <div id="ffmpeg_test_result" class="test-result" style="display: none;"></div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">💾 Sauvegarder</button>
                </div>
            </form>
        </div>

        <!-- Section Export -->
        <div id="section-export" class="settings-section <?php echo $section === 'export' ? 'active' : ''; ?>">
            <div class="section-header">
                <h3>🎥 Paramètres Export</h3>
                <p>Configuration des presets de qualité</p>
            </div>
            
            <div class="presets-grid">
                <div class="preset-card">
                    <h4>🏆 Haute Qualité</h4>
                    <div class="preset-info">
                        <div class="preset-param">
                            <span>CRF:</span>
                            <span><?php echo json_decode($settings['clipmaker_export_quality_high'], true)['crf'] ?? 18; ?></span>
                        </div>
                        <div class="preset-param">
                            <span>Preset:</span>
                            <span><?php echo json_decode($settings['clipmaker_export_quality_high'], true)['preset'] ?? 'medium'; ?></span>
                        </div>
                    </div>
                    <p>Meilleure qualité, fichiers plus lourds</p>
                </div>
                
                <div class="preset-card">
                    <h4>⚡ Qualité Standard</h4>
                    <div class="preset-info">
                        <div class="preset-param">
                            <span>CRF:</span>
                            <span><?php echo json_decode($settings['clipmaker_export_quality_medium'], true)['crf'] ?? 23; ?></span>
                        </div>
                        <div class="preset-param">
                            <span>Preset:</span>
                            <span><?php echo json_decode($settings['clipmaker_export_quality_medium'], true)['preset'] ?? 'fast'; ?></span>
                        </div>
                    </div>
                    <p>Bon compromis qualité/taille</p>
                </div>
                
                <div class="preset-card">
                    <h4>🌐 Qualité Web</h4>
                    <div class="preset-info">
                        <div class="preset-param">
                            <span>CRF:</span>
                            <span><?php echo json_decode($settings['clipmaker_export_quality_low'], true)['crf'] ?? 28; ?></span>
                        </div>
                        <div class="preset-param">
                            <span>Preset:</span>
                            <span><?php echo json_decode($settings['clipmaker_export_quality_low'], true)['preset'] ?? 'faster'; ?></span>
                        </div>
                    </div>
                    <p>Optimisé web, fichiers légers</p>
                </div>
            </div>
            
            <div class="export-info">
                <h4>ℹ️ Informations CRF</h4>
                <ul>
                    <li><strong>CRF 18-22:</strong> Excellente qualité, fichiers volumineux</li>
                    <li><strong>CRF 23-25:</strong> Très bonne qualité, taille raisonnable</li>
                    <li><strong>CRF 26-30:</strong> Bonne qualité, fichiers compacts</li>
                </ul>
            </div>
        </div>

        <!-- Section Fonctionnalités -->
        <div id="section-features" class="settings-section <?php echo $section === 'features' ? 'active' : ''; ?>">
            <div class="section-header">
                <h3>✨ Fonctionnalités</h3>
                <p>Activation/désactivation des modules</p>
            </div>
            
            <form class="settings-form" data-section="features">
                <div class="features-grid">
                    <div class="feature-card">
                        <div class="feature-header">
                            <div class="feature-icon">🤖</div>
                            <div class="feature-info">
                                <h4>Intelligence Artificielle</h4>
                                <p>Fonctionnalités IA pour amélioration automatique</p>
                            </div>
                            <div class="feature-toggle">
                                <input type="checkbox" id="ai_enabled" name="clipmaker_ai_enabled" 
                                       <?php echo $settings['clipmaker_ai_enabled'] === '1' ? 'checked' : ''; ?>>
                                <label for="ai_enabled" class="toggle-switch"></label>
                            </div>
                        </div>
                    </div>
                    
                    <div class="feature-card">
                        <div class="feature-header">
                            <div class="feature-icon">🎬</div>
                            <div class="feature-info">
                                <h4>Transitions</h4>
                                <p>Effets de transition entre clips</p>
                            </div>
                            <div class="feature-toggle">
                                <input type="checkbox" id="transitions_enabled" name="clipmaker_transitions_enabled" 
                                       <?php echo $settings['clipmaker_transitions_enabled'] === '1' ? 'checked' : ''; ?>>
                                <label for="transitions_enabled" class="toggle-switch"></label>
                            </div>
                        </div>
                    </div>
                    
                    <div class="feature-card">
                        <div class="feature-header">
                            <div class="feature-icon">🎨</div>
                            <div class="feature-info">
                                <h4>Templates</h4>
                                <p>Modèles prêts pour création rapide</p>
                            </div>
                            <div class="feature-toggle">
                                <input type="checkbox" id="templates_enabled" name="clipmaker_templates_enabled" 
                                       <?php echo $settings['clipmaker_templates_enabled'] === '1' ? 'checked' : ''; ?>>
                                <label for="templates_enabled" class="toggle-switch"></label>
                            </div>
                        </div>
                    </div>
                    
                    <div class="feature-card">
                        <div class="feature-header">
                            <div class="feature-icon">👥</div>
                            <div class="feature-info">
                                <h4>Collaboration</h4>
                                <p>Partage et travail en équipe</p>
                            </div>
                            <div class="feature-toggle">
                                <input type="checkbox" id="collaboration_enabled" name="clipmaker_collaboration_enabled" 
                                       <?php echo $settings['clipmaker_collaboration_enabled'] === '1' ? 'checked' : ''; ?>>
                                <label for="collaboration_enabled" class="toggle-switch"></label>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">💾 Sauvegarder</button>
                </div>
            </form>
        </div>

        <!-- Section Interface -->
        <div id="section-interface" class="settings-section <?php echo $section === 'interface' ? 'active' : ''; ?>">
            <div class="section-header">
                <h3>🎨 Interface Utilisateur</h3>
                <p>Personnalisation de l'interface</p>
            </div>
            
            <form class="settings-form" data-section="interface">
                <div class="form-grid">
                    <div class="form-group">
                        <label for="theme">Thème</label>
                        <select id="theme" name="clipmaker_theme">
                            <option value="default" <?php echo $settings['clipmaker_theme'] === 'default' ? 'selected' : ''; ?>>Défaut</option>
                            <option value="dark" <?php echo $settings['clipmaker_theme'] === 'dark' ? 'selected' : ''; ?>>Sombre</option>
                            <option value="light" <?php echo $settings['clipmaker_theme'] === 'light' ? 'selected' : ''; ?>>Clair</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="auto_save_interval">Intervalle sauvegarde auto (sec)</label>
                        <input type="number" id="auto_save_interval" name="clipmaker_auto_save_interval" 
                               value="<?php echo $settings['clipmaker_auto_save_interval']; ?>"
                               min="10" max="300">
                    </div>
                </div>
                
                <div class="interface-options">
                    <div class="option-item">
                        <input type="checkbox" id="auto_save" name="clipmaker_auto_save" 
                               <?php echo $settings['clipmaker_auto_save'] === '1' ? 'checked' : ''; ?>>
                        <label for="auto_save">Sauvegarde automatique</label>
                    </div>
                    
                    <div class="option-item">
                        <input type="checkbox" id="show_tooltips" name="clipmaker_show_tooltips" 
                               <?php echo $settings['clipmaker_show_tooltips'] === '1' ? 'checked' : ''; ?>>
                        <label for="show_tooltips">Afficher les infobulles</label>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">💾 Sauvegarder</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- ================================================================
 * 📥 MODAL IMPORT CONFIGURATION
 * ================================================================ -->
<div id="importModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>📥 Importer Configuration</h3>
            <button class="btn-close" onclick="closeImportModal()">&times;</button>
        </div>
        
        <form id="importForm" enctype="multipart/form-data" onsubmit="importConfig(event)">
            <div class="modal-body">
                <div class="form-group">
                    <label for="config_file">Fichier de configuration</label>
                    <input type="file" id="config_file" name="config_file" accept=".json" required>
                    <small>Fichier JSON exporté depuis Clipmaker</small>
                </div>
                
                <div class="import-warning">
                    <div class="warning-icon">⚠️</div>
                    <div class="warning-text">
                        <strong>Attention:</strong> Cette opération remplacera vos paramètres actuels.
                        Il est recommandé d'exporter votre configuration actuelle avant l'import.
                    </div>
                </div>
            </div>
            
            <div class="modal-actions">
                <button type="button" class="btn btn-secondary" onclick="closeImportModal()">
                    Annuler
                </button>
                <button type="submit" class="btn btn-primary">
                    📥 Importer
                </button>
            </div>
        </form>
    </div>
</div>

<style>
/* ================================================================
 * 🎨 STYLES MODULE SETTINGS
 * ================================================================ */

.settings-module {
    padding: 0;
}

.module-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 30px;
    padding-bottom: 20px;
    border-bottom: 2px solid rgba(255, 107, 107, 0.1);
}

.header-content h2 {
    color: #333;
    margin-bottom: 5px;
    font-size: 24px;
}

.header-content p {
    color: #666;
    font-size: 14px;
}

.header-actions {
    display: flex;
    gap: 10px;
}

.header-actions .btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-weight: 600;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6b6b, #ee5a24);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 107, 107, 0.3);
}

.btn-secondary {
    background: #f8f9fa;
    color: #666;
    border: 1px solid #ddd;
}

.btn-secondary:hover {
    background: #e9ecef;
}

.btn-sm {
    padding: 4px 8px;
    font-size: 11px;
}

.btn-warning {
    background: #ffc107;
    color: #212529;
}

.system-status {
    background: rgba(255, 255, 255, 0.95);
    border-radius: 16px;
    padding: 25px;
    margin-bottom: 30px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
}

.system-status h3 {
    color: #333;
    margin-bottom: 20px;
    font-size: 18px;
}

.status-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
}

.status-card {
    background: rgba(255, 255, 255, 0.9);
    border: 2px solid rgba(255, 107, 107, 0.1);
    border-radius: 12px;
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
    transition: all 0.3s ease;
}

.status-card:hover {
    border-color: rgba(255, 107, 107, 0.3);
    transform: translateY(-2px);
}

.status-icon {
    font-size: 32px;
    opacity: 0.8;
}

.status-info {
    flex: 1;
}

.status-label {
    font-size: 12px;
    color: #666;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-bottom: 5px;
}

.status-value {
    font-size: 16px;
    font-weight: 600;
    color: #333;
    margin-bottom: 3px;
}

.status-value.success {
    color: #28a745;
}

.status-value.error {
    color: #dc3545;
}

.status-detail {
    font-size: 11px;
    color: #999;
}

.settings-nav {
    display: flex;
    gap: 10px;
    margin-bottom: 30px;
    background: rgba(255, 255, 255, 0.9);
    padding: 20px;
    border-radius: 12px;
    overflow-x: auto;
}

.nav-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    background: transparent;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s ease;
    white-space: nowrap;
    font-weight: 500;
    color: #666;
}

.nav-btn:hover {
    background: rgba(255, 107, 107, 0.1);
    border-color: rgba(255, 107, 107, 0.4);
}

.nav-btn.active {
    background: linear-gradient(135deg, #ff6b6b, #ee5a24);
    color: white;
    border-color: #ff6b6b;
}

.settings-content {
    background: rgba(255, 255, 255, 0.95);
    border-radius: 16px;
    padding: 30px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
}

.settings-section {
    display: none;
}

.settings-section.active {
    display: block;
}

.section-header {
    margin-bottom: 30px;
    padding-bottom: 20px;
    border-bottom: 1px solid rgba(255, 107, 107, 0.2);
}

.section-header h3 {
    color: #333;
    margin-bottom: 8px;
    font-size: 20px;
}

.section-header p {
    color: #666;
    font-size: 14px;
}

.form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #333;
    font-size: 14px;
}

.form-group input, .form-group select {
    width: 100%;
    padding: 10px 12px;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 8px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.form-group input:focus, .form-group select:focus {
    outline: none;
    border-color: #ff6b6b;
    box-shadow: 0 0 0 3px rgba(255, 107, 107, 0.1);
}

.form-group small {
    display: block;
    margin-top: 5px;
    font-size: 12px;
    color: #999;
}

.input-group {
    display: flex;
    gap: 10px;
}

.input-group input {
    flex: 1;
}

.test-result {
    background: rgba(255, 255, 255, 0.9);
    border-radius: 8px;
    padding: 15px;
    margin: 20px 0;
    border-left: 4px solid #28a745;
}

.test-result.error {
    border-left-color: #dc3545;
}

.presets-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.preset-card {
    background: rgba(255, 107, 107, 0.05);
    border: 1px solid rgba(255, 107, 107, 0.2);
    border-radius: 12px;
    padding: 20px;
    text-align: center;
}

.preset-card h4 {
    color: #333;
    margin-bottom: 15px;
}

.preset-info {
    display: flex;
    justify-content: space-around;
    margin-bottom: 15px;
}

.preset-param {
    display: flex;
    flex-direction: column;
    gap: 5px;
}

.preset-param span:first-child {
    font-size: 12px;
    color: #666;
    font-weight: 600;
}

.preset-param span:last-child {
    font-size: 16px;
    color: #ff6b6b;
    font-weight: 700;
}

.export-info {
    background: rgba(255, 255, 255, 0.8);
    padding: 20px;
    border-radius: 8px;
    margin-top: 30px;
}

.export-info h4 {
    color: #333;
    margin-bottom: 15px;
}

.export-info ul {
    margin: 0;
    padding-left: 20px;
}

.export-info li {
    margin-bottom: 8px;
    color: #666;
}

.features-grid {
    display: grid;
    gap: 20px;
    margin-bottom: 30px;
}

.feature-card {
    background: rgba(255, 255, 255, 0.9);
    border: 2px solid rgba(255, 107, 107, 0.1);
    border-radius: 12px;
    padding: 20px;
    transition: all 0.3s ease;
}

.feature-card:hover {
    border-color: rgba(255, 107, 107, 0.3);
}

.feature-header {
    display: flex;
    align-items: center;
    gap: 15px;
}

.feature-icon {
    font-size: 32px;
    opacity: 0.8;
}

.feature-info {
    flex: 1;
}

.feature-info h4 {
    color: #333;
    margin-bottom: 5px;
    font-size: 16px;
}

.feature-info p {
    color: #666;
    font-size: 13px;
    margin: 0;
}

.feature-toggle {
    position: relative;
}

.feature-toggle input[type="checkbox"] {
    display: none;
}

.toggle-switch {
    display: block;
    width: 50px;
    height: 26px;
    background: #ddd;
    border-radius: 13px;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
}

.toggle-switch::after {
    content: '';
    position: absolute;
    top: 2px;
    left: 2px;
    width: 22px;
    height: 22px;
    background: white;
    border-radius: 50%;
    transition: all 0.3s ease;
}

.feature-toggle input[type="checkbox"]:checked + .toggle-switch {
    background: #ff6b6b;
}

.feature-toggle input[type="checkbox"]:checked + .toggle-switch::after {
    transform: translateX(24px);
}

.interface-options {
    margin-bottom: 30px;
}

.option-item {
    display: flex;
    align-items: center;
    gap: 10px;
    margin-bottom: 15px;
}

.option-item input[type="checkbox"] {
    width: auto;
}

.form-actions {
    padding-top: 20px;
    border-top: 1px solid rgba(255, 107, 107, 0.2);
    text-align: right;
}

/* Modal styles */
.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.8);
    z-index: 1000;
    backdrop-filter: blur(5px);
}

.modal-content {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: white;
    border-radius: 16px;
    width: 90%;
    max-width: 500px;
    max-height: 90vh;
    overflow-y: auto;
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 30px;
    border-bottom: 1px solid #eee;
}

.modal-header h3 {
    margin: 0;
    color: #333;
}

.btn-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #666;
    padding: 5px;
}

.modal-body {
    padding: 30px;
}

.import-warning {
    display: flex;
    gap: 15px;
    background: rgba(255, 193, 7, 0.1);
    border: 1px solid rgba(255, 193, 7, 0.3);
    border-radius: 8px;
    padding: 15px;
    margin-top: 20px;
}

.warning-icon {
    font-size: 24px;
    color: #ffc107;
}

.warning-text {
    color: #856404;
    font-size: 14px;
}

.modal-actions {
    display: flex;
    gap: 15px;
    justify-content: flex-end;
    padding: 20px 30px;
    border-top: 1px solid #eee;
}

/* Responsive */
@media (max-width: 768px) {
    .form-grid {
        grid-template-columns: 1fr;
    }
    
    .status-grid {
        grid-template-columns: 1fr;
    }
    
    .presets-grid {
        grid-template-columns: 1fr;
    }
    
    .settings-nav {
        flex-direction: column;
    }
    
    .feature-header {
        flex-direction: column;
        text-align: center;
        gap: 10px;
    }
}
</style>

<script>
/**
 * ================================================================
 * 🎯 JAVASCRIPT MODULE SETTINGS
 * ================================================================ */

let currentSection = '<?php echo $section; ?>';

// Navigation sections
function switchSection(section) {
    // Masquer toutes les sections
    document.querySelectorAll('.settings-section').forEach(s => {
        s.classList.remove('active');
    });
    
    // Retirer active de tous les boutons nav
    document.querySelectorAll('.nav-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    
    // Afficher section sélectionnée
    document.getElementById(`section-${section}`).classList.add('active');
    document.querySelector(`[onclick="switchSection('${section}')"]`).classList.add('active');
    
    currentSection = section;
    
    // Mise à jour URL
    const url = new URL(window.location);
    url.searchParams.set('section', section);
    window.history.pushState({}, '', url);
}

// Sauvegarde settings
document.querySelectorAll('.settings-form').forEach(form => {
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        
        const section = this.dataset.section;
        const formData = new FormData(this);
        
        // Conversion checkbox en valeurs 0/1
        const checkboxes = this.querySelectorAll('input[type="checkbox"]');
        checkboxes.forEach(checkbox => {
            if (!formData.has(checkbox.name)) {
                formData.append(checkbox.name, '0');
            } else {
                formData.set(checkbox.name, '1');
            }
        });
        
        formData.append('section', section);
        
        const submitBtn = this.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '⏳ Sauvegarde...';
        submitBtn.disabled = true;
        
        fetch('?action=save_settings', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('✅ ' + data.message, 'success');
            } else {
                showNotification('❌ ' + data.error, 'error');
            }
        })
        .catch(error => {
            showNotification('❌ Erreur sauvegarde: ' + error.message, 'error');
        })
        .finally(() => {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    });
});

// Test FFmpeg
function testFFmpeg() {
    const ffmpegPath = document.getElementById('ffmpeg_path').value;
    const resultDiv = document.getElementById('ffmpeg_test_result');
    
    resultDiv.innerHTML = '⏳ Test en cours...';
    resultDiv.style.display = 'block';
    resultDiv.className = 'test-result';
    
    const formData = new FormData();
    formData.append('ffmpeg_path', ffmpegPath);
    
    fetch('?action=test_ffmpeg', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            resultDiv.innerHTML = `
                <div style="color: #28a745; font-weight: 600;">✅ ${data.message}</div>
                <div style="margin-top: 10px; font-size: 12px;">Version: ${data.version}</div>
                <details style="margin-top: 10px;">
                    <summary style="cursor: pointer; font-size: 12px;">Voir détails</summary>
                    <pre style="margin-top: 10px; font-size: 11px; background: #f8f9fa; padding: 10px; border-radius: 4px; overflow-x: auto;">${data.output}</pre>
                </details>
            `;
            resultDiv.className = 'test-result';
        } else {
            resultDiv.innerHTML = `
                <div style="color: #dc3545; font-weight: 600;">❌ ${data.error}</div>
                <div style="margin-top: 10px; font-size: 12px;">Vérifiez que FFmpeg est installé et accessible</div>
            `;
            resultDiv.className = 'test-result error';
        }
    })
    .catch(error => {
        resultDiv.innerHTML = `
            <div style="color: #dc3545; font-weight: 600;">❌ Erreur test: ${error.message}</div>
        `;
        resultDiv.className = 'test-result error';
    });
}

// Nettoyage fichiers temporaires
function cleanupTempFiles() {
    if (!confirm('Nettoyer tous les fichiers temporaires ?')) {
        return;
    }
    
    fetch('?action=cleanup_temp', {
        method: 'POST'
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification(
                `✅ ${data.message} - ${data.deleted_files} fichiers supprimés (${data.freed_space} libérés)`, 
                'success'
            );
            // Recharger stats
            setTimeout(() => window.location.reload(), 2000);
        } else {
            showNotification('❌ ' + data.error, 'error');
        }
    })
    .catch(error => {
        showNotification('❌ Erreur nettoyage: ' + error.message, 'error');
    });
}

// Export configuration
function exportConfig() {
    const link = document.createElement('a');
    link.href = '?action=export_config';
    link.download = `clipmaker_config_${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    showNotification('📤 Configuration exportée', 'success');
}

// Import configuration
function showImportModal() {
    document.getElementById('importModal').style.display = 'block';
}

function closeImportModal() {
    document.getElementById('importModal').style.display = 'none';
    document.getElementById('importForm').reset();
}

function importConfig(event) {
    event.preventDefault();
    
    const formData = new FormData(event.target);
    const submitBtn = event.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    submitBtn.innerHTML = '⏳ Import...';
    submitBtn.disabled = true;
    
    fetch('?action=import_config', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('✅ ' + data.message, 'success');
            closeImportModal();
            
            // Recharger page après délai
            setTimeout(() => {
                window.location.reload();
            }, 2000);
        } else {
            showNotification('❌ ' + data.error, 'error');
        }
    })
    .catch(error => {
        showNotification('❌ Erreur import: ' + error.message, 'error');
    })
    .finally(() => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

// Utilitaires
function showNotification(message, type) {
    // Création notification toast
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#28a745' : type === 'warning' ? '#ffc107' : '#dc3545'};
        color: ${type === 'warning' ? '#212529' : 'white'};
        padding: 15px 20px;
        border-radius: 8px;
        z-index: 10001;
        font-weight: 600;
        box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        animation: slideIn 0.3s ease;
        max-width: 400px;
        word-wrap: break-word;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 5000);
}

// Validation temps réel
document.addEventListener('DOMContentLoaded', function() {
    // Validation paths
    const pathInputs = document.querySelectorAll('input[name*="path"]');
    pathInputs.forEach(input => {
        input.addEventListener('blur', function() {
            if (this.value && !this.value.startsWith('/')) {
                this.style.borderColor = '#ffc107';
                showNotification('⚠️ Les chemins doivent commencer par /', 'warning');
            } else {
                this.style.borderColor = '';
            }
        });
    });
    
    // Validation tailles fichiers
    const sizeInputs = document.querySelectorAll('input[name*="size"], input[name*="duration"]');
    sizeInputs.forEach(input => {
        input.addEventListener('input', function() {
            const value = parseInt(this.value);
            if (isNaN(value) || value < 0) {
                this.style.borderColor = '#dc3545';
            } else {
                this.style.borderColor = '';
                
                // Affichage taille formatée
                if (this.name.includes('size')) {
                    const formatted = formatFileSize(value);
                    let helpText = this.parentNode.querySelector('small');
                    if (helpText) {
                        helpText.textContent = `Actuellement: ${formatted}`;
                    }
                } else if (this.name.includes('duration')) {
                    const formatted = formatDuration(value);
                    let helpText = this.parentNode.querySelector('small');
                    if (helpText) {
                        helpText.textContent = `Actuellement: ${formatted}`;
                    }
                }
            }
        });
    });
    
    // Auto-sauvegarde settings (draft)
    let saveTimeout;
    document.querySelectorAll('.settings-form input, .settings-form select').forEach(input => {
        input.addEventListener('change', function() {
            clearTimeout(saveTimeout);
            saveTimeout = setTimeout(() => {
                // Indication visuelle de changement non sauvé
                const form = this.closest('.settings-form');
                const saveBtn = form.querySelector('button[type="submit"]');
                if (saveBtn) {
                    saveBtn.style.background = '#ffc107';
                    saveBtn.textContent = '💾 Sauvegarder (modifié)';
                }
            }, 500);
        });
    });
    
    // Gestion raccourcis clavier
    document.addEventListener('keydown', function(e) {
        // Ctrl+S = Sauvegarde section courante
        if ((e.ctrlKey || e.metaKey) && e.key === 's') {
            e.preventDefault();
            const activeForm = document.querySelector(`#section-${currentSection} .settings-form`);
            if (activeForm) {
                activeForm.dispatchEvent(new Event('submit'));
            }
        }
        
        // Escape = Fermer modals
        if (e.key === 'Escape') {
            closeImportModal();
        }
        
        // Chiffres 1-6 = Navigation sections
        if (e.key >= '1' && e.key <= '6' && !e.ctrlKey && !e.metaKey) {
            const sections = ['general', 'storage', 'ffmpeg', 'export', 'features', 'interface'];
            const sectionIndex = parseInt(e.key) - 1;
            if (sections[sectionIndex]) {
                switchSection(sections[sectionIndex]);
            }
        }
    });
    
    // Tooltips pour raccourcis
    if (getCurrentSetting('clipmaker_show_tooltips', '1') === '1') {
        document.querySelectorAll('.nav-btn').forEach((btn, index) => {
            btn.title = `Raccourci: ${index + 1}`;
        });
        
        document.querySelectorAll('.settings-form button[type="submit"]').forEach(btn => {
            btn.title = 'Raccourci: Ctrl+S';
        });
    }
    
    console.log('⚙️ Module Settings Ready!');
    console.log('📍 Section courante:', currentSection);
});

// Fonctions utilitaires
function formatFileSize(bytes) {
    if (bytes >= 1073741824) {
        return (bytes / 1073741824).toFixed(2) + ' GB';
    } else if (bytes >= 1048576) {
        return (bytes / 1048576).toFixed(2) + ' MB';
    } else if (bytes >= 1024) {
        return (bytes / 1024).toFixed(2) + ' KB';
    }
    return bytes + ' B';
}

function formatDuration(seconds) {
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = Math.floor(seconds % 60);
    
    if (hours > 0) {
        return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    } else {
        return `${minutes}:${secs.toString().padStart(2, '0')}`;
    }
}

function getCurrentSetting(key, defaultValue) {
    // Simulation - en production, récupérer depuis PHP ou API
    const settings = <?php echo json_encode($settings); ?>;
    return settings[key] || defaultValue;
}

// Vérification périodique système
setInterval(() => {
    // Vérification espace disque si page visible
    if (!document.hidden) {
        fetch('?action=check_system_status&ajax=1')
            .then(response => response.json())
            .then(data => {
                if (data.warning) {
                    showNotification(`⚠️ ${data.warning}`, 'warning');
                }
            })
            .catch(error => {
                console.log('Erreur vérification système:', error);
            });
    }
}, 60000); // Toutes les minutes

// Styles animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    
    @keyframes pulse {
        0% { transform: scale(1); }
        50% { transform: scale(1.05); }
        100% { transform: scale(1); }
    }
    
    .settings-form button[type="submit"]:hover {
        animation: pulse 0.5s ease;
    }
    
    .nav-btn:focus {
        outline: 2px solid rgba(255, 107, 107, 0.5);
        outline-offset: 2px;
    }
    
    .toggle-switch:focus-within {
        box-shadow: 0 0 0 3px rgba(255, 107, 107, 0.3);
    }
`;
document.head.appendChild(style);

// Gestion thème (si implémenté)
function applyTheme(theme) {
    document.documentElement.setAttribute('data-theme', theme);
    localStorage.setItem('clipmaker_theme', theme);
}

// Chargement thème sauvé
const savedTheme = getCurrentSetting('clipmaker_theme', 'default');
if (savedTheme !== 'default') {
    applyTheme(savedTheme);
}
</script>