<?php
/**
 * ================================================================
 * 📁 CLIPMAKER PROJECTS - MODULE GESTION PROJETS
 * ================================================================
 * 
 * 🎯 FONCTIONNALITÉS :
 * - Liste projets avec filtres
 * - Création nouveau projet
 * - Édition propriétés projet
 * - Duplication/Templates
 * - Suppression sécurisée
 * 
 * 🔥 BY PAPY 77 - MODULE AUTONOME !
 * ================================================================
 */

require_once '../auth_functions.php';
requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

$action = $_GET['action'] ?? 'list';
$project_id = $_GET['project'] ?? null;
$message = '';
$error = '';

// Traitement des actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        switch ($action) {
            case 'create':
                $name = trim($_POST['name']);
                $description = trim($_POST['description']);
                $client_id = $_POST['client_id'] ?: null;
                $resolution_width = (int)$_POST['resolution_width'];
                $resolution_height = (int)$_POST['resolution_height'];
                $fps = (int)$_POST['fps'];
                
                if (empty($name)) {
                    throw new Exception("Le nom du projet est obligatoire");
                }
                
                $stmt = $pdo->prepare("
                    INSERT INTO clipmaker_projects 
                    (user_id, client_id, name, description, resolution_width, resolution_height, fps, timeline_data, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, '{}', NOW())
                ");
                $stmt->execute([$user['id'], $client_id, $name, $description, $resolution_width, $resolution_height, $fps]);
                
                $new_project_id = $pdo->lastInsertId();
                $message = "Projet '$name' créé avec succès !";
                
                // Redirection vers édition
                header("Location: ../clipmaker.php?module=timeline&project=$new_project_id");
                exit;
                break;
                
            case 'update':
                $name = trim($_POST['name']);
                $description = trim($_POST['description']);
                $client_id = $_POST['client_id'] ?: null;
                $status = $_POST['status'];
                
                $stmt = $pdo->prepare("
                    UPDATE clipmaker_projects 
                    SET name = ?, description = ?, client_id = ?, status = ?, updated_at = NOW() 
                    WHERE id = ? AND user_id = ?
                ");
                $stmt->execute([$name, $description, $client_id, $status, $project_id, $user['id']]);
                
                $message = "Projet mis à jour avec succès !";
                break;
                
            case 'duplicate':
                $stmt = $pdo->prepare("SELECT * FROM clipmaker_projects WHERE id = ? AND user_id = ?");
                $stmt->execute([$project_id, $user['id']]);
                $original = $stmt->fetch();
                
                if ($original) {
                    $new_name = $original['name'] . ' (Copie)';
                    $stmt = $pdo->prepare("
                        INSERT INTO clipmaker_projects 
                        (user_id, client_id, name, description, timeline_data, settings, resolution_width, resolution_height, fps, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
                    ");
                    $stmt->execute([
                        $user['id'], $original['client_id'], $new_name, $original['description'],
                        $original['timeline_data'], $original['settings'], 
                        $original['resolution_width'], $original['resolution_height'], $original['fps']
                    ]);
                    $message = "Projet dupliqué avec succès !";
                }
                break;
                
            case 'delete':
                $stmt = $pdo->prepare("DELETE FROM clipmaker_projects WHERE id = ? AND user_id = ?");
                $stmt->execute([$project_id, $user['id']]);
                $message = "Projet supprimé avec succès !";
                break;
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Récupération des projets avec filtres
$filter_status = $_GET['filter_status'] ?? 'all';
$search = $_GET['search'] ?? '';
$sort = $_GET['sort'] ?? 'updated_at';
$order = $_GET['order'] ?? 'DESC';

$where_conditions = ["user_id = ?"];
$params = [$user['id']];

if ($filter_status !== 'all') {
    $where_conditions[] = "status = ?";
    $params[] = $filter_status;
}

if (!empty($search)) {
    $where_conditions[] = "(name LIKE ? OR description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$where_clause = implode(' AND ', $where_conditions);

try {
    $stmt = $pdo->prepare("
        SELECT p.*, c.nom as client_name,
               COUNT(DISTINCT cm.id) as media_count,
               COUNT(DISTINCT ce.id) as export_count
        FROM clipmaker_projects p
        LEFT JOIN clients c ON p.client_id = c.id
        LEFT JOIN clipmaker_media cm ON p.id = cm.project_id
        LEFT JOIN clipmaker_exports ce ON p.id = ce.project_id
        WHERE $where_clause
        GROUP BY p.id
        ORDER BY p.$sort $order
    ");
    $stmt->execute($params);
    $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Clients pour formulaire
    $stmt = $pdo->prepare("SELECT id, nom FROM clients WHERE created_by = ? ORDER BY nom");
    $stmt->execute([$user['id']]);
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    $error = "Erreur lors de la récupération des projets";
    $projects = [];
    $clients = [];
}

// Si c'est une requête AJAX, retourner JSON
if (isset($_GET['ajax'])) {
    header('Content-Type: application/json');
    echo json_encode([
        'success' => empty($error),
        'message' => $message,
        'error' => $error,
        'projects' => $projects
    ]);
    exit;
}

?>
<div class="projects-module">
    <!-- ================================================================
     * 📁 HEADER MODULE PROJECTS
     * ================================================================ -->
    <div class="module-header">
        <div class="header-content">
            <h2>📁 Gestion des Projets</h2>
            <p>Créez, organisez et gérez vos projets vidéo Clipmaker</p>
        </div>
        <div class="header-actions">
            <button class="btn btn-primary" onclick="showCreateModal()">
                <span>➕</span>
                <span>Nouveau Projet</span>
            </button>
        </div>
    </div>

    <!-- Messages -->
    <?php if ($message): ?>
        <div class="alert alert-success">
            <span>✅</span>
            <span><?php echo htmlspecialchars($message); ?></span>
        </div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-error">
            <span>❌</span>
            <span><?php echo htmlspecialchars($error); ?></span>
        </div>
    <?php endif; ?>

    <!-- ================================================================
     * 🔍 FILTRES ET RECHERCHE
     * ================================================================ -->
    <div class="filters-bar">
        <div class="search-box">
            <input type="text" id="searchInput" placeholder="🔍 Rechercher un projet..." 
                   value="<?php echo htmlspecialchars($search); ?>" onkeyup="debounceSearch()">
        </div>
        
        <div class="filter-select">
            <select id="statusFilter" onchange="applyFilters()">
                <option value="all" <?php echo $filter_status === 'all' ? 'selected' : ''; ?>>Tous les statuts</option>
                <option value="draft" <?php echo $filter_status === 'draft' ? 'selected' : ''; ?>>Brouillon</option>
                <option value="editing" <?php echo $filter_status === 'editing' ? 'selected' : ''; ?>>En édition</option>
                <option value="rendering" <?php echo $filter_status === 'rendering' ? 'selected' : ''; ?>>Rendu</option>
                <option value="completed" <?php echo $filter_status === 'completed' ? 'selected' : ''; ?>>Terminé</option>
            </select>
        </div>
        
        <div class="sort-options">
            <select id="sortSelect" onchange="applyFilters()">
                <option value="updated_at" <?php echo $sort === 'updated_at' ? 'selected' : ''; ?>>Dernière modification</option>
                <option value="created_at" <?php echo $sort === 'created_at' ? 'selected' : ''; ?>>Date création</option>
                <option value="name" <?php echo $sort === 'name' ? 'selected' : ''; ?>>Nom</option>
                <option value="duration" <?php echo $sort === 'duration' ? 'selected' : ''; ?>>Durée</option>
            </select>
            
            <button class="btn-sort" onclick="toggleOrder()" title="Ordre de tri">
                <?php echo $order === 'DESC' ? '⬇️' : '⬆️'; ?>
            </button>
        </div>

        <div class="view-toggle">
            <button class="btn-view active" data-view="grid" onclick="switchView('grid')">⊞</button>
            <button class="btn-view" data-view="list" onclick="switchView('list')">☰</button>
        </div>
    </div>

    <!-- ================================================================
     * 📊 STATISTIQUES RAPIDES
     * ================================================================ -->
    <div class="stats-summary">
        <?php
        $total_projects = count($projects);
        $draft_count = count(array_filter($projects, fn($p) => $p['status'] === 'draft'));
        $editing_count = count(array_filter($projects, fn($p) => $p['status'] === 'editing'));
        $completed_count = count(array_filter($projects, fn($p) => $p['status'] === 'completed'));
        ?>
        
        <div class="stat-item">
            <span class="stat-number"><?php echo $total_projects; ?></span>
            <span class="stat-label">Total</span>
        </div>
        <div class="stat-item">
            <span class="stat-number"><?php echo $draft_count; ?></span>
            <span class="stat-label">Brouillons</span>
        </div>
        <div class="stat-item">
            <span class="stat-number"><?php echo $editing_count; ?></span>
            <span class="stat-label">En cours</span>
        </div>
        <div class="stat-item">
            <span class="stat-number"><?php echo $completed_count; ?></span>
            <span class="stat-label">Terminés</span>
        </div>
    </div>

    <!-- ================================================================
     * 📋 LISTE DES PROJETS
     * ================================================================ -->
    <div class="projects-container" id="projectsContainer">
        <?php if (empty($projects)): ?>
            <div class="empty-state">
                <div class="empty-icon">📁</div>
                <h3>Aucun projet trouvé</h3>
                <p>Créez votre premier projet vidéo pour commencer !</p>
                <button class="btn btn-primary" onclick="showCreateModal()">
                    ➕ Créer un projet
                </button>
            </div>
        <?php else: ?>
            <div class="projects-grid" id="projectsGrid">
                <?php foreach ($projects as $project): ?>
                    <div class="project-card" data-project-id="<?php echo $project['id']; ?>">
                        <div class="project-thumbnail">
                            <?php if ($project['thumbnail_url']): ?>
                                <img src="<?php echo htmlspecialchars($project['thumbnail_url']); ?>" alt="Thumbnail">
                            <?php else: ?>
                                <div class="placeholder-thumbnail">🎬</div>
                            <?php endif; ?>
                            
                            <div class="project-overlay">
                                <button class="btn-overlay" onclick="openProject(<?php echo $project['id']; ?>)" title="Ouvrir">
                                    ▶️
                                </button>
                                <button class="btn-overlay" onclick="showProjectMenu(<?php echo $project['id']; ?>, event)" title="Menu">
                                    ⋮
                                </button>
                            </div>
                        </div>
                        
                        <div class="project-info">
                            <div class="project-header">
                                <h4 class="project-name"><?php echo htmlspecialchars($project['name']); ?></h4>
                                <div class="project-status status-<?php echo $project['status']; ?>">
                                    <?php echo ucfirst($project['status']); ?>
                                </div>
                            </div>
                            
                            <div class="project-meta">
                                <div class="meta-row">
                                    <span class="meta-label">Résolution:</span>
                                    <span class="meta-value"><?php echo $project['resolution_width']; ?>x<?php echo $project['resolution_height']; ?></span>
                                </div>
                                <div class="meta-row">
                                    <span class="meta-label">Durée:</span>
                                    <span class="meta-value"><?php echo $project['duration'] ? round($project['duration'], 1) . 's' : 'N/A'; ?></span>
                                </div>
                                <div class="meta-row">
                                    <span class="meta-label">Médias:</span>
                                    <span class="meta-value"><?php echo $project['media_count']; ?></span>
                                </div>
                                <?php if ($project['client_name']): ?>
                                <div class="meta-row">
                                    <span class="meta-label">Client:</span>
                                    <span class="meta-value"><?php echo htmlspecialchars($project['client_name']); ?></span>
                                </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="project-dates">
                                <small>Modifié: <?php echo date('d/m/Y H:i', strtotime($project['updated_at'])); ?></small>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- ================================================================
 * 🆕 MODAL CRÉATION PROJET
 * ================================================================ -->
<div id="createProjectModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>🆕 Nouveau Projet</h3>
            <button class="btn-close" onclick="closeCreateModal()">&times;</button>
        </div>
        
        <form id="createProjectForm" method="POST" action="?action=create">
            <div class="form-group">
                <label for="projectName">Nom du projet *</label>
                <input type="text" id="projectName" name="name" required 
                       placeholder="Mon super projet vidéo">
            </div>
            
            <div class="form-group">
                <label for="projectDescription">Description</label>
                <textarea id="projectDescription" name="description" rows="3"
                          placeholder="Description du projet..."></textarea>
            </div>
            
            <div class="form-group">
                <label for="projectClient">Client (optionnel)</label>
                <select id="projectClient" name="client_id">
                    <option value="">Aucun client</option>
                    <?php foreach ($clients as $client): ?>
                        <option value="<?php echo $client['id']; ?>">
                            <?php echo htmlspecialchars($client['nom']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label for="resolutionWidth">Largeur</label>
                    <select id="resolutionWidth" name="resolution_width">
                        <option value="1920">1920 (HD)</option>
                        <option value="1080">1080 (Carré)</option>
                        <option value="720">720 (Mobile)</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="resolutionHeight">Hauteur</label>
                    <select id="resolutionHeight" name="resolution_height">
                        <option value="1080">1080</option>
                        <option value="1920">1920 (Portrait)</option>
                        <option value="720">720</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="fps">FPS</label>
                    <select id="fps" name="fps">
                        <option value="30">30 FPS</option>
                        <option value="25">25 FPS</option>
                        <option value="60">60 FPS</option>
                    </select>
                </div>
            </div>
            
            <div class="modal-actions">
                <button type="button" class="btn btn-secondary" onclick="closeCreateModal()">
                    Annuler
                </button>
                <button type="submit" class="btn btn-primary">
                    🚀 Créer le projet
                </button>
            </div>
        </form>
    </div>
</div>

<!-- ================================================================
 * ⚙️ MENU CONTEXTUEL PROJET
 * ================================================================ -->
<div id="projectMenu" class="context-menu">
    <div class="menu-item" onclick="openProject(window.currentProjectId)">
        <span>▶️</span>
        <span>Ouvrir</span>
    </div>
    <div class="menu-item" onclick="editProject(window.currentProjectId)">
        <span>✏️</span>
        <span>Modifier</span>
    </div>
    <div class="menu-item" onclick="duplicateProject(window.currentProjectId)">
        <span>📋</span>
        <span>Dupliquer</span>
    </div>
    <div class="menu-item" onclick="exportProject(window.currentProjectId)">
        <span>📤</span>
        <span>Exporter</span>
    </div>
    <div class="menu-separator"></div>
    <div class="menu-item danger" onclick="deleteProject(window.currentProjectId)">
        <span>🗑️</span>
        <span>Supprimer</span>
    </div>
</div>

<style>
/* ================================================================
 * 🎨 STYLES MODULE PROJECTS
 * ================================================================ */

.projects-module {
    padding: 0;
}

.module-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 30px;
    padding-bottom: 20px;
    border-bottom: 2px solid rgba(255, 107, 107, 0.1);
}

.header-content h2 {
    color: #333;
    margin-bottom: 5px;
    font-size: 24px;
}

.header-content p {
    color: #666;
    font-size: 14px;
}

.header-actions .btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border: none;
    border-radius: 10px;
    cursor: pointer;
    font-weight: 600;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6b6b, #ee5a24);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 107, 107, 0.3);
}

.btn-secondary {
    background: #f8f9fa;
    color: #666;
    border: 1px solid #ddd;
}

.btn-secondary:hover {
    background: #e9ecef;
}

.alert {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 15px;
    border-radius: 10px;
    margin-bottom: 20px;
    font-weight: 500;
}

.alert-success {
    background: rgba(40, 167, 69, 0.1);
    color: #28a745;
    border: 1px solid rgba(40, 167, 69, 0.2);
}

.alert-error {
    background: rgba(220, 53, 69, 0.1);
    color: #dc3545;
    border: 1px solid rgba(220, 53, 69, 0.2);
}

.filters-bar {
    display: flex;
    gap: 15px;
    align-items: center;
    margin-bottom: 20px;
    padding: 20px;
    background: rgba(255, 255, 255, 0.7);
    border-radius: 12px;
    backdrop-filter: blur(10px);
}

.search-box input {
    width: 300px;
    padding: 10px 15px;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 8px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.search-box input:focus {
    outline: none;
    border-color: #ff6b6b;
    box-shadow: 0 0 0 3px rgba(255, 107, 107, 0.1);
}

.filter-select select, .sort-options select {
    padding: 10px 15px;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 8px;
    background: white;
    cursor: pointer;
}

.btn-sort {
    background: #f8f9fa;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 8px;
    padding: 10px 12px;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-sort:hover {
    background: rgba(255, 107, 107, 0.1);
}

.view-toggle {
    margin-left: auto;
    display: flex;
    gap: 5px;
}

.btn-view {
    background: #f8f9fa;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 6px;
    padding: 8px 12px;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-view.active {
    background: #ff6b6b;
    color: white;
}

.stats-summary {
    display: flex;
    gap: 20px;
    margin-bottom: 30px;
    padding: 20px;
    background: rgba(255, 255, 255, 0.7);
    border-radius: 12px;
    backdrop-filter: blur(10px);
}

.stat-item {
    text-align: center;
    flex: 1;
}

.stat-number {
    display: block;
    font-size: 24px;
    font-weight: 700;
    color: #ff6b6b;
    margin-bottom: 5px;
}

.stat-label {
    font-size: 12px;
    color: #666;
    font-weight: 500;
}

.projects-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
    gap: 20px;
}

.project-card {
    background: rgba(255, 255, 255, 0.95);
    border-radius: 16px;
    overflow: hidden;
    transition: all 0.3s ease;
    border: 2px solid transparent;
    cursor: pointer;
}

.project-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 15px 40px rgba(0, 0, 0, 0.1);
    border-color: rgba(255, 107, 107, 0.3);
}

.project-thumbnail {
    position: relative;
    height: 180px;
    background: linear-gradient(135deg, rgba(255, 107, 107, 0.1), rgba(238, 90, 36, 0.1));
    overflow: hidden;
}

.project-thumbnail img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.placeholder-thumbnail {
    display: flex;
    align-items: center;
    justify-content: center;
    height: 100%;
    font-size: 48px;
    color: rgba(255, 107, 107, 0.5);
}

.project-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.7);
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 15px;
    opacity: 0;
    transition: all 0.3s ease;
}

.project-card:hover .project-overlay {
    opacity: 1;
}

.btn-overlay {
    background: rgba(255, 255, 255, 0.9);
    border: none;
    border-radius: 50%;
    width: 50px;
    height: 50px;
    cursor: pointer;
    font-size: 18px;
    transition: all 0.3s ease;
}

.btn-overlay:hover {
    background: white;
    transform: scale(1.1);
}

.project-info {
    padding: 20px;
}

.project-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 15px;
}

.project-name {
    font-size: 16px;
    font-weight: 600;
    color: #333;
    margin: 0;
    flex: 1;
    margin-right: 10px;
}

.project-status {
    padding: 4px 8px;
    border-radius: 20px;
    font-size: 10px;
    font-weight: 600;
    text-transform: uppercase;
}

.status-draft {
    background: rgba(255, 193, 7, 0.2);
    color: #ff8f00;
}

.status-editing {
    background: rgba(54, 162, 235, 0.2);
    color: #0066cc;
}

.status-rendering {
    background: rgba(255, 107, 107, 0.2);
    color: #ff6b6b;
}

.status-completed {
    background: rgba(40, 167, 69, 0.2);
    color: #28a745;
}

.project-meta {
    margin-bottom: 15px;
}

.meta-row {
    display: flex;
    justify-content: space-between;
    margin-bottom: 5px;
    font-size: 12px;
}

.meta-label {
    color: #666;
}

.meta-value {
    color: #333;
    font-weight: 500;
}

.project-dates {
    font-size: 11px;
    color: #999;
}

.empty-state {
    text-align: center;
    padding: 60px 20px;
    color: #666;
}

.empty-icon {
    font-size: 64px;
    margin-bottom: 20px;
    opacity: 0.5;
}

.empty-state h3 {
    margin-bottom: 10px;
    color: #333;
}

.empty-state p {
    margin-bottom: 30px;
}

/* Modal styles */
.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.7);
    z-index: 1000;
    backdrop-filter: blur(5px);
}

.modal-content {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: white;
    border-radius: 16px;
    width: 90%;
    max-width: 500px;
    max-height: 90vh;
    overflow-y: auto;
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 30px;
    border-bottom: 1px solid #eee;
}

.modal-header h3 {
    margin: 0;
    color: #333;
}

.btn-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #666;
    padding: 5px;
}

.modal form {
    padding: 30px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #333;
}

.form-group input, .form-group textarea, .form-group select {
    width: 100%;
    padding: 12px;
    border: 2px solid rgba(255, 107, 107, 0.2);
    border-radius: 8px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.form-group input:focus, .form-group textarea:focus, .form-group select:focus {
    outline: none;
    border-color: #ff6b6b;
    box-shadow: 0 0 0 3px rgba(255, 107, 107, 0.1);
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr 1fr;
    gap: 15px;
}

.modal-actions {
    display: flex;
    gap: 15px;
    justify-content: flex-end;
    padding-top: 20px;
    border-top: 1px solid #eee;
}

/* Context menu */
.context-menu {
    display: none;
    position: fixed;
    background: white;
    border-radius: 8px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
    z-index: 1001;
    min-width: 160px;
    border: 1px solid #eee;
}

.menu-item {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 12px 16px;
    cursor: pointer;
    font-size: 14px;
    transition: all 0.2s ease;
}

.menu-item:hover {
    background: rgba(255, 107, 107, 0.1);
}

.menu-item.danger:hover {
    background: rgba(220, 53, 69, 0.1);
    color: #dc3545;
}

.menu-separator {
    height: 1px;
    background: #eee;
    margin: 5px 0;
}

/* Responsive */
@media (max-width: 768px) {
    .filters-bar {
        flex-direction: column;
        gap: 10px;
    }
    
    .search-box input {
        width: 100%;
    }
    
    .stats-summary {
        flex-direction: column;
        gap: 10px;
    }
    
    .projects-grid {
        grid-template-columns: 1fr;
    }
    
    .form-row {
        grid-template-columns: 1fr;
    }
}
</style>

<script>
/**
 * ================================================================
 * 🎯 JAVASCRIPT MODULE PROJECTS
 * ================================================================
 */

let searchTimeout;
let currentOrder = '<?php echo $order; ?>';

// Fonctions principales
function showCreateModal() {
    document.getElementById('createProjectModal').style.display = 'block';
    document.getElementById('projectName').focus();
}

function closeCreateModal() {
    document.getElementById('createProjectModal').style.display = 'none';
    document.getElementById('createProjectForm').reset();
}

function openProject(projectId) {
    window.parent.location.href = `../clipmaker.php?module=timeline&project=${projectId}`;
}

function showProjectMenu(projectId, event) {
    event.stopPropagation();
    const menu = document.getElementById('projectMenu');
    window.currentProjectId = projectId;
    
    menu.style.display = 'block';
    menu.style.left = event.pageX + 'px';
    menu.style.top = event.pageY + 'px';
    
    // Fermer menu si clic ailleurs
    setTimeout(() => {
        document.addEventListener('click', closeProjectMenu);
    }, 10);
}

function closeProjectMenu() {
    document.getElementById('projectMenu').style.display = 'none';
    document.removeEventListener('click', closeProjectMenu);
}

function editProject(projectId) {
    // TODO: Implémenter modal édition
    console.log('Édition projet:', projectId);
    closeProjectMenu();
}

function duplicateProject(projectId) {
    if (confirm('Dupliquer ce projet ?')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = '?action=duplicate&project=' + projectId;
        document.body.appendChild(form);
        form.submit();
    }
    closeProjectMenu();
}

function exportProject(projectId) {
    window.parent.location.href = `../clipmaker.php?module=export&project=${projectId}`;
    closeProjectMenu();
}

function deleteProject(projectId) {
    if (confirm('⚠️ Supprimer définitivement ce projet ?\n\nCette action est irréversible.')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = '?action=delete&project=' + projectId;
        document.body.appendChild(form);
        form.submit();
    }
    closeProjectMenu();
}

// Filtres et recherche
function debounceSearch() {
    clearTimeout(searchTimeout);
    searchTimeout = setTimeout(applyFilters, 300);
}

function applyFilters() {
    const search = document.getElementById('searchInput').value;
    const status = document.getElementById('statusFilter').value;
    const sort = document.getElementById('sortSelect').value;
    
    const url = new URL(window.location);
    url.searchParams.set('search', search);
    url.searchParams.set('filter_status', status);
    url.searchParams.set('sort', sort);
    url.searchParams.set('order', currentOrder);
    
    window.location.href = url.toString();
}

function toggleOrder() {
    currentOrder = currentOrder === 'DESC' ? 'ASC' : 'DESC';
    applyFilters();
}

function switchView(view) {
    document.querySelectorAll('.btn-view').forEach(btn => {
        btn.classList.remove('active');
    });
    document.querySelector(`[data-view="${view}"]`).classList.add('active');
    
    const container = document.getElementById('projectsGrid');
    if (view === 'list') {
        container.style.gridTemplateColumns = '1fr';
    } else {
        container.style.gridTemplateColumns = 'repeat(auto-fill, minmax(320px, 1fr))';
    }
}

// Gestion presets résolution
document.addEventListener('DOMContentLoaded', function() {
    const widthSelect = document.getElementById('resolutionWidth');
    const heightSelect = document.getElementById('resolutionHeight');
    
    if (widthSelect && heightSelect) {
        widthSelect.addEventListener('change', function() {
            const presets = {
                '1920': '1080',
                '1080': '1080', 
                '720': '720'
            };
            
            heightSelect.value = presets[this.value] || '1080';
        });
    }
    
    // Fermer modal avec Escape
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeCreateModal();
            closeProjectMenu();
        }
    });
    
    console.log('📁 Module Projects Ready!');
});
</script>