<?php
/**
 * ================================================================
 * 🎞️ CLIPMAKER MEDIA - VERSION CORRIGÉE DÉFINITIVE
 * ================================================================
 * 
 * 🔧 TOUS LES PROBLÈMES FIXES :
 * ✅ Constantes SMG_* définies AVANT utilisation
 * ✅ Suppression complète médias + thumbnails
 * ✅ Interface responsive tablette
 * ✅ Gestion erreurs améliorée
 * 
 * 💡 BY PAPY 77 - VERSION FINALE QUI MARCHE !
 * ================================================================
 */

require_once '../auth_functions.php';

// 🔧 FIX PROBLÈME 4 : DÉFINIR CONSTANTES AVANT TOUT
if (!defined('SMG_UPLOAD_WEB_FULL')) {
    define('SMG_UPLOAD_WEB_FULL', '/social-media-generator/uploads/clipmaker/media');
    define('SMG_THUMBNAIL_WEB_FULL', '/social-media-generator/uploads/clipmaker/thumbnails');
    define('SMG_UPLOAD_SERVER_FULL', '/var/www/laloee-cours.fr/social-media-generator/uploads/clipmaker/media');
    define('SMG_THUMBNAIL_SERVER_FULL', '/var/www/laloee-cours.fr/social-media-generator/uploads/clipmaker/thumbnails');
    define('SMG_SERVER_ROOT', '/var/www/laloee-cours.fr/social-media-generator');
}

// Maintenant inclure le fix
require_once 'fix_thumb.php';

requireLogin();

$user = getCurrentUser();
$pdo = getDbConnection();

$action = $_GET['action'] ?? 'library';
$filter_type = $_GET['type'] ?? 'all';
$search = $_GET['search'] ?? '';
$sort = $_GET['sort'] ?? 'created_at';
$order = $_GET['order'] ?? 'DESC';

// 🔧 UTILISER LES CONSTANTES DÉFINIES
$upload_web_path = SMG_UPLOAD_WEB_FULL;
$thumbnails_web_path = SMG_THUMBNAIL_WEB_FULL;
$upload_server_path = SMG_UPLOAD_SERVER_FULL;
$thumbnails_server_path = SMG_THUMBNAIL_SERVER_FULL;

$allowed_formats = ['mp4', 'webm', 'avi', 'mov', 'mkv', 'jpg', 'jpeg', 'png', 'webp', 'gif', 'mp3', 'wav', 'ogg'];
$max_file_size = 100 * 1024 * 1024; // 100MB

// Traitement upload AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'upload') {
    header('Content-Type: application/json');
    
    try {
        if (empty($_FILES['files'])) {
            throw new Exception("Aucun fichier sélectionné");
        }
        
        $uploaded_files = [];
        $files = $_FILES['files'];
        
        // Traitement multiple fichiers
        for ($i = 0; $i < count($files['name']); $i++) {
            if ($files['error'][$i] !== UPLOAD_ERR_OK) {
                continue;
            }
            
            // Préparer fichier pour upload brutal
            $file_data = [
                'name' => $files['name'][$i],
                'tmp_name' => $files['tmp_name'][$i],
                'size' => $files['size'][$i],
                'type' => $files['type'][$i],
                'error' => $files['error'][$i]
            ];
            
            // 🔥 UTILISER LA FONCTION BRUTALE
            $result = uploadAndInsertBrutal($pdo, $user['id'], null, $file_data);
            
            if ($result['success']) {
                $uploaded_files[] = [
                    'id' => $result['media_id'],
                    'filename' => $file_data['name'],
                    'type' => 'success',
                    'size' => $file_data['size'],
                    'thumbnail' => $result['thumbnail_path']
                ];
            } else {
                throw new Exception("Erreur upload: " . $result['error']);
            }
        }
        
        echo json_encode([
            'success' => true,
            'message' => count($uploaded_files) . ' fichier(s) uploadé(s) avec succès',
            'files' => $uploaded_files
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Action pour preview
if ($action === 'preview' && isset($_GET['id'])) {
    header('Content-Type: application/json');
    
    try {
        $media_id = $_GET['id'];
        $stmt = $pdo->prepare("SELECT * FROM clipmaker_media WHERE id = ? AND user_id = ?");
        $stmt->execute([$media_id, $user['id']]);
        $media = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$media) {
            throw new Exception("Média introuvable");
        }
        
        echo json_encode([
            'success' => true,
            'media' => $media
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Action pour télécharger
if ($action === 'download' && isset($_GET['id'])) {
    try {
        $media_id = $_GET['id'];
        $stmt = $pdo->prepare("SELECT * FROM clipmaker_media WHERE id = ? AND user_id = ?");
        $stmt->execute([$media_id, $user['id']]);
        $media = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$media) {
            die("Média introuvable");
        }
        
        $web_path = $media['file_path'];
        if (strpos($web_path, '/social-media-generator/') === 0) {
            $server_path = '/var/www/laloee-cours.fr' . $web_path;
        } else {
            $server_path = $upload_server_path . '/' . basename($web_path);
        }

        if (!file_exists($server_path)) {
            die("Fichier introuvable sur le serveur: " . $server_path);
        }
        
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $media['original_filename'] . '"');
        header('Content-Length: ' . filesize($server_path));
        
        readfile($server_path);
        exit;
        
    } catch (Exception $e) {
        die("Erreur téléchargement: " . $e->getMessage());
    }
}

// 🔧 SUPPRESSION COMPLÈTE MÉDIA + THUMBNAIL
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'delete') {
    header('Content-Type: application/json');
    
    try {
        $media_id = $_POST['media_id'] ?? null;
        if (!$media_id) {
            throw new Exception("ID média manquant");
        }
        
        // Utiliser la fonction de suppression complète
        $result = deleteMediaCompletely($pdo, $media_id, $user['id']);
        
        echo json_encode($result);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Action réparation thumbnails
if ($action === 'repair_thumbnails') {
    header('Content-Type: application/json');
    
    try {
        $result = repairPathsBrutal($pdo, $user['id']);
        
        echo json_encode([
            'success' => true,
            'message' => "Réparation terminée: {$result['repaired']} médias réparés",
            'total' => $result['total'],
            'repaired' => $result['repaired']
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Récupération médias avec filtres
$where_conditions = ["user_id = ?"];
$params = [$user['id']];

if ($filter_type !== 'all') {
    $where_conditions[] = "file_type = ?";
    $params[] = $filter_type;
}

if (!empty($search)) {
    $where_conditions[] = "original_filename LIKE ?";
    $params[] = "%$search%";
}

$where_clause = implode(' AND ', $where_conditions);

try {
    // Récupération médias
    $stmt = $pdo->prepare("
        SELECT * FROM clipmaker_media 
        WHERE $where_clause 
        ORDER BY $sort $order 
        LIMIT 50
    ");
    $stmt->execute($params);
    $medias = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Stats médias
    $stmt = $pdo->prepare("
        SELECT 
            file_type,
            COUNT(*) as count,
            SUM(file_size) as total_size
        FROM clipmaker_media 
        WHERE user_id = ? 
        GROUP BY file_type
    ");
    $stmt->execute([$user['id']]);
    $stats_by_type = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Stats totales
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_count,
            SUM(file_size) as total_size
        FROM clipmaker_media 
        WHERE user_id = ?
    ");
    $stmt->execute([$user['id']]);
    $total_stats = $stmt->fetch();
    
} catch (PDOException $e) {
    error_log("Erreur récupération médias: " . $e->getMessage());
    $medias = [];
    $stats_by_type = [];
    $total_stats = ['total_count' => 0, 'total_size' => 0];
}

function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    }
    return $bytes . ' B';
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🎞️ Clipmaker Media - Bibliothèque CORRIGÉE</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #1a202c 0%, #2d3748 100%);
            color: #ffffff;
            min-height: 100vh;
        }

        .studio-header {
            background: linear-gradient(135deg, #2d3748, #4a5568);
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 2px solid #667eea;
            box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        }

        .studio-logo {
            display: flex;
            align-items: center;
            gap: 15px;
            font-size: 24px;
            font-weight: 700;
            background: linear-gradient(135deg, #667eea, #764ba2);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .studio-actions {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 14px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
        }

        .btn-secondary {
            background: rgba(255, 255, 255, 0.1);
            color: white;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .btn-danger {
            background: linear-gradient(135deg, #f56565, #e53e3e);
            color: white;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
        }

        .dashboard-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }

        .dashboard-header {
            text-align: center;
            margin-bottom: 30px;
        }

        .dashboard-header h1 {
            font-size: 42px;
            margin-bottom: 15px;
            background: linear-gradient(135deg, #667eea, #764ba2);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            font-weight: 800;
        }

        .dashboard-header p {
            font-size: 16px;
            color: #a0aec0;
        }

        .fix-status {
            background: linear-gradient(135deg, #48bb78, #38a169);
            color: white;
            padding: 15px 25px;
            border-radius: 12px;
            margin-bottom: 25px;
            text-align: center;
            font-weight: 600;
        }

        .media-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.05);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 25px;
            text-align: center;
            transition: all 0.3s ease;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .stat-card:hover {
            transform: translateY(-5px);
            border-color: rgba(102, 126, 234, 0.5);
        }

        .stat-card .stat-icon {
            font-size: 32px;
            margin-bottom: 15px;
        }

        .stat-card .stat-number {
            font-size: 24px;
            font-weight: 700;
            color: #667eea;
            margin-bottom: 8px;
        }

        .stat-card .stat-label {
            color: #a0aec0;
            font-size: 14px;
        }

        .upload-zone {
            background: rgba(255, 255, 255, 0.05);
            border: 3px dashed rgba(102, 126, 234, 0.3);
            border-radius: 16px;
            padding: 40px;
            text-align: center;
            margin-bottom: 30px;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .upload-zone:hover, .upload-zone.dragover {
            border-color: #667eea;
            background: rgba(102, 126, 234, 0.1);
        }

        .media-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 25px;
        }

        .media-card {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 16px;
            overflow: hidden;
            transition: all 0.3s ease;
            border: 2px solid transparent;
            cursor: pointer;
            backdrop-filter: blur(20px);
        }

        .media-card:hover {
            transform: translateY(-5px);
            border-color: rgba(102, 126, 234, 0.5);
        }

        .media-thumbnail {
            position: relative;
            height: 180px;
            background: linear-gradient(135deg, rgba(102, 126, 234, 0.1), rgba(118, 75, 162, 0.1));
            overflow: hidden;
        }

        .media-thumbnail img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .placeholder-thumbnail {
            display: flex;
            align-items: center;
            justify-content: center;
            height: 100%;
            font-size: 48px;
            color: rgba(102, 126, 234, 0.5);
        }

        .media-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.8);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 15px;
            opacity: 0;
            transition: all 0.3s ease;
        }

        .media-card:hover .media-overlay {
            opacity: 1;
        }

        .btn-overlay {
            background: rgba(255, 255, 255, 0.9);
            border: none;
            border-radius: 50%;
            width: 45px;
            height: 45px;
            cursor: pointer;
            font-size: 16px;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .btn-overlay:hover {
            background: white;
            transform: scale(1.1);
        }

        .btn-overlay.delete {
            background: rgba(245, 101, 101, 0.9);
            color: white;
        }

        .btn-overlay.delete:hover {
            background: #f56565;
        }

        .media-info {
            padding: 20px;
        }

        .media-name {
            font-weight: 600;
            color: #fff;
            margin-bottom: 10px;
            font-size: 14px;
        }

        .media-meta {
            display: flex;
            gap: 10px;
            margin-bottom: 10px;
            flex-wrap: wrap;
        }

        .meta-item {
            font-size: 12px;
            color: #a0aec0;
            background: rgba(102, 126, 234, 0.1);
            padding: 2px 8px;
            border-radius: 4px;
        }

        /* 🔧 RESPONSIVE TABLETTE */
        @media (max-width: 1024px) {
            .dashboard-container {
                padding: 15px;
            }
            
            .dashboard-header h1 {
                font-size: 32px;
            }
            
            .media-grid {
                grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
                gap: 20px;
            }
            
            .studio-actions {
                flex-direction: column;
                gap: 10px;
            }
            
            .btn {
                justify-content: center;
                width: 100%;
            }
        }

        @media (max-width: 768px) {
            .media-grid {
                grid-template-columns: 1fr;
            }
            
            .media-stats {
                grid-template-columns: repeat(2, 1fr);
            }
        }
    </style>
</head>

<body>
    <div class="studio-header">
        <div class="studio-logo">
            <span>🎞️</span>
            <span>Clipmaker Media CORRIGÉ</span>
        </div>

        <div class="studio-actions">
            <button class="btn btn-primary" onclick="showUploadZone()">
                📤 Upload Fichiers
            </button>
            <button class="btn btn-secondary" onclick="repairThumbnails()">
                🔧 Réparer Thumbnails
            </button>
            <a href="../clipmaker.php" class="btn btn-secondary">← Dashboard</a>
        </div>
    </div>

    <div class="dashboard-container">
        <div class="dashboard-header">
            <h1>🎞️ Bibliothèque Médias</h1>
            <p>Gestion complète avec suppression et thumbnails corrigés</p>
        </div>

        <div class="fix-status">
            🔧 ✅ TOUS LES FIXES APPLIQUÉS - Constantes SMG définies, suppression complète active, interface responsive !
        </div>

        <div class="media-stats">
            <div class="stat-card">
                <div class="stat-icon">📁</div>
                <div class="stat-number"><?php echo $total_stats['total_count']; ?></div>
                <div class="stat-label">Total fichiers</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">💾</div>
                <div class="stat-number"><?php echo formatFileSize($total_stats['total_size']); ?></div>
                <div class="stat-label">Stockage utilisé</div>
            </div>
            
            <?php foreach ($stats_by_type as $stat): ?>
            <div class="stat-card">
                <div class="stat-icon">
                    <?php 
                    echo match($stat['file_type']) {
                        'video' => '🎬',
                        'image' => '🖼️',
                        'audio' => '🎵',
                        default => '📄'
                    };
                    ?>
                </div>
                <div class="stat-number"><?php echo $stat['count']; ?></div>
                <div class="stat-label"><?php echo ucfirst($stat['file_type']); ?></div>
            </div>
            <?php endforeach; ?>
        </div>

        <div id="uploadZone" class="upload-zone" style="display: none;">
            <div>📤 Glissez vos fichiers ici ou cliquez pour sélectionner</div>
            <div style="font-size: 12px; color: #a0aec0; margin-top: 10px;">
                Formats supportés: MP4, WebM, AVI, MOV, JPG, PNG, MP3, WAV
            </div>
            <input type="file" id="fileInput" multiple accept="video/*,image/*,audio/*" style="display: none;">
            <div style="margin-top: 15px;">
                <button class="btn btn-secondary" onclick="hideUploadZone()">Annuler</button>
            </div>
        </div>

        <div class="media-grid">
            <?php if (empty($medias)): ?>
                <div style="grid-column: 1 / -1; text-align: center; padding: 60px; color: #a0aec0;">
                    <div style="font-size: 64px; margin-bottom: 20px;">🎞️</div>
                    <h3>Aucun média pour le moment</h3>
                    <p style="margin: 15px 0;">Importez vos premiers fichiers !</p>
                    <button class="btn btn-primary" onclick="showUploadZone()">📤 Importer médias</button>
                </div>
            <?php else: ?>
                <?php foreach ($medias as $media): ?>
                    <div class="media-card" data-media-id="<?php echo $media['id']; ?>">
                        <div class="media-thumbnail">
                            <?php if ($media['thumbnail_path']): ?>
                                <img src="<?php echo htmlspecialchars($media['thumbnail_path']); ?>" alt="Thumbnail">
                            <?php else: ?>
                                <div class="placeholder-thumbnail">
                                    <?php 
                                    echo match($media['file_type']) {
                                        'video' => '🎬',
                                        'image' => '🖼️',
                                        'audio' => '🎵',
                                        default => '📄'
                                    };
                                    ?>
                                </div>
                            <?php endif; ?>
                            
                            <div class="media-overlay">
                                <button class="btn-overlay" onclick="previewMedia(<?php echo $media['id']; ?>)" title="Aperçu">
                                    👁️
                                </button>
                                <button class="btn-overlay" onclick="downloadMedia(<?php echo $media['id']; ?>)" title="Télécharger">
                                    ⬇️
                                </button>
                                <button class="btn-overlay delete" onclick="deleteMediaComplete(<?php echo $media['id']; ?>)" title="Supprimer complètement">
                                    🗑️
                                </button>
                            </div>
                        </div>
                        
                        <div class="media-info">
                            <div class="media-name"><?php echo htmlspecialchars($media['original_filename']); ?></div>
                            <div class="media-meta">
                                <span class="meta-item"><?php echo formatFileSize($media['file_size']); ?></span>
                                <span class="meta-item"><?php echo strtoupper($media['file_type']); ?></span>
                                <?php if ($media['width'] && $media['height']): ?>
                                    <span class="meta-item"><?php echo $media['width']; ?>x<?php echo $media['height']; ?></span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>

    <script>
        function showUploadZone() {
            document.getElementById('uploadZone').style.display = 'block';
            setupDragAndDrop();
        }

        function hideUploadZone() {
            document.getElementById('uploadZone').style.display = 'none';
        }

        function setupDragAndDrop() {
            const uploadZone = document.getElementById('uploadZone');
            const fileInput = document.getElementById('fileInput');
            
            uploadZone.addEventListener('click', (e) => {
                if (e.target === uploadZone || e.target.closest('.upload-zone')) {
                    fileInput.click();
                }
            });
            
            uploadZone.addEventListener('dragover', (e) => {
                e.preventDefault();
                uploadZone.classList.add('dragover');
            });
            
            uploadZone.addEventListener('dragleave', (e) => {
                e.preventDefault();
                uploadZone.classList.remove('dragover');
            });
            
            uploadZone.addEventListener('drop', (e) => {
                e.preventDefault();
                uploadZone.classList.remove('dragover');
                uploadFiles(e.dataTransfer.files);
            });
            
            fileInput.addEventListener('change', (e) => {
                uploadFiles(e.target.files);
            });
        }

        function uploadFiles(files) {
            const formData = new FormData();
            
            for (let i = 0; i < files.length; i++) {
                formData.append('files[]', files[i]);
            }
            
            showNotification('📤 Upload en cours...', 'info');
            
            fetch('?action=upload', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('✅ ' + data.message, 'success');
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification('❌ ' + data.error, 'error');
                }
            })
            .catch(error => {
                showNotification('❌ Erreur: ' + error.message, 'error');
            });
        }

        // 🔧 SUPPRESSION COMPLÈTE CORRIGÉE
        function deleteMediaComplete(mediaId) {
            if (confirm('🗑️ Supprimer COMPLÈTEMENT ce média ?\n\n⚠️ Fichier + thumbnail + références BDD seront supprimés définitivement !')) {
                const formData = new FormData();
                formData.append('media_id', mediaId);
                
                showNotification('🗑️ Suppression en cours...', 'info');
                
                fetch('?action=delete', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showNotification('✅ ' + data.message, 'success');
                        
                        // Supprimer visuellement la carte
                        const card = document.querySelector(`[data-media-id="${mediaId}"]`);
                        if (card) {
                            card.style.opacity = '0';
                            card.style.transform = 'scale(0.8)';
                            setTimeout(() => card.remove(), 300);
                        }
                        
                        setTimeout(() => location.reload(), 2000);
                    } else {
                        showNotification('❌ ' + data.error, 'error');
                    }
                })
                .catch(error => {
                    showNotification('❌ Erreur suppression: ' + error.message, 'error');
                });
            }
        }

        function repairThumbnails() {
            if (confirm('🔧 Réparer tous les thumbnails manquants ?')) {
                showNotification('🔧 Réparation en cours...', 'info');
                
                fetch('?action=repair_thumbnails')
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            showNotification('✅ ' + data.message, 'success');
                            setTimeout(() => location.reload(), 2000);
                        } else {
                            showNotification('❌ ' + data.error, 'error');
                        }
                    })
                    .catch(error => {
                        showNotification('❌ Erreur: ' + error.message, 'error');
                    });
            }
        }

        function previewMedia(mediaId) {
            // Ouvrir modal preview
            showNotification('👁️ Fonction preview à implémenter', 'info');
        }

        function downloadMedia(mediaId) {
            window.open(`?action=download&id=${mediaId}`, '_blank');
        }

        function showNotification(message, type) {
            const notification = document.createElement('div');
            notification.textContent = message;
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: ${type === 'success' ? '#48bb78' : type === 'error' ? '#f56565' : '#4299e1'};
                color: white;
                padding: 15px 20px;
                border-radius: 8px;
                z-index: 10000;
                font-weight: 600;
                box-shadow: 0 5px 15px rgba(0,0,0,0.3);
            `;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.remove();
            }, 4000);
        }

        document.addEventListener('DOMContentLoaded', function() {
            console.log('🔧 Clipmaker Media CORRIGÉ chargé');
            console.log('✅ Constantes SMG définies');
            console.log('✅ Suppression complète active');
            console.log('✅ Interface responsive tablette');
            
            showNotification('🔧 ✅ Tous les fixes appliqués !', 'success');
        });
    </script>
</body>
</html>