<?php
/*
 * SMG Studio v2.0 - Script d'installation Workflow Destructif
 * Installation automatique de la structure et vérification
 * Path: /var/www/laloee-cours.fr/social-media-generator/install_workflow.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "🔥 SMG Studio v2.0 - Installation Workflow DESTRUCTIF\n";
echo "=====================================================\n\n";

class SMGWorkflowInstaller {
    private $basePath;
    private $errors = [];
    private $warnings = [];
    private $success = [];
    
    public function __construct() {
        $this->basePath = dirname(__FILE__);
        echo "📂 Base path: {$this->basePath}\n\n";
    }
    
    public function install() {
        echo "🚀 Démarrage installation...\n\n";
        
        $this->checkSystemRequirements();
        $this->createDirectoryStructure();
        $this->setPermissions();
        $this->createConfigurationFiles();
        $this->initializeDatabase();
        $this->createTestUser();
        $this->runTests();
        
        $this->displayResults();
    }
    
    private function checkSystemRequirements() {
        echo "1️⃣ Vérification prérequis système...\n";
        
        // PHP Version
        if (version_compare(PHP_VERSION, '7.4.0', '<')) {
            $this->errors[] = "PHP 7.4+ requis (actuel: " . PHP_VERSION . ")";
        } else {
            $this->success[] = "PHP version OK: " . PHP_VERSION;
        }
        
        // Extensions PHP
        $requiredExtensions = ['json', 'mysqli', 'curl', 'fileinfo'];
        foreach ($requiredExtensions as $ext) {
            if (!extension_loaded($ext)) {
                $this->errors[] = "Extension PHP manquante: $ext";
            } else {
                $this->success[] = "Extension $ext: OK";
            }
        }
        
        // FFmpeg
        $ffmpegPath = '/usr/bin/ffmpeg';
        if (!file_exists($ffmpegPath)) {
            $ffmpegPath = trim(shell_exec('which ffmpeg 2>/dev/null'));
        }
        
        if (empty($ffmpegPath) || !file_exists($ffmpegPath)) {
            $this->warnings[] = "FFmpeg non trouvé - Certaines fonctionnalités seront limitées";
        } else {
            $this->success[] = "FFmpeg trouvé: $ffmpegPath";
        }
        
        // Espace disque
        $freeSpace = disk_free_space($this->basePath);
        $requiredSpace = 5 * 1024 * 1024 * 1024; // 5GB
        
        if ($freeSpace < $requiredSpace) {
            $this->warnings[] = "Espace disque faible: " . $this->formatBytes($freeSpace) . " libre";
        } else {
            $this->success[] = "Espace disque OK: " . $this->formatBytes($freeSpace);
        }
        
        echo "   ✅ Prérequis vérifiés\n\n";
    }
    
    private function createDirectoryStructure() {
        echo "2️⃣ Création structure de dossiers...\n";
        
        $directories = [
            'user_workspaces/',
            'user_workspaces/user_001/',
            'user_workspaces/user_001/active_clip/',
            'user_workspaces/user_001/active_clip/01_clipmaker/',
            'user_workspaces/user_001/active_clip/02_audio_postprod/',
            'user_workspaces/user_001/active_clip/03_emission_final/',
            'user_workspaces/user_001/active_clip/04_social_ready/',
            'user_workspaces/user_001/archives/',
            'user_workspaces/user_001/logs/',
            'user_workspaces/user_001/temp/',
            'user_workspaces/user_002/',
            'user_workspaces/user_002/active_clip/',
            'user_workspaces/user_002/active_clip/01_clipmaker/',
            'user_workspaces/user_002/active_clip/02_audio_postprod/',
            'user_workspaces/user_002/active_clip/03_emission_final/',
            'user_workspaces/user_002/active_clip/04_social_ready/',
            'user_workspaces/user_002/archives/',
            'user_workspaces/user_002/logs/',
            'user_workspaces/user_002/temp/',
            'user_workspaces/user_003/',
            'user_workspaces/user_003/active_clip/',
            'user_workspaces/user_003/active_clip/01_clipmaker/',
            'user_workspaces/user_003/active_clip/02_audio_postprod/',
            'user_workspaces/user_003/active_clip/03_emission_final/',
            'user_workspaces/user_003/active_clip/04_social_ready/',
            'user_workspaces/user_003/archives/',
            'user_workspaces/user_003/logs/',
            'user_workspaces/user_003/temp/',
            'api/',
            'config/',
            'modules/',
            'css/',
            'js/',
            'logs/',
            'temp/',
            'uploads/clipmaker/exports/',
            'uploads/audio/',
            'uploads/final/',
            'uploads/social/'
        ];
        
        foreach ($directories as $dir) {
            $fullPath = $this->basePath . '/' . $dir;
            
            if (!is_dir($fullPath)) {
                if (mkdir($fullPath, 0755, true)) {
                    $this->success[] = "Dossier créé: $dir";
                } else {
                    $this->errors[] = "Impossible de créer: $dir";
                }
            } else {
                $this->success[] = "Dossier existe: $dir";
            }
        }
        
        echo "   ✅ Structure créée\n\n";
    }
    
    private function setPermissions() {
        echo "3️⃣ Configuration permissions...\n";
        
        $permissions = [
            'user_workspaces/' => 0755,
            'uploads/' => 0755,
            'logs/' => 0755,
            'temp/' => 0777
        ];
        
        foreach ($permissions as $path => $perm) {
            $fullPath = $this->basePath . '/' . $path;
            
            if (is_dir($fullPath)) {
                if (chmod($fullPath, $perm)) {
                    $this->success[] = "Permissions $path: " . decoct($perm);
                } else {
                    $this->warnings[] = "Impossible de modifier permissions: $path";
                }
            }
        }
        
        // Permissions récursives pour user_workspaces
        $this->setRecursivePermissions($this->basePath . '/user_workspaces/', 0755);
        
        echo "   ✅ Permissions configurées\n\n";
    }
    
    private function setRecursivePermissions($path, $perm) {
        if (!is_dir($path)) return;
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($iterator as $item) {
            if ($item->isDir()) {
                chmod($item->getPathname(), $perm);
            }
        }
    }
    
    private function createConfigurationFiles() {
        echo "4️⃣ Création fichiers de configuration...\n";
        
        // .htaccess pour sécuriser user_workspaces
        $htaccessContent = <<<HTACCESS
# SMG Studio v2.0 - Sécurisation workspace utilisateurs
Order Deny,Allow
Deny from all

# Autoriser seulement les fichiers media
<FilesMatch "\.(mp4|mov|avi|webm|wav|mp3|jpg|png|gif)$">
    Order Allow,Deny
    Allow from all
</FilesMatch>

# Bloquer les scripts
<FilesMatch "\.(php|js|html|htm)$">
    Order Deny,Allow
    Deny from all
</FilesMatch>
HTACCESS;
        
        file_put_contents($this->basePath . '/user_workspaces/.htaccess', $htaccessContent);
        $this->success[] = "Fichier .htaccess créé";
        
        // Configuration nginx (optionnel)
        $nginxConfig = <<<NGINX
# SMG Studio v2.0 - Configuration Nginx
location /social-media-generator/user_workspaces/ {
    # Bloquer accès direct aux fichiers sensibles
    location ~ \.(php|js|html|htm|json|log)$ {
        deny all;
    }
    
    # Autoriser seulement les médias
    location ~ \.(mp4|mov|avi|webm|wav|mp3|jpg|png|gif)$ {
        expires 1h;
        add_header Cache-Control "public, no-transform";
    }
}
NGINX;
        
        file_put_contents($this->basePath . '/nginx_config.txt', $nginxConfig);
        $this->success[] = "Configuration Nginx générée";
        
        echo "   ✅ Configuration créée\n\n";
    }
    
    private function initializeDatabase() {
        echo "5️⃣ Initialisation base de données...\n";
        
        // Vérifier si les tables existent déjà
        $requiredTables = ['users', 'workflow_exports', 'clipmaker_projects'];
        $missingTables = [];
        
        try {
            // Utiliser les paramètres de connexion de transition.sql
            $host = '57.128.183.39';
            $dbname = 'laloedata';
            $username = 'root'; // À adapter selon vos paramètres
            $password = '';     // À adapter selon vos paramètres
            
            $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            foreach ($requiredTables as $table) {
                $stmt = $pdo->prepare("SHOW TABLES LIKE ?");
                $stmt->execute([$table]);
                
                if (!$stmt->fetch()) {
                    $missingTables[] = $table;
                }
            }
            
            if (empty($missingTables)) {
                $this->success[] = "Base de données OK - Toutes les tables présentes";
            } else {
                $this->warnings[] = "Tables manquantes: " . implode(', ', $missingTables);
                $this->warnings[] = "Exécutez transition.sql pour créer les tables";
            }
            
        } catch (PDOException $e) {
            $this->warnings[] = "Connexion DB échouée: " . $e->getMessage();
            $this->warnings[] = "Vérifiez les paramètres dans transition.sql";
        }
        
        echo "   ✅ Base de données vérifiée\n\n";
    }
    
    private function createTestUser() {
        echo "6️⃣ Création utilisateur de test...\n";
        
        $testUsers = ['user_001', 'user_002', 'user_003'];
        
        foreach ($testUsers as $userId) {
            $userWorkspace = $this->basePath . "/user_workspaces/$userId/";
            
            // Créer fichier de stats initial
            $initialStats = [
                'totalProjects' => 0,
                'activeWorkflows' => 0,
                'completedToday' => 0,
                'avgTime' => '0h',
                'lastUpdate' => date('Y-m-d H:i:s')
            ];
            
            file_put_contents($userWorkspace . 'stats.json', json_encode($initialStats, JSON_PRETTY_PRINT));
            
            // Créer fichier readme
            $readme = <<<README
SMG Studio v2.0 - Workspace Utilisateur
=======================================

User ID: $userId
Créé le: {date('Y-m-d H:i:s')}
Type: Workflow DESTRUCTIF

Structure:
- active_clip/     : Clip en cours de création
- archives/        : Clips terminés et archivés  
- logs/           : Logs d'activité
- temp/           : Fichiers temporaires

IMPORTANT: 
- Un seul clip actif à la fois
- Workflow destructif - pas de retour sans perte
- Archivage automatique après completion

README;
            
            file_put_contents($userWorkspace . 'README.txt', $readme);
            
            $this->success[] = "Utilisateur $userId initialisé";
        }
        
        echo "   ✅ Utilisateurs de test créés\n\n";
    }
    
    private function runTests() {
        echo "7️⃣ Exécution tests de fonctionnement...\n";
        
        // Test 1: Création/suppression fichier
        $testFile = $this->basePath . '/user_workspaces/user_001/test_write.tmp';
        
        if (file_put_contents($testFile, 'test') && unlink($testFile)) {
            $this->success[] = "Test écriture fichier: OK";
        } else {
            $this->errors[] = "Test écriture fichier: ÉCHEC";
        }
        
        // Test 2: Permissions dossier
        $testDir = $this->basePath . '/user_workspaces/user_001/test_dir';
        
        if (mkdir($testDir) && rmdir($testDir)) {
            $this->success[] = "Test création dossier: OK";
        } else {
            $this->errors[] = "Test création dossier: ÉCHEC";
        }
        
        // Test 3: API endpoints
        if (file_exists($this->basePath . '/api/workflow.php')) {
            $this->success[] = "API workflow.php trouvée";
        } else {
            $this->warnings[] = "API workflow.php non trouvée";
        }
        
        // Test 4: Configuration
        if (file_exists($this->basePath . '/config/workflow_config.php')) {
            $this->success[] = "Configuration workflow trouvée";
        } else {
            $this->warnings[] = "Fichier de configuration manquant";
        }
        
        echo "   ✅ Tests terminés\n\n";
    }
    
    private function displayResults() {
        echo "📊 RÉSULTATS INSTALLATION\n";
        echo "=========================\n\n";
        
        if (!empty($this->success)) {
            echo "✅ SUCCÈS (" . count($this->success) . "):\n";
            foreach ($this->success as $msg) {
                echo "   • $msg\n";
            }
            echo "\n";
        }
        
        if (!empty($this->warnings)) {
            echo "⚠️  AVERTISSEMENTS (" . count($this->warnings) . "):\n";
            foreach ($this->warnings as $msg) {
                echo "   • $msg\n";
            }
            echo "\n";
        }
        
        if (!empty($this->errors)) {
            echo "❌ ERREURS (" . count($this->errors) . "):\n";
            foreach ($this->errors as $msg) {
                echo "   • $msg\n";
            }
            echo "\n";
        }
        
        // Verdict final
        if (empty($this->errors)) {
            if (empty($this->warnings)) {
                echo "🎉 INSTALLATION PARFAITE !\n";
                echo "Le système SMG Studio v2.0 est prêt à l'emploi.\n\n";
            } else {
                echo "✅ INSTALLATION RÉUSSIE avec avertissements\n";
                echo "Le système fonctionne mais certains éléments peuvent être optimisés.\n\n";
            }
            
            $this->displayNextSteps();
        } else {
            echo "💥 INSTALLATION ÉCHOUÉE\n";
            echo "Corrigez les erreurs ci-dessus avant d'utiliser le système.\n\n";
        }
    }
    
    private function displayNextSteps() {
        echo "🚀 PROCHAINES ÉTAPES\n";
        echo "===================\n";
        echo "1. Importez transition.sql dans votre base de données\n";
        echo "2. Configurez les paramètres dans config/workflow_config.php\n";
        echo "3. Testez avec: workflow_manager.php?user=user_001\n";
        echo "4. Créez vos premiers utilisateurs\n";
        echo "5. Configurez les intégrations (FFmpeg, etc.)\n\n";
        
        echo "📝 FICHIERS IMPORTANTS\n";
        echo "=====================\n";
        echo "• workflow_manager.php          : Interface principale\n";
        echo "• api/workflow_manager_api.php  : API destructive\n";
        echo "• config/workflow_config.php    : Configuration\n";
        echo "• css_de_reference.css          : Styles\n";
        echo "• user_workspaces/              : Données utilisateurs\n\n";
        
        echo "🔗 URLS DE TEST\n";
        echo "===============\n";
        $baseUrl = 'https://' . ($_SERVER['HTTP_HOST'] ?? 'localhost') . dirname($_SERVER['REQUEST_URI']);
        echo "• Workflow Manager: $baseUrl/workflow_manager.php?user=user_001\n";
        echo "• API Test: $baseUrl/api/workflow_manager_api.php?action=get_workflow_state&user=user_001\n\n";
        
        echo "🆘 SUPPORT\n";
        echo "==========\n";
        echo "En cas de problème:\n";
        echo "1. Vérifiez les logs dans logs/\n";
        echo "2. Consultez les README.txt dans user_workspaces/\n";
        echo "3. Testez les permissions avec ls -la\n\n";
        
        echo "🔥 SMG Studio v2.0 DESTRUCTIF - Installation terminée !\n";
    }
    
    private function formatBytes($size) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        for ($i = 0; $size > 1024 && $i < 4; $i++) {
            $size /= 1024;
        }
        return round($size, 2) . ' ' . $units[$i];
    }
}

// Exécution du script
if (php_sapi_name() === 'cli' || isset($_GET['install'])) {
    $installer = new SMGWorkflowInstaller();
    $installer->install();
} else {
    echo <<<HTML
<!DOCTYPE html>
<html>
<head>
    <title>SMG Studio v2.0 - Installation</title>
    <style>
        body { font-family: monospace; background: #1a1a1a; color: #00ff00; padding: 20px; }
        .container { max-width: 800px; margin: 0 auto; }
        .warning { color: #ffff00; background: rgba(255,255,0,0.1); padding: 10px; margin: 20px 0; border-left: 4px solid #ffff00; }
        .button { display: inline-block; background: #00ff00; color: #000; padding: 15px 30px; text-decoration: none; font-weight: bold; margin: 20px 0; border-radius: 5px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔥 SMG Studio v2.0 - Installation Workflow DESTRUCTIF</h1>
        
        <div class="warning">
            ⚠️ <strong>ATTENTION :</strong> Ce script va créer des dossiers et modifier les permissions du serveur.
            Assurez-vous d'avoir les droits nécessaires avant de continuer.
        </div>
        
        <p><strong>Ce script va :</strong></p>
        <ul>
            <li>Vérifier les prérequis système</li>
            <li>Créer la structure de dossiers</li>
            <li>Configurer les permissions</li>
            <li>Initialiser les utilisateurs de test</li>
            <li>Tester le fonctionnement</li>
        </ul>
        
        <a href="?install=1" class="button">🚀 LANCER L'INSTALLATION</a>
        
        <h3>Prérequis</h3>
        <ul>
            <li>PHP 7.4+</li>
            <li>Extensions: json, mysqli, curl, fileinfo</li>
            <li>Base de données MySQL avec transition.sql importé</li>
            <li>Permissions d'écriture sur le serveur</li>
            <li>5GB d'espace disque libre minimum</li>
        </ul>
    </div>
</body>
</html>
HTML;
}
?>