<?php
/*
 * SMG Studio v2.0 - Fonctions de Sécurité
 * Path: /var/www/laloee-cours.fr/social-media-generator/includes/security.php
 */

/**
 * Vérifier si l'utilisateur est connecté
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Récupérer l'utilisateur actuellement connecté
 */
function getCurrentUser() {
    global $pdo;
    
    if (!isLoggedIn()) {
        return null;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND status = 'active'");
        $stmt->execute([$_SESSION['user_id']]);
        return $stmt->fetch();
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Vérifier si l'utilisateur est admin
 */
function isAdmin() {
    $user = getCurrentUser();
    return $user && $user['role'] === 'admin';
}

/**
 * Forcer la déconnexion
 */
function forceLogout() {
    session_destroy();
    header('Location: login.php?expired=1');
    exit;
}

/**
 * Sécuriser une page (redirection si non connecté)
 */
function requireLogin() {
    if (!isLoggedIn()) {
        header('Location: login.php');
        exit;
    }
}

/**
 * Sécuriser une page admin
 */
function requireAdmin() {
    requireLogin();
    if (!isAdmin()) {
        header('Location: dashboard.php?error=access_denied');
        exit;
    }
}

/**
 * Nettoyer et valider les données
 */
function sanitize($data) {
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

/**
 * Générer un token CSRF
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Vérifier le token CSRF
 */
function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Générer un mot de passe aléatoire
 */
function generateRandomPassword($length = 12) {
    $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%&*';
    return substr(str_shuffle($chars), 0, $length);
}

/**
 * Logger les tentatives de connexion
 */
function logLoginAttempt($email, $success, $reason = '') {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO login_attempts (email, ip_address, user_agent, success, failure_reason, attempted_at) 
            VALUES (?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $email,
            $_SERVER['REMOTE_ADDR'] ?? '',
            $_SERVER['HTTP_USER_AGENT'] ?? '',
            $success ? 1 : 0,
            $reason
        ]);
    } catch (Exception $e) {
        // Ignore silencieusement si erreur de log
    }
}

/**
 * Vérifier les tentatives de force brute
 */
function checkBruteForce($email, $max_attempts = 5, $time_window = 900) { // 15 minutes
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as attempts 
            FROM login_attempts 
            WHERE email = ? 
            AND success = 0 
            AND attempted_at > DATE_SUB(NOW(), INTERVAL ? SECOND)
        ");
        $stmt->execute([$email, $time_window]);
        $result = $stmt->fetch();
        
        return $result['attempts'] >= $max_attempts;
    } catch (Exception $e) {
        return false; // En cas d'erreur, on laisse passer
    }
}

/**
 * Valider un email
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Valider la force d'un mot de passe
 */
function isStrongPassword($password) {
    return strlen($password) >= 8 && 
           preg_match('/[a-z]/', $password) && 
           preg_match('/[A-Z]/', $password) && 
           preg_match('/[0-9]/', $password);
}