<?php
/*
 * SMG Studio v2.0 - Debug Workflow
 * Test de l'API et des redirections pour diagnostiquer les problèmes
 * Path: /var/www/laloee-cours.fr/social-media-generator/debug_workflow.php
 */

header('Content-Type: text/html; charset=UTF-8');

echo '<!DOCTYPE html>
<html>
<head>
    <title>🔧 SMG Debug Workflow</title>
    <style>
        body { font-family: monospace; background: #1a1a1a; color: #00ff00; padding: 20px; line-height: 1.5; }
        .container { max-width: 1000px; margin: 0 auto; }
        .section { background: rgba(0,255,0,0.1); padding: 15px; margin: 15px 0; border-left: 4px solid #00ff00; }
        .error { color: #ff6b6b; background: rgba(255,107,107,0.1); border-left-color: #ff6b6b; }
        .warning { color: #ffd700; background: rgba(255,215,0,0.1); border-left-color: #ffd700; }
        .success { color: #48bb78; background: rgba(72,187,120,0.1); border-left-color: #48bb78; }
        .code { background: rgba(255,255,255,0.1); padding: 10px; margin: 10px 0; border-radius: 5px; white-space: pre-wrap; }
        .button { display: inline-block; padding: 10px 20px; background: #00ff00; color: #000; text-decoration: none; margin: 5px; border-radius: 3px; font-weight: bold; }
        table { width: 100%; border-collapse: collapse; margin: 10px 0; }
        th, td { padding: 8px; text-align: left; border: 1px solid #333; }
        th { background: rgba(0,255,0,0.2); }
    </style>
</head>
<body>
<div class="container">';

echo '<h1>🔧 SMG Studio v2.0 - Debug Workflow</h1>';
echo '<p>Diagnostic complet du système workflow destructif</p>';

// 1. TEST FICHIERS ET STRUCTURE
echo '<div class="section">';
echo '<h2>1️⃣ Test Structure Fichiers</h2>';

$requiredFiles = [
    'api/workflow.php' => 'API de base',
    'api/workflow_manager_api.php' => 'API workflow destructif', 
    'config/workflow_config.php' => 'Configuration',
    'modules/clipmaker_workflow_interface.html' => 'Interface Clipmaker',
    'css/workflow_destructif.css' => 'CSS workflow',
    'user_workspaces/' => 'Répertoire utilisateurs'
];

$missingFiles = [];
$existingFiles = [];

foreach ($requiredFiles as $file => $description) {
    if (file_exists(__DIR__ . '/' . $file)) {
        $existingFiles[] = ['file' => $file, 'desc' => $description, 'size' => filesize(__DIR__ . '/' . $file)];
    } else {
        $missingFiles[] = ['file' => $file, 'desc' => $description];
    }
}

if (empty($missingFiles)) {
    echo '<div class="success">✅ Tous les fichiers requis sont présents !</div>';
} else {
    echo '<div class="error">❌ Fichiers manquants :</div>';
    foreach ($missingFiles as $missing) {
        echo "<div class=\"error\">  • {$missing['file']} - {$missing['desc']}</div>";
    }
}

echo '<table><tr><th>Fichier</th><th>Description</th><th>Taille</th></tr>';
foreach ($existingFiles as $file) {
    $size = $file['size'] > 1024 ? round($file['size']/1024, 1).'KB' : $file['size'].'B';
    echo "<tr><td>{$file['file']}</td><td>{$file['desc']}</td><td>{$size}</td></tr>";
}
echo '</table>';

echo '</div>';

// 2. TEST API WORKFLOW
echo '<div class="section">';
echo '<h2>2️⃣ Test API Workflow</h2>';

$apiTests = [
    'workflow.php?action=stats&user=user_001' => 'API Base - Stats',
    'workflow_manager_api.php?action=get_workflow_state&user=user_001' => 'API Destructif - État'
];

foreach ($apiTests as $endpoint => $desc) {
    $apiUrl = 'api/' . $endpoint;
    
    echo "<h3>Test: {$desc}</h3>";
    
    if (file_exists(__DIR__ . '/' . $apiUrl)) {
        echo '<div class="success">✅ Fichier API trouvé</div>';
        
        // Test inclusion
        try {
            $apiPath = __DIR__ . '/api/' . explode('?', $endpoint)[0];
            if (is_readable($apiPath)) {
                echo '<div class="success">✅ API lisible</div>';
            } else {
                echo '<div class="error">❌ API non lisible - Vérifiez les permissions</div>';
            }
        } catch (Exception $e) {
            echo '<div class="error">❌ Erreur inclusion API: ' . $e->getMessage() . '</div>';
        }
        
        echo "<div class=\"code\">curl -X GET 'https://{$_SERVER['HTTP_HOST']}/social-media-generator/{$apiUrl}'</div>";
        
    } else {
        echo '<div class="error">❌ API non trouvée</div>';
    }
}

echo '</div>';

// 3. TEST CONFIGURATION
echo '<div class="section">';
echo '<h2>3️⃣ Test Configuration</h2>';

if (file_exists(__DIR__ . '/config/workflow_config.php')) {
    try {
        include_once __DIR__ . '/config/workflow_config.php';
        
        if (class_exists('SMGWorkflowConfig')) {
            echo '<div class="success">✅ Configuration chargée avec succès</div>';
            
            $stats = SMGWorkflowConfig::getWorkflowStats();
            echo '<table>';
            echo '<tr><th>Paramètre</th><th>Valeur</th></tr>';
            foreach ($stats as $key => $value) {
                $displayValue = is_bool($value) ? ($value ? 'Oui' : 'Non') : $value;
                echo "<tr><td>{$key}</td><td>{$displayValue}</td></tr>";
            }
            echo '</table>';
            
        } else {
            echo '<div class="error">❌ Classe SMGWorkflowConfig non trouvée</div>';
        }
        
    } catch (Exception $e) {
        echo '<div class="error">❌ Erreur configuration: ' . $e->getMessage() . '</div>';
    }
} else {
    echo '<div class="error">❌ Fichier de configuration non trouvé</div>';
}

echo '</div>';

// 4. TEST RÉPERTOIRES UTILISATEURS
echo '<div class="section">';
echo '<h2>4️⃣ Test Répertoires Utilisateurs</h2>';

$testUsers = ['user_001', 'user_002', 'user_003'];
$workspaceBase = __DIR__ . '/user_workspaces/';

foreach ($testUsers as $userId) {
    $userDir = $workspaceBase . $userId . '/';
    
    echo "<h3>Utilisateur: {$userId}</h3>";
    
    if (is_dir($userDir)) {
        echo '<div class="success">✅ Répertoire utilisateur existe</div>';
        
        $requiredDirs = [
            'active_clip/',
            'active_clip/01_clipmaker/',
            'active_clip/02_audio_postprod/', 
            'active_clip/03_emission_final/',
            'active_clip/04_social_ready/',
            'archives/',
            'logs/',
            'temp/'
        ];
        
        $missingDirs = [];
        foreach ($requiredDirs as $dir) {
            if (!is_dir($userDir . $dir)) {
                $missingDirs[] = $dir;
            }
        }
        
        if (empty($missingDirs)) {
            echo '<div class="success">✅ Structure complète</div>';
        } else {
            echo '<div class="warning">⚠️ Dossiers manquants: ' . implode(', ', $missingDirs) . '</div>';
        }
        
        // Vérifier permissions
        if (is_writable($userDir)) {
            echo '<div class="success">✅ Permissions écriture OK</div>';
        } else {
            echo '<div class="error">❌ Pas de permissions d\'écriture</div>';
        }
        
    } else {
        echo '<div class="error">❌ Répertoire utilisateur manquant</div>';
    }
}

echo '</div>';

// 5. TEST URL CLIPMAKER
echo '<div class="section">';
echo '<h2>5️⃣ Test URL Clipmaker</h2>';

$clipmakerUrls = [
    'modules/clipmaker_workflow_interface.html',
    'https://laloee-cours.fr/social-media-generator/modules/clipmaker_workflow_interface.html'
];

foreach ($clipmakerUrls as $url) {
    echo "<h3>Test URL: {$url}</h3>";
    
    if (strpos($url, 'http') === 0) {
        // URL absolue - Test avec curl si disponible
        if (function_exists('curl_init')) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_NOBODY, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            
            $result = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode == 200) {
                echo '<div class="success">✅ URL accessible (HTTP 200)</div>';
            } else {
                echo "<div class=\"error\">❌ URL inaccessible (HTTP {$httpCode})</div>";
            }
        } else {
            echo '<div class="warning">⚠️ cURL non disponible - Test manuel requis</div>';
        }
    } else {
        // URL relative - Test fichier
        if (file_exists(__DIR__ . '/' . $url)) {
            echo '<div class="success">✅ Fichier trouvé</div>';
            $size = filesize(__DIR__ . '/' . $url);
            echo "<div class=\"success\">Taille: " . ($size > 1024 ? round($size/1024, 1).'KB' : $size.'B') . "</div>";
        } else {
            echo '<div class="error">❌ Fichier non trouvé</div>';
        }
    }
}

echo '</div>';

// 6. TEST JAVASCRIPT WORKFLOW
echo '<div class="section">';
echo '<h2>6️⃣ Test JavaScript Interface</h2>';

echo '<div class="code">
// Test de redirection JavaScript
function testRedirect() {
    const userId = "user_001";
    const clipName = "Test Clip";
    const url = `modules/clipmaker_workflow_interface.html?user=${encodeURIComponent(userId)}&project=${encodeURIComponent(clipName)}`;
    
    console.log("URL de redirection:", url);
    
    // Test ouverture dans nouvel onglet
    window.open(url, "_blank");
}
</div>';

echo '<button class="button" onclick="testRedirect()">🧪 Tester Redirection</button>';

echo '<script>
function testRedirect() {
    const userId = "user_001";
    const clipName = "Test Clip Debug";
    const url = `modules/clipmaker_workflow_interface.html?user=${encodeURIComponent(userId)}&project=${encodeURIComponent(clipName)}`;
    
    console.log("URL de redirection:", url);
    alert("Ouverture dans nouvel onglet: " + url);
    
    window.open(url, "_blank");
}
</script>';

echo '</div>';

// 7. ACTIONS RECOMMANDÉES
echo '<div class="section">';
echo '<h2>7️⃣ Actions Recommandées</h2>';

$recommendations = [
    'Vérifier que tous les fichiers requis existent',
    'Tester l\'API manuellement avec curl ou Postman',
    'Vérifier les permissions des répertoires utilisateurs', 
    'Tester l\'URL Clipmaker dans un navigateur',
    'Vérifier les logs d\'erreur PHP/Apache',
    'Tester la redirection JavaScript dans la console navigateur'
];

echo '<ol>';
foreach ($recommendations as $rec) {
    echo "<li>{$rec}</li>";
}
echo '</ol>';

echo '</div>';

// 8. LIENS UTILES
echo '<div class="section">';
echo '<h2>8️⃣ Liens de Test</h2>';

$baseUrl = 'https://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['REQUEST_URI']);

$testLinks = [
    $baseUrl . '/workflow_manager.php?user=user_001' => 'Workflow Manager',
    $baseUrl . '/api/workflow_manager_api.php?action=get_workflow_state&user=user_001' => 'Test API État',
    $baseUrl . '/modules/clipmaker_workflow_interface.html?user=user_001' => 'Clipmaker Direct',
    $baseUrl . '/install_workflow.php' => 'Installation',
    $baseUrl . '/user_workspaces/user_001/' => 'Workspace User 001'
];

foreach ($testLinks as $url => $name) {
    echo "<a href=\"{$url}\" target=\"_blank\" class=\"button\">{$name}</a>";
}

echo '</div>';

echo '<div class="section success">';
echo '<h2>🏁 Diagnostic Terminé</h2>';
echo '<p>Utilisez les informations ci-dessus pour identifier et corriger les problèmes de redirection.</p>';
echo '<p><strong>Problème courant :</strong> Si la boucle persiste, vérifiez que l\'API répond correctement et que l\'URL Clipmaker est accessible.</p>';
echo '</div>';

echo '</div>
</body>
</html>';
?>