<?php
/**
 * ================================================================
 * 📊 DASHBOARD PRINCIPAL SMG STUDIO v2.0 - CORRIGÉ
 * ================================================================
 * 
 * 🎯 Navigation complète et fonctionnelle entre tous les modules IA
 * ✅ Liens bidirectionnels dashboard ↔ IA modules
 * 🔄 Retour dashboard depuis tous les modules
 * 💰 FONCTIONS DÉBIT TOKENS INTÉGRÉES
 * 
 * 💡 BY PAPY 77 - DASHBOARD COMPLET AVEC DÉBIT AUTO !
 * ================================================================
 */

session_start();
require_once 'config/db.php';
require_once 'helpers/common_functions.php';

// Vérification session
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?expired=1');
    exit;
}

// Récupération utilisateur
try {
    $stmt = $pdo->prepare("SELECT id, name, email, role FROM users WHERE id = ? AND status = 'active'");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        session_destroy();
        header('Location: login.php?expired=1');
        exit;
    }
} catch (Exception $e) {
    error_log("Erreur user dashboard: " . $e->getMessage());
    header('Location: login.php?error=1');
    exit;
}

// ================================================================
// 🔧 FONCTIONS MÉTIER (DÉFINIR AVANT UTILISATION)
// ================================================================

function getUserNetworkStats($user_id) {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT pn.network, COUNT(DISTINCT p.id) as total_posts,
                   COUNT(DISTINCT sp.id) as scheduled_posts,
                   COUNT(DISTINCT CASE WHEN sp.status = 'sent' THEN sp.id END) as sent_posts,
                   sp2.display_name, sp2.icon
            FROM posts p
            JOIN post_networks pn ON p.id = pn.post_id AND pn.is_active = 1
            LEFT JOIN scheduled_posts sp ON p.id = sp.post_id AND pn.network = sp.network
            LEFT JOIN social_platforms sp2 ON pn.network = sp2.name
            WHERE p.user_id = ? AND p.statut IN ('draft', 'scheduled')
            GROUP BY pn.network ORDER BY total_posts DESC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getUserNetworkStats: " . $e->getMessage());
        return [];
    }
}

function getAvailablePosts($user_id, $network_filter = 'all') {
    global $pdo;
    try {
        $where_network = $network_filter !== 'all' ? "AND pn.network = ?" : "";
        $params = [$user_id];
        if ($network_filter !== 'all') {
            $params[] = $network_filter;
        }
        
        $stmt = $pdo->prepare("
            SELECT DISTINCT p.*, 
                   GROUP_CONCAT(DISTINCT pn.network ORDER BY pn.network) as networks,
                   COUNT(DISTINCT pm.id) as media_count,
                   MIN(pm.media_path) as preview_media_path,
                   MIN(pm.media_type) as preview_media_type
            FROM posts p
            LEFT JOIN post_networks pn ON p.id = pn.post_id
            LEFT JOIN post_medias pm ON p.id = pm.post_id
            WHERE p.user_id = ? AND p.statut IN ('draft', 'scheduled') {$where_network}
            GROUP BY p.id ORDER BY p.created_at DESC
        ");
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getAvailablePosts: " . $e->getMessage());
        return [];
    }
}

function getScheduledPosts($user_id) {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT sp.*, p.titre, p.contenu, 
                   sp2.display_name as network_name, sp2.icon as network_icon
            FROM scheduled_posts sp
            JOIN posts p ON sp.post_id = p.id
            LEFT JOIN social_platforms sp2 ON sp.network = sp2.name
            WHERE sp.user_id = ? AND sp.status IN ('pending', 'processing')
            ORDER BY sp.scheduled_datetime ASC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getScheduledPosts: " . $e->getMessage());
        return [];
    }
}

function getRecentTransmissions($user_id) {
    global $pdo;
    try {
        $stmt = $pdo->prepare("
            SELECT tl.*, p.titre, sp.scheduled_datetime,
                   sp2.display_name as network_name, sp2.icon as network_icon
            FROM transmission_logs tl
            JOIN scheduled_posts sp ON tl.scheduled_post_id = sp.id
            JOIN posts p ON sp.post_id = p.id
            LEFT JOIN social_platforms sp2 ON tl.network = sp2.name
            WHERE sp.user_id = ? ORDER BY tl.transmitted_at DESC LIMIT 20
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getRecentTransmissions: " . $e->getMessage());
        return [];
    }
}

function getUserAIStats($user_id) {
    global $pdo;
    
    try {
        // Vérifier si les tables IA existent
        $table_check = $pdo->query("SHOW TABLES LIKE 'user_ai_profiles'");
        if ($table_check->rowCount() == 0) {
            return [
                'profiles_count' => 0,
                'total_requests' => 0,
                'total_tokens_used' => 0,
                'total_cost' => 0,
                'avg_processing_time' => 0,
                'failed_requests' => 0,
                'today_requests' => 0,
                'today_tokens' => 0
            ];
        }
        
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(DISTINCT uap.id) as profiles_count,
                COALESCE(COUNT(DISTINCT aul.id), 0) as total_requests,
                COALESCE(SUM(aul.tokens_total), 0) as total_tokens_used,
                COALESCE(SUM(aul.cost_estimate), 0) as total_cost,
                COALESCE(AVG(aul.processing_time_ms), 0) as avg_processing_time,
                COALESCE(COUNT(CASE WHEN aul.success = 0 THEN 1 END), 0) as failed_requests,
                COALESCE(COUNT(CASE WHEN DATE(aul.created_at) = CURDATE() THEN 1 END), 0) as today_requests,
                COALESCE(SUM(CASE WHEN DATE(aul.created_at) = CURDATE() THEN aul.tokens_total ELSE 0 END), 0) as today_tokens
            FROM user_ai_profiles uap
            LEFT JOIN ai_usage_logs aul ON uap.id = aul.ai_profile_id AND aul.user_id = ?
            WHERE uap.user_id = ? AND uap.is_active = 1
        ");
        $stmt->execute([$user_id, $user_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ?: [
            'profiles_count' => 0,
            'total_requests' => 0,
            'total_tokens_used' => 0,
            'total_cost' => 0,
            'avg_processing_time' => 0,
            'failed_requests' => 0,
            'today_requests' => 0,
            'today_tokens' => 0
        ];
    } catch (Exception $e) {
        error_log("Erreur getUserAIStats: " . $e->getMessage());
        return [
            'profiles_count' => 0,
            'total_requests' => 0,
            'total_tokens_used' => 0,
            'total_cost' => 0,
            'avg_processing_time' => 0,
            'failed_requests' => 0,
            'today_requests' => 0,
            'today_tokens' => 0
        ];
    }
}

function getActiveAIQuotas($user_id) {
    global $pdo;
    
    try {
        // Vérifier si la table quotas existe
        $table_check = $pdo->query("SHOW TABLES LIKE 'user_ai_quotas'");
        if ($table_check->rowCount() == 0) {
            return [];
        }
        
        $stmt = $pdo->prepare("
            SELECT 
                quota_period,
                tokens_limit,
                COALESCE(tokens_used, 0) as tokens_used,
                cost_limit,
                COALESCE(cost_used, 0) as cost_used,
                ROUND((COALESCE(tokens_used, 0) / GREATEST(tokens_limit, 1)) * 100, 1) as usage_percentage,
                tokens_limit - COALESCE(tokens_used, 0) as tokens_remaining,
                CASE 
                    WHEN COALESCE(tokens_used, 0) >= tokens_limit THEN 'EXCEEDED'
                    WHEN (COALESCE(tokens_used, 0) / GREATEST(tokens_limit, 1)) >= 0.8 THEN 'WARNING'
                    ELSE 'OK'
                END as status,
                DATEDIFF(quota_end_date, CURDATE()) as days_remaining
            FROM user_ai_quotas 
            WHERE user_id = ? AND is_active = 1 
            AND CURDATE() BETWEEN quota_start_date AND quota_end_date
            ORDER BY quota_period DESC
            LIMIT 3
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getActiveAIQuotas: " . $e->getMessage());
        return [];
    }
}

function getUserTokenBalance($user_id) {
    global $pdo;
    
    try {
        // Vérifier si la table existe
        $table_check = $pdo->query("SHOW TABLES LIKE 'user_token_packages'");
        if ($table_check->rowCount() == 0) {
            return ['current_balance' => 0, 'total_purchased' => 0, 'total_used' => 0];
        }
        
        $stmt = $pdo->prepare("
            SELECT 
                COALESCE(SUM(tokens_purchased), 0) as total_purchased,
                COALESCE(SUM(tokens_used), 0) as total_used,
                COALESCE(SUM(tokens_purchased - tokens_used), 0) as current_balance
            FROM user_token_packages 
            WHERE user_id = ? AND status = 'active'
        ");
        $stmt->execute([$user_id]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ?: ['current_balance' => 0, 'total_purchased' => 0, 'total_used' => 0];
        
    } catch (Exception $e) {
        error_log("Erreur getUserTokenBalance: " . $e->getMessage());
        return ['current_balance' => 0, 'total_purchased' => 0, 'total_used' => 0];
    }
}

function getUserPackages($user_id) {
    global $pdo;
    
    try {
        // Vérifier si la table existe
        $table_check = $pdo->query("SHOW TABLES LIKE 'user_token_packages'");
        if ($table_check->rowCount() == 0) {
            return [];
        }
        
        $stmt = $pdo->prepare("
            SELECT 
                utp.*,
                ROUND((utp.tokens_used / GREATEST(utp.tokens_purchased, 1)) * 100, 1) as usage_percentage,
                CASE 
                    WHEN utp.tokens_used >= utp.tokens_purchased THEN 'COMPLETED'
                    WHEN (utp.tokens_used / GREATEST(utp.tokens_purchased, 1)) >= 0.8 THEN 'WARNING'
                    ELSE 'OK'
                END as package_status
            FROM user_token_packages utp
            WHERE utp.user_id = ? AND utp.status = 'active'
            ORDER BY utp.purchase_date DESC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getUserPackages: " . $e->getMessage());
        return [];
    }
}

function getClipmakerProjects($user_id) {
    global $pdo;
    
    try {
        // Vérifier si les tables Clipmaker existent
        $table_check = $pdo->query("SHOW TABLES LIKE 'clipmaker_projects'");
        if ($table_check->rowCount() == 0) {
            return [];
        }
        
        $stmt = $pdo->prepare("
            SELECT cp.*, 
                   COALESCE(COUNT(DISTINCT cm.id), 0) as media_count,
                   COALESCE(COUNT(DISTINCT ce.id), 0) as export_count,
                   MAX(ce.created_at) as last_export
            FROM clipmaker_projects cp
            LEFT JOIN clipmaker_media cm ON cp.id = cm.project_id
            LEFT JOIN clipmaker_exports ce ON cp.id = ce.project_id
            WHERE cp.user_id = ?
            GROUP BY cp.id
            ORDER BY cp.updated_at DESC
            LIMIT 5
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getClipmakerProjects: " . $e->getMessage());
        return [];
    }
}

function getUserActiveClipSession($user_id) {
    global $pdo;
    
    try {
        $table_check = $pdo->query("SHOW TABLES LIKE 'clipmaker_projects'");
        if ($table_check->rowCount() == 0) {
            return null;
        }
        
        $stmt = $pdo->prepare("
            SELECT cp.id, cp.name, cp.status, cp.updated_at,
                   COUNT(DISTINCT cm.id) as media_count
            FROM clipmaker_projects cp
            LEFT JOIN clipmaker_media cm ON cp.id = cm.project_id
            WHERE cp.user_id = ? 
            AND cp.status IN ('draft', 'editing')
            GROUP BY cp.id
            ORDER BY cp.updated_at DESC
            LIMIT 1
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getUserActiveClipSession: " . $e->getMessage());
        return null;
    }
}

function createNewClipSession($user_id, $clip_name) {
    global $pdo;
    
    try {
        // Vérifier si les tables existent
        $table_check = $pdo->query("SHOW TABLES LIKE 'clipmaker_projects'");
        if ($table_check->rowCount() == 0) {
            return ['success' => false, 'error' => 'Tables Clipmaker non disponibles'];
        }
        
        // 🚨 RÈGLE : Un seul clip actif par user
        $active_session = getUserActiveClipSession($user_id);
        if ($active_session) {
            return [
                'success' => false,
                'error' => 'Session active existante: "' . $active_session['name'] . '". Terminez-la avant d\'en créer une nouvelle.'
            ];
        }
        
        // Créer nouvelle session
        $stmt = $pdo->prepare("
            INSERT INTO clipmaker_projects (user_id, name, timeline_data, status, created_at, updated_at)
            VALUES (?, ?, '{}', 'draft', NOW(), NOW())
        ");
        
        $success = $stmt->execute([$user_id, $clip_name]);
        
        if ($success) {
            $project_id = $pdo->lastInsertId();
            return [
                'success' => true,
                'project_id' => $project_id,
                'clip_name' => $clip_name
            ];
        } else {
            return ['success' => false, 'error' => 'Erreur création session'];
        }
        
    } catch (Exception $e) {
        error_log("Erreur createNewClipSession: " . $e->getMessage());
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

function getActiveExports($user_id) {
    global $pdo;
    
    try {
        // Vérifier si les tables existent
        $table_check = $pdo->query("SHOW TABLES LIKE 'clipmaker_exports'");
        if ($table_check->rowCount() == 0) {
            return [];
        }
        
        $stmt = $pdo->prepare("
            SELECT ce.*, cp.name as project_name
            FROM clipmaker_exports ce
            JOIN clipmaker_projects cp ON ce.project_id = cp.id
            WHERE ce.user_id = ? 
            AND ce.status IN ('pending', 'processing')
            ORDER BY ce.created_at DESC
            LIMIT 3
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getActiveExports: " . $e->getMessage());
        return [];
    }
}

// Fonctions utilitaires
function formatDate($date) {
    return $date ? date('d/m/Y H:i', strtotime($date)) : 'N/A';
}

function getNetworkIcon($network) {
    $icons = [
        'instagram' => '📸', 'facebook' => '📘', 'twitter' => '🐦',
        'linkedin' => '💼', 'tiktok' => '🎵', 'snapchat' => '👻', 'youtube' => '📺'
    ];
    return $icons[$network] ?? '📱';
}

function getStatusBadge($status) {
    $badges = [
        'pending' => '<span class="badge badge-warning">⏰ En attente</span>',
        'processing' => '<span class="badge badge-info">🔄 En cours</span>',
        'sent' => '<span class="badge badge-success">✅ Envoyé</span>',
        'failed' => '<span class="badge badge-danger">❌ Échec</span>',
        'success' => '<span class="badge badge-success">✅ Succès</span>',
        'retry' => '<span class="badge badge-warning">🔄 Retry</span>'
    ];
    return $badges[$status] ?? $status;
}

// ================================================================
// 🔧 TRAITEMENT DES ACTIONS
// ================================================================

$action = $_GET['action'] ?? 'dashboard';
$message = '';
$message_type = '';

if (isset($_POST['action']) && $_POST['action'] === 'open_clipmaker') {
    $user_id = $_SESSION['user_id'];
    
    // Vérifier session active
    $active_session = getUserActiveClipSession($user_id);
    
    if ($active_session) {
        // Session existante - redirection directe
        $clipmaker_url = "https://laloee-cours.fr/social-media-generator/clipmaker_workflow_interface.html";
        $clipmaker_url .= "?user=" . urlencode($user['id']);
        $clipmaker_url .= "&project=" . $active_session['id'];
        $clipmaker_url .= "&clip_name=" . urlencode($active_session['name']);
        $clipmaker_url .= "&session=existing";
        
        header("Location: " . $clipmaker_url);
        exit;
        
    } else {
        // Pas de session - demander nom du clip
        if (isset($_POST['clip_name']) && !empty($_POST['clip_name'])) {
            $clip_name = trim($_POST['clip_name']);
            $result = createNewClipSession($user_id, $clip_name);
            
            if ($result['success']) {
                $clipmaker_url = "https://laloee-cours.fr/social-media-generator/clipmaker_workflow_interface.html";
                $clipmaker_url .= "?user=" . urlencode($user['id']);
                $clipmaker_url .= "&project=" . $result['project_id'];
                $clipmaker_url .= "&clip_name=" . urlencode($clip_name);
                $clipmaker_url .= "&session=new";
                
                header("Location: " . $clipmaker_url);
                exit;
            } else {
                $message = "❌ " . $result['error'];
                $message_type = 'error';
            }
        } else {
            // Afficher popup pour nom du clip
            $show_clip_name_popup = true;
        }
    }
}

// ================================================================
// 📊 CHARGEMENT DONNÉES DASHBOARD (APRÈS DÉFINITION DES FONCTIONS)
// ================================================================

$network_filter = $_GET['network'] ?? 'all';

// Stats utilisateur par réseau
$network_stats = getUserNetworkStats($user['id']);

// Posts disponibles pour diffusion
$available_posts = getAvailablePosts($user['id'], $network_filter);

// Posts programmés
$scheduled_posts = getScheduledPosts($user['id']);

// Logs transmission récents
$recent_transmissions = getRecentTransmissions($user['id']);

// DONNÉES IA CORRIGÉES
$ai_stats = getUserAIStats($user['id']);
$ai_quotas = getActiveAIQuotas($user['id']);

// DONNÉES PACKAGES TOKENS
$user_packages = getUserPackages($user['id']);
$user_balance = getUserTokenBalance($user['id']);

// DONNÉES CLIPMAKER  
$clipmaker_projects = getClipmakerProjects($user['id']);
$active_session = getUserActiveClipSession($user['id']);
$active_exports = getActiveExports($user['id']);

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>📊 Dashboard SMG Studio v2.0 - DÉBIT TOKENS AUTO</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }

        .main-container {
            max-width: 1600px;
            margin: 0 auto;
            padding: 20px;
        }

        .main-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 25px 35px;
            margin-bottom: 30px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 15px;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header-logo {
            width: 60px;
            height: 60px;
            background: linear-gradient(45deg, #4ecdc4, #44a08d);
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .user-badge {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
            color: white;
            padding: 10px 20px;
            border-radius: 12px;
            font-weight: 600;
            font-size: 14px;
        }

        .nav-container {
            display: flex;
            align-items: center;
            gap: 15px;
            flex-wrap: wrap;
        }

        .nav-ia-principale {
            display: flex;
            gap: 8px;
            background: rgba(255, 107, 107, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(255, 107, 107, 0.2);
        }

        .nav-ia-principale a {
            padding: 8px 16px;
            background: rgba(255, 107, 107, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .nav-ia-principale a:hover {
            background: #ff6b6b;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        .nav-ia-principale a.active {
            background: #ff6b6b;
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        /* 🆕 NAVIGATION ADMIN - Seulement si admin */
        .nav-admin {
            display: flex;
            gap: 8px;
            background: rgba(155, 89, 182, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(155, 89, 182, 0.2);
        }

        .nav-admin a {
            padding: 8px 16px;
            background: rgba(155, 89, 182, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .nav-admin a:hover {
            background: #9b59b6;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(155, 89, 182, 0.4);
        }

        .quick-nav {
            display: flex;
            gap: 8px;
        }

        .quick-nav a {
            padding: 8px 16px;
            background: rgba(255, 255, 255, 0.9);
            color: #4ecdc4;
            text-decoration: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.3s ease;
        }

        .quick-nav a:hover, .quick-nav a.active {
            background: white;
            transform: translateY(-1px);
        }

        .balance-section {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            text-align: center;
        }

        .balance-value {
            font-size: 36px;
            font-weight: 700;
            margin-bottom: 5px;
        }

        .balance-label {
            font-size: 14px;
            opacity: 0.9;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .stat-icon {
            font-size: 32px;
            margin-bottom: 10px;
            display: block;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 700;
            color: #4ecdc4;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .stat-details {
            margin-top: 10px;
            font-size: 11px;
            color: #999;
        }

        .content-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }

        .card-title {
            font-size: 20px;
            font-weight: 600;
            color: #333;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .section-actions {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
            gap: 15px;
            margin: 20px 0;
        }

        .action-card {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 20px;
            border-radius: 12px;
            text-align: center;
            text-decoration: none;
            transition: all 0.3s ease;
        }

        .action-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
        }

        .action-card.ai {
            background: linear-gradient(135deg, #ff6b6b, #ff8e53);
        }

        .action-card.admin {
            background: linear-gradient(135deg, #9b59b6, #8e44ad);
        }

        .action-card.clipmaker {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
        }

        .action-icon {
            font-size: 32px;
            margin-bottom: 10px;
            display: block;
        }

        .action-title {
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 8px;
        }

        .action-subtitle {
            font-size: 12px;
            opacity: 0.9;
        }

        .btn {
            padding: 8px 16px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            transition: all 0.3s ease;
            font-size: 13px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
            color: white;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, #ffc107, #fd7e14);
            color: white;
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
        }

        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border: 1px solid transparent;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-success {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }

        .alert-error {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }

        .ia-section {
            border-top: 4px solid #ff6b6b;
            background: linear-gradient(135deg, rgba(255, 107, 107, 0.05), rgba(255, 142, 83, 0.05));
        }

        .stat-mini {
            background: rgba(255, 107, 107, 0.1); 
            padding: 15px; 
            border-radius: 10px; 
            text-align: center;
            border: 1px solid rgba(255, 107, 107, 0.2);
        }

        .quota-bar {
            width: 100%; 
            height: 6px; 
            background: #e9ecef; 
            border-radius: 3px; 
            overflow: hidden;
        }

        @media (max-width: 768px) {
            .main-container {
                padding: 15px;
            }
            
            .main-header {
                flex-direction: column;
                text-align: center;
            }
            
            .nav-container {
                justify-content: center;
            }
            
            .nav-ia-principale, .nav-admin {
                flex-wrap: wrap;
                justify-content: center;
            }
            
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }

            .section-actions {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header avec Navigation Complète -->
        <header class="main-header">
            <div class="header-left">
                <div class="header-logo">📊</div>
                <div>
                    <h1>SMG Studio v2.0</h1>
                    <p>Dashboard Principal - Administration Complète</p>
                </div>
            </div>
            
            <div class="user-badge">
                <?= htmlspecialchars($user['name']) ?> - <?= ucfirst($user['role']) ?>
            </div>
            
            <div class="nav-container">
                <!-- NAVIGATION IA PRINCIPALE -->
                <div class="nav-ia-principale">
                    <a href="dashboard.php" class="active">
                        📊 Dashboard Principal
                    </a>
                    <a href="ai_content_generator.php">
                        🎨 IA Générateur
                    </a>
                    <a href="ai_dashboard.php">
                        📊 IA Dashboard
                    </a>
                    <a href="ai_profiles.php">
                        ⚙️ IA Profils
                    </a>
                </div>
                
                <!-- 🆕 NAVIGATION ADMIN (si admin) -->
                <?php if ($user['role'] === 'admin'): ?>
                    <div class="nav-admin">
                        <a href="users_management.php">
                            👥 Gestion Utilisateurs
                        </a>
                        <a href="tokens_management.php">
                            💰 Gestion Tokens
                        </a>
                        <a href="system_admin.php">
                            ⚙️ Administration Système
                        </a>
                    </div>
                <?php endif; ?>
                
                <!-- Navigation classique -->
                <div class="quick-nav">
                    <a href="posts.php">📝 Posts</a>
                    <a href="images.php">📸 Médias</a>
                    <a href="logout.php">🚪 Sortie</a>
                </div>
            </div>
        </header>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type ?>">
                <span><?= $message ?></span>
            </div>
        <?php endif; ?>

        <!-- Balance Tokens si packages disponibles -->
        <?php if ($user_balance && $user_balance['current_balance'] > 0): ?>
            <div class="balance-section">
                <div class="balance-value"><?= number_format($user_balance['current_balance']) ?></div>
                <div class="balance-label">💰 Tokens disponibles</div>
                <div style="font-size: 12px; margin-top: 8px; opacity: 0.8;">
                    <?= number_format($user_balance['total_used']) ?> utilisés sur <?= number_format($user_balance['total_purchased']) ?> achetés
                </div>
            </div>
        <?php endif; ?>

        <!-- Stats principales -->
        <div class="stats-grid">
            <!-- Posts Stats -->
            <?php if (!empty($network_stats)): ?>
                <?php foreach ($network_stats as $stat): ?>
                    <div class="stat-card">
                        <span class="stat-icon"><?= getNetworkIcon($stat['network']) ?></span>
                        <div class="stat-value"><?= $stat['total_posts'] ?></div>
                        <div class="stat-label"><?= htmlspecialchars($stat['display_name'] ?? ucfirst($stat['network'])) ?></div>
                        <div class="stat-details">
                            <?= $stat['scheduled_posts'] ?? 0 ?> programmés • <?= $stat['sent_posts'] ?? 0 ?> envoyés
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>

            <!-- IA Stats -->
            <div class="stat-card" style="border-left: 4px solid #ff6b6b;">
                <span class="stat-icon">🤖</span>
                <div class="stat-value"><?= $ai_stats['profiles_count'] ?? 0 ?></div>
                <div class="stat-label">Profils IA</div>
                <div class="stat-details">
                    <?= number_format($ai_stats['total_tokens_used'] ?? 0) ?> tokens utilisés
                </div>
            </div>

            <!-- Package Stats -->
            <div class="stat-card" style="border-left: 4px solid #28a745;">
                <span class="stat-icon">💰</span>
                <div class="stat-value"><?= count($user_packages) ?></div>
                <div class="stat-label">Packages Actifs</div>
                <div class="stat-details">
                    <?= number_format($user_balance['current_balance'] ?? 0) ?> tokens restants
                </div>
            </div>

            <!-- Clipmaker Stats -->
            <div class="stat-card" style="border-left: 4px solid #4ecdc4;">
                <span class="stat-icon">🎬</span>
                <div class="stat-value"><?= count($clipmaker_projects) ?></div>
                <div class="stat-label">Projets Clipmaker</div>
                <div class="stat-details">
                    <?= count($active_exports) ?> exports en cours
                </div>
            </div>
        </div>

        <!-- Actions rapides -->
        <div class="content-card">
            <div class="card-header">
                <h2 class="card-title">🚀 Actions Rapides</h2>
            </div>
            
            <div class="section-actions">
                <!-- Actions utilisateur classiques -->
                <a href="posts.php?action=new" class="action-card">
                    <span class="action-icon">📝</span>
                    <div class="action-title">Nouveau Post</div>
                    <div class="action-subtitle">Créer du contenu pour vos réseaux</div>
                </a>
                
                <a href="ai_content_generator.php" class="action-card ai">
                    <span class="action-icon">🎨</span>
                    <div class="action-title">Générateur IA</div>
                    <div class="action-subtitle">Contenu intelligent multi-réseaux</div>
                </a>
                
                <a href="https://laloee-cours.fr/social-media-generator/clipmaker_workflow_interface.html?user=<?= urlencode($user['id']) ?>" 
                   target="_blank" class="action-card clipmaker">
                    <span class="action-icon">🎬</span>
                    <div class="action-title">Clipmaker Pro</div>
                    <div class="action-subtitle">Montage vidéo professionnel</div>
                </a>
                
                <a href="ai_profiles.php" class="action-card">
                    <span class="action-icon">⚙️</span>
                    <div class="action-title">Config IA</div>
                    <div class="action-subtitle">Gérer profils et quotas</div>
                </a>
                
                <a href="ai_dashboard.php" class="action-card">
                    <span class="action-icon">📊</span>
                    <div class="action-title">Analytics IA</div>
                    <div class="action-subtitle">Suivi tokens temps réel</div>
                </a>
                
                <!-- 🆕 ACTIONS ADMIN (si admin) -->
                <?php if ($user['role'] === 'admin'): ?>
                    <a href="users_management.php" class="action-card admin">
                        <span class="action-icon">👥</span>
                        <div class="action-title">Gestion Utilisateurs</div>
                        <div class="action-subtitle">CRUD complet + attribution tokens</div>
                    </a>
                    
                    <a href="tokens_management.php" class="action-card admin">
                        <span class="action-icon">💰</span>
                        <div class="action-title">Gestion Tokens</div>
                        <div class="action-subtitle">Packages + quotas + analytics</div>
                    </a>
                    
                    <a href="system_admin.php" class="action-card admin">
                        <span class="action-icon">⚙️</span>
                        <div class="action-title">Administration Système</div>
                        <div class="action-subtitle">Config + maintenance + monitoring</div>
                    </a>
                <?php endif; ?>
                
                <a href="images.php" class="action-card">
                    <span class="action-icon">📸</span>
                    <div class="action-title">Médias</div>
                    <div class="action-subtitle">Bibliothèque d'images</div>
                </a>
            </div>
        </div>

        <!-- Section IA Principale -->
        <div class="content-card ia-section">
            <div class="card-header">
                <h2 class="card-title">
                    🤖 Intelligence Artificielle - Hub Central
                    <span style="background: #ff6b6b; color: white; padding: 4px 12px; border-radius: 15px; font-size: 14px; margin-left: 10px;">
                        DÉBIT AUTO
                    </span>
                </h2>
                <div style="display: flex; gap: 10px;">
                    <a href="ai_content_generator.php" class="btn btn-primary">🎨 Générateur</a>
                    <a href="ai_dashboard.php" class="btn btn-secondary">📊 Dashboard IA</a>
                    <a href="ai_profiles.php" class="btn btn-warning">⚙️ Profils</a>
                </div>
            </div>
            
            <!-- Stats IA rapides -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-bottom: 20px;">
                <div class="stat-mini">
                    <div style="font-size: 20px; font-weight: 700; color: #ff6b6b; margin-bottom: 5px;">
                        <?= $ai_stats['profiles_count'] ?? 0 ?>
                    </div>
                    <div style="font-size: 12px; color: #666; text-transform: uppercase;">Profils IA</div>
                </div>
                <div class="stat-mini">
                    <div style="font-size: 20px; font-weight: 700; color: #ff6b6b; margin-bottom: 5px;">
                        <?= number_format($ai_stats['total_requests'] ?? 0) ?>
                    </div>
                    <div style="font-size: 12px; color: #666; text-transform: uppercase;">Requêtes IA</div>
                </div>
                <div class="stat-mini">
                    <div style="font-size: 20px; font-weight: 700; color: #ff6b6b; margin-bottom: 5px;">
                        <?= $ai_stats['today_requests'] ?? 0 ?>
                    </div>
                    <div style="font-size: 12px; color: #666; text-transform: uppercase;">Aujourd'hui</div>
                </div>
                <div class="stat-mini">
                    <div style="font-size: 20px; font-weight: 700; color: #ff6b6b; margin-bottom: 5px;">
                        <?= number_format($ai_stats['total_tokens_used'] ?? 0) ?>
                    </div>
                    <div style="font-size: 12px; color: #666; text-transform: uppercase;">Tokens</div>
                </div>
                <div class="stat-mini">
                    <div style="font-size: 20px; font-weight: 700; color: #ff6b6b; margin-bottom: 5px;">
                        <?= number_format($ai_stats['total_cost'] ?? 0, 2) ?>€
                    </div>
                    <div style="font-size: 12px; color: #666; text-transform: uppercase;">Coût total</div>
                </div>
            </div>
            
            <!-- Quotas IA si configurés -->
            <?php if (!empty($ai_quotas)): ?>
                <div style="margin-bottom: 20px;">
                    <h4 style="color: #333; margin-bottom: 15px; font-size: 16px;">📊 Quotas Actifs</h4>
                    
                    <?php foreach ($ai_quotas as $quota): ?>
                        <div style="margin-bottom: 10px; padding: 12px; background: #f8f9fa; border-radius: 8px; border-left: 4px solid <?= 
                            $quota['status'] === 'OK' ? '#28a745' : 
                            ($quota['status'] === 'WARNING' ? '#ffc107' : '#dc3545') 
                        ?>;">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                                <strong><?= ucfirst($quota['quota_period']) ?></strong>
                                <span style="background: <?= 
                                    $quota['status'] === 'OK' ? '#d4edda' : 
                                    ($quota['status'] === 'WARNING' ? '#fff3cd' : '#f8d7da') 
                                ?>; color: <?= 
                                    $quota['status'] === 'OK' ? '#155724' : 
                                    ($quota['status'] === 'WARNING' ? '#856404' : '#721c24') 
                                ?>; padding: 2px 8px; border-radius: 10px; font-size: 11px; font-weight: 600;">
                                    <?= $quota['usage_percentage'] ?>%
                                </span>
                            </div>
                            <div class="quota-bar">
                                <div style="height: 100%; width: <?= min(100, $quota['usage_percentage']) ?>%; background: <?= 
                                    $quota['status'] === 'OK' ? '#28a745' : 
                                    ($quota['status'] === 'WARNING' ? '#ffc107' : '#dc3545') 
                                ?>; transition: width 0.3s ease;"></div>
                            </div>
                            <div style="display: flex; justify-content: space-between; font-size: 11px; color: #666; margin-top: 5px;">
                                <span><?= number_format($quota['tokens_used']) ?> / <?= number_format($quota['tokens_limit']) ?> tokens</span>
                                <span><?= $quota['days_remaining'] ?> jours restants</span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            
            <!-- Message si pas de profils IA -->
            <?php if (($ai_stats['profiles_count'] ?? 0) === 0): ?>
                <div style="text-align: center; padding: 30px; background: #f8f9fa; border-radius: 10px; margin-top: 20px; border: 2px dashed #dee2e6;">
                    <div style="font-size: 48px; margin-bottom: 15px;">🤖</div>
                    <h4 style="color: #333; margin-bottom: 10px;">Démarrez avec l'IA</h4>
                    <p style="color: #666; margin-bottom: 20px;">
                        Configurez votre premier profil IA pour commencer à générer du contenu intelligent avec débit automatique des tokens
                    </p>
                    <a href="ai_profiles.php" class="btn btn-primary">
                        ➕ Créer mon premier profil IA
                    </a>
                </div>
            <?php endif; ?>
        </div>

        <!-- Posts disponibles (version simplifiée) -->
        <?php if (!empty($available_posts)): ?>
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">
                        📋 Posts Disponibles
                        <span style="background: #4ecdc4; color: white; padding: 4px 12px; border-radius: 15px; font-size: 14px; margin-left: 10px;">
                            <?= count($available_posts) ?>
                        </span>
                    </h2>
                    <a href="posts.php" class="btn btn-primary">🔧 Gérer Posts</a>
                </div>
                
                <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 15px;">
                    <?php foreach (array_slice($available_posts, 0, 6) as $post): ?>
                        <div style="background: white; border-radius: 10px; padding: 15px; border: 1px solid #e9ecef;">
                            <h4 style="margin-bottom: 8px; color: #333;">
                                <?= htmlspecialchars($post['titre']) ?>
                            </h4>
                            <p style="color: #666; font-size: 14px; margin-bottom: 10px;">
                                <?= substr(htmlspecialchars($post['contenu']), 0, 100) ?>...
                            </p>
                            <div style="display: flex; justify-content: space-between; align-items: center;">
                                <small style="color: #999;">
                                    <?= count(explode(',', $post['networks'] ?? '')) ?> réseau(x)
                                </small>
                                <a href="posts.php?action=edit&id=<?= $post['id'] ?>" class="btn btn-primary btn-sm">
                                    ✏️ Éditer
                                </a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Popup Clipmaker -->
        <?php if (isset($show_clip_name_popup) && $show_clip_name_popup): ?>
        <div id="clipNamePopup" style="
            position: fixed; top: 0; left: 0; right: 0; bottom: 0; 
            background: rgba(0,0,0,0.8); z-index: 10000; 
            display: flex; align-items: center; justify-content: center;
        ">
            <div style="
                background: white; border-radius: 15px; padding: 30px; 
                max-width: 400px; width: 90%; box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            ">
                <div style="text-align: center; margin-bottom: 20px;">
                    <div style="font-size: 48px; margin-bottom: 15px;">🎬</div>
                    <h3 style="color: #333; margin-bottom: 10px;">Nouveau Clip Clipmaker</h3>
                    <p style="color: #666; font-size: 14px;">
                        Donnez un nom à votre nouveau projet de montage
                    </p>
                </div>
                
                <form method="POST" style="margin-bottom: 20px;">
                    <input type="hidden" name="action" value="open_clipmaker">
                    <div style="margin-bottom: 15px;">
                        <label style="display: block; color: #333; font-weight: 600; margin-bottom: 8px;">
                            📝 Nom du Clip
                        </label>
                        <input type="text" name="clip_name" required
                               placeholder="Ex: Interview Jean Dupont"
                               style="
                                   width: 100%; padding: 12px; border: 2px solid #e9ecef; 
                                   border-radius: 8px; font-size: 16px;
                               ">
                    </div>
                    
                    <div style="display: flex; gap: 10px;">
                        <button type="submit" style="
                            flex: 1; padding: 12px; background: linear-gradient(135deg, #4ecdc4, #44a08d); 
                            color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer;
                        ">
                            🚀 Créer & Ouvrir
                        </button>
                        <button type="button" onclick="closeClipPopup()" style="
                            padding: 12px 20px; background: #6c757d; color: white; 
                            border: none; border-radius: 8px; cursor: pointer;
                        ">
                            ❌ Annuler
                        </button>
                    </div>
                </form>
                
                <div style="text-align: center; font-size: 12px; color: #999;">
                    ⚠️ Un seul clip à la fois par utilisateur
                </div>
            </div>
        </div>

        <script>
        function closeClipPopup() {
            document.getElementById('clipNamePopup').style.display = 'none';
        }
        </script>
        <?php endif; ?>

        <!-- Message si vide -->
        <?php if (empty($available_posts) && empty($clipmaker_projects) && ($ai_stats['profiles_count'] ?? 0) == 0): ?>
            <div class="content-card" style="text-align: center; padding: 60px;">
                <div style="font-size: 64px; margin-bottom: 20px;">🚀</div>
                <h3>Bienvenue dans SMG Studio v2.0 !</h3>
                <p style="color: #666; margin: 20px 0;">
                    Créez du contenu, utilisez l'IA avec débit automatique et montez des vidéos comme un pro
                </p>
                <div style="display: flex; gap: 15px; justify-content: center;">
                    <a href="posts.php?action=new" class="btn btn-primary">📝 Premier Post</a>
                    <a href="ai_profiles.php" class="btn btn-secondary">🤖 Configurer IA</a>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script>
        console.log('📊 SMG Studio v2.0 Dashboard Ready!');
        console.log('✅ Navigation IA + Admin Corrigée par Papy 77');
        console.log('🎯 Dashboard Principal → IA Modules + Admin');
        console.log('🔄 Navigation bidirectionnelle opérationnelle');
        console.log('💰 Système débit tokens automatique intégré');
        console.log('🛠️ Toutes les erreurs variables corrigées !');
    </script>
</body>
</html>