<?php
/*
 * SMG Studio v2.0 - Social Media Generator - Audiovizu Edition
 * Configuration centrale - Isolation utilisateur & sécurité
 * Path: /var/www/laloee-cours.fr/social-media-generator/config/smg_config.php
 */

// Empêcher accès direct
if (!defined('SMG_LOADED')) {
    die('Accès direct interdit');
}

/**
 * Configuration générale SMG Studio v2.0
 */
class SMGConfig {
    
    // Version et identification
    const VERSION = '2.0.0';
    const NAME = 'SMG Studio - Social Media Generator';
    const EDITION = 'Audiovizu Edition';
    const AUTHOR = 'PAPY 77';
    
    // Chemins système
    const BASE_PATH = '/var/www/laloee-cours.fr/social-media-generator/';
    const ASSETS_PATH = '/assets/';
    const CSS_PATH = '/assets/css/';
    const JS_PATH = '/assets/js/';
    const API_PATH = '/api/';
    const USER_WORKSPACES_PATH = '/user_workspaces/';
    
    // Configuration isolation utilisateur
    const MAX_USERS = 100;
    const USER_QUOTA_MB = 1000; // 1GB par utilisateur
    const SESSION_TIMEOUT = 3600; // 1 heure
    const LOG_RETENTION_DAYS = 30;
    
    // Workflow stages
    const WORKFLOW_STAGES = [
        1 => [
            'name' => 'Montage Vidéo',
            'file' => 'clipmaker.html',
            'icon' => '🎬',
            'description' => 'Découpe, assemblage et montage des rushes vidéo'
        ],
        2 => [
            'name' => 'Post-Production Audio',
            'file' => 'smg_audio_postprod.html',
            'icon' => '🎵',
            'description' => 'Mixage, mastering et synchronisation audio'
        ],
        3 => [
            'name' => 'Émission Finale',
            'file' => 'smg_emission_finale.html',
            'icon' => '📺',
            'description' => 'Finalisation et encodage pour diffusion'
        ],
        4 => [
            'name' => 'Social Ready',
            'file' => 'smg_social_ready.html',
            'icon' => '📱',
            'description' => 'Adaptation et diffusion réseaux sociaux'
        ]
    ];
    
    // Plateformes sociales supportées
    const SOCIAL_PLATFORMS = [
        'youtube' => [
            'name' => 'YouTube',
            'icon' => '▶️',
            'color' => '#ff0000',
            'formats' => [
                'standard' => ['width' => 1920, 'height' => 1080, 'ratio' => '16:9'],
                'shorts' => ['width' => 1080, 'height' => 1920, 'ratio' => '9:16']
            ],
            'max_duration' => 0, // Illimité
            'max_size_mb' => 128000 // 128GB
        ],
        'instagram' => [
            'name' => 'Instagram',
            'icon' => '📷',
            'color' => '#e4405f',
            'formats' => [
                'post' => ['width' => 1080, 'height' => 1080, 'ratio' => '1:1'],
                'story' => ['width' => 1080, 'height' => 1920, 'ratio' => '9:16'],
                'reel' => ['width' => 1080, 'height' => 1920, 'ratio' => '9:16']
            ],
            'max_duration' => 60,
            'max_size_mb' => 100
        ],
        'tiktok' => [
            'name' => 'TikTok',
            'icon' => '🎵',
            'color' => '#000000',
            'formats' => [
                'video' => ['width' => 1080, 'height' => 1920, 'ratio' => '9:16']
            ],
            'max_duration' => 180,
            'max_size_mb' => 72
        ],
        'facebook' => [
            'name' => 'Facebook',
            'icon' => '📘',
            'color' => '#1877f2',
            'formats' => [
                'post' => ['width' => 1920, 'height' => 1080, 'ratio' => '16:9'],
                'story' => ['width' => 1080, 'height' => 1920, 'ratio' => '9:16']
            ],
            'max_duration' => 14400, // 4h
            'max_size_mb' => 4000
        ],
        'linkedin' => [
            'name' => 'LinkedIn',
            'icon' => '💼',
            'color' => '#0077b5',
            'formats' => [
                'post' => ['width' => 1920, 'height' => 1080, 'ratio' => '16:9']
            ],
            'max_duration' => 600, // 10min
            'max_size_mb' => 200
        ]
    ];
    
    // Formats audio supportés
    const AUDIO_FORMATS = [
        'input' => ['mp3', 'wav', 'm4a', 'aac', 'flac'],
        'output' => ['mp3', 'wav', 'aac'],
        'kb11_compatible' => ['wav', 'aac']
    ];
    
    // Formats vidéo supportés
    const VIDEO_FORMATS = [
        'input' => ['mp4', 'mov', 'avi', 'mkv', 'webm'],
        'output' => ['mp4', 'webm'],
        'preferred' => 'mp4'
    ];
    
    // Formats image supportés
    const IMAGE_FORMATS = [
        'input' => ['jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp'],
        'output' => ['jpg', 'png', 'webp'],
        'preferred' => 'jpg'
    ];
    
    // Configuration sécurité
    const SECURITY = [
        'allowed_domains' => ['laloee-cours.fr', 'localhost'],
        'max_upload_size' => 100 * 1024 * 1024, // 100MB
        'allowed_ips' => [], // Vide = tous autorisés
        'rate_limit_requests' => 100, // Par minute
        'csrf_protection' => true,
        'sanitize_inputs' => true
    ];
    
    // Configuration base de données (future)
    const DATABASE = [
        'enabled' => false, // Pour v2.0, utilisation fichiers JSON
        'host' => 'localhost',
        'name' => 'smg_studio',
        'user' => 'smg_user',
        'charset' => 'utf8mb4'
    ];
    
    // Configuration logs
    const LOGGING = [
        'enabled' => true,
        'level' => 'INFO', // DEBUG, INFO, WARNING, ERROR
        'max_file_size_mb' => 10,
        'compress_old_logs' => true,
        'log_user_actions' => true,
        'log_security_events' => true,
        'log_api_calls' => true
    ];
    
    // Configuration cache
    const CACHE = [
        'enabled' => true,
        'ttl_seconds' => 3600, // 1 heure
        'max_size_mb' => 50,
        'compress' => true
    ];
    
    /**
     * Méthodes utilitaires
     */
    
    public static function getVersion() {
        return self::VERSION;
    }
    
    public static function getFullName() {
        return self::NAME . ' v' . self::VERSION . ' - ' . self::EDITION;
    }
    
    public static function getWorkflowStage($stageNumber) {
        return self::WORKFLOW_STAGES[$stageNumber] ?? null;
    }
    
    public static function getSocialPlatform($platform) {
        return self::SOCIAL_PLATFORMS[$platform] ?? null;
    }
    
    public static function getUserWorkspacePath($userId) {
        return self::BASE_PATH . 'user_workspaces/' . $userId . '/';
    }
    
    public static function validateUserId($userId) {
        return preg_match('/^user_\d{3}$/', $userId);
    }
    
    public static function generateUserId() {
        // En production, récupérer depuis DB
        $existing = glob(self::BASE_PATH . 'user_workspaces/user_*');
        $maxId = 0;
        
        foreach ($existing as $dir) {
            if (preg_match('/user_(\d{3})$/', basename($dir), $matches)) {
                $maxId = max($maxId, intval($matches[1]));
            }
        }
        
        return sprintf('user_%03d', $maxId + 1);
    }
    
    public static function isFileTypeAllowed($filename, $type = 'video') {
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        switch ($type) {
            case 'video':
                return in_array($extension, self::VIDEO_FORMATS['input']);
            case 'audio':
                return in_array($extension, self::AUDIO_FORMATS['input']);
            case 'image':
                return in_array($extension, self::IMAGE_FORMATS['input']);
            default:
                return false;
        }
    }
    
    public static function formatFileSize($bytes) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        
        $bytes /= pow(1024, $pow);
        
        return round($bytes, 2) . ' ' . $units[$pow];
    }
    
    public static function generateFilename($userId, $stage, $project = '', $extension = 'mp4') {
        $timestamp = date('Y-m-d_H-i-s');
        $projectSlug = $project ? preg_replace('/[^a-zA-Z0-9]/', '_', $project) : 'untitled';
        $projectSlug = substr($projectSlug, 0, 20);
        
        $stageNames = [
            1 => 'CLIP',
            2 => 'AUDIO', 
            3 => 'EMISSION',
            4 => 'SOCIAL'
        ];
        
        $stageName = $stageNames[$stage] ?? 'FILE';
        
        return "{$stageName}_{$projectSlug}_{$timestamp}.{$extension}";
    }
    
    public static function createUserWorkspace($userId) {
        if (!self::validateUserId($userId)) {
            throw new Exception('User ID invalide');
        }
        
        $basePath = self::getUserWorkspacePath($userId);
        
        // Créer structure de dossiers
        $directories = [
            '',
            '01_clipmaker',
            '02_audio_postprod', 
            '03_emission_final',
            '04_social_ready',
            '04_social_ready/youtube',
            '04_social_ready/instagram',
            '04_social_ready/tiktok',
            '04_social_ready/facebook',
            '04_social_ready/linkedin',
            'logs',
            'temp',
            'archives'
        ];
        
        foreach ($directories as $dir) {
            $fullPath = $basePath . $dir;
            if (!is_dir($fullPath)) {
                mkdir($fullPath, 0755, true);
            }
        }
        
        // Créer fichiers de configuration
        $configFile = $basePath . 'config.json';
        if (!file_exists($configFile)) {
            $config = [
                'user_id' => $userId,
                'created_at' => date('Y-m-d H:i:s'),
                'quota_mb' => self::USER_QUOTA_MB,
                'settings' => [
                    'auto_save' => true,
                    'notifications' => true,
                    'theme' => 'default'
                ]
            ];
            
            file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT));
        }
        
        return $basePath;
    }
    
    public static function getDirectorySize($directory) {
        $size = 0;
        if (is_dir($directory)) {
            foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($directory)) as $file) {
                if ($file->isFile()) {
                    $size += $file->getSize();
                }
            }
        }
        return $size;
    }
    
    public static function checkUserQuota($userId) {
        $workspace = self::getUserWorkspacePath($userId);
        if (!is_dir($workspace)) {
            return ['used' => 0, 'quota' => self::USER_QUOTA_MB * 1024 * 1024, 'available' => self::USER_QUOTA_MB * 1024 * 1024];
        }
        
        $used = self::getDirectorySize($workspace);
        $quota = self::USER_QUOTA_MB * 1024 * 1024;
        $available = max(0, $quota - $used);
        
        return [
            'used' => $used,
            'quota' => $quota,
            'available' => $available,
            'used_percentage' => round(($used / $quota) * 100, 2)
        ];
    }
    
    public static function cleanupOldLogs($userId) {
        $logsDir = self::getUserWorkspacePath($userId) . 'logs/';
        if (!is_dir($logsDir)) {
            return;
        }
        
        $cutoffDate = date('Y-m-d', strtotime('-' . self::LOG_RETENTION_DAYS . ' days'));
        
        foreach (glob($logsDir . '*.log') as $logFile) {
            $filename = basename($logFile, '.log');
            if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $filename) && $filename < $cutoffDate) {
                if (self::LOGGING['compress_old_logs']) {
                    // Compresser avant suppression
                    $compressed = gzencode(file_get_contents($logFile));
                    file_put_contents($logFile . '.gz', $compressed);
                }
                unlink($logFile);
            }
        }
    }
    
    public static function logSystemEvent($event, $data = []) {
        if (!self::LOGGING['enabled']) {
            return;
        }
        
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'event' => $event,
            'data' => $data,
            'memory_usage' => memory_get_usage(true),
            'peak_memory' => memory_get_peak_usage(true)
        ];
        
        $logFile = self::BASE_PATH . 'logs/system_' . date('Y-m-d') . '.log';
        file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND | LOCK_EX);
    }
}

// Définir constante de chargement
define('SMG_LOADED', true);

// Auto-créer dossiers logs système si nécessaire
$systemLogsDir = SMGConfig::BASE_PATH . 'logs/';
if (!is_dir($systemLogsDir)) {
    mkdir($systemLogsDir, 0755, true);
}

// Log de démarrage système
SMGConfig::logSystemEvent('config_loaded', [
    'version' => SMGConfig::VERSION,
    'php_version' => PHP_VERSION,
    'timestamp' => time()
]);