<?php
/**
 * ================================================================
 * 🛠️ SMG STUDIO v2.0 - FONCTIONS COMMUNES CENTRALISÉES
 * ================================================================
 * 
 * 🎯 Objectif : Éviter les doublons, centraliser le code
 * ✅ Toutes les fonctions avec protection function_exists()
 * 🔧 Compatible avec tous tes modules IA
 * 
 * 💡 BY PAPY 77 - ARCHITECTURE PROPRE !
 * ================================================================
 */

// ================================================================
// 🎨 FONCTIONS PROVIDERS IA
// ================================================================

if (!function_exists('getProviderIcon')) {
    function getProviderIcon($provider_key) {
        static $icons = [
            'smg_free' => '🆓',
            'openai' => '🤖',
            'claude' => '🧠',
            'gemini' => '💎',
            'huggingface' => '🤗',
            'local' => '🏠',
            'anthropic' => '🧠',
            'google' => '💎',
            'mistral' => '🌊',
            'cohere' => '⚡',
            'stability' => '🎨',
            'palm' => '🌴',
            'llama' => '🦙',
            'gpt4' => '🤖',
            'chatgpt' => '💬'
        ];
        return $icons[$provider_key] ?? '🔮';
    }
}

if (!function_exists('getProviderName')) {
    function getProviderName($provider_key) {
        static $names = [
            'smg_free' => 'SMG Free',
            'openai' => 'OpenAI',
            'claude' => 'Claude (Anthropic)',
            'gemini' => 'Gemini (Google)',
            'huggingface' => 'Hugging Face',
            'local' => 'Local Model',
            'anthropic' => 'Anthropic',
            'google' => 'Google AI',
            'mistral' => 'Mistral AI',
            'cohere' => 'Cohere',
            'stability' => 'Stability AI'
        ];
        return $names[$provider_key] ?? ucfirst($provider_key);
    }
}

// ================================================================
// 💰 FONCTIONS TOKENS ET COÛTS
// ================================================================

if (!function_exists('formatTokens')) {
    function formatTokens($tokens) {
        if ($tokens >= 1000000) {
            return round($tokens / 1000000, 1) . 'M';
        } elseif ($tokens >= 1000) {
            return round($tokens / 1000, 1) . 'k';
        }
        return number_format($tokens);
    }
}

if (!function_exists('calculateOpenAICost')) {
    function calculateOpenAICost($tokens, $model) {
        $pricing = [
            'gpt-3.5-turbo' => 0.002,
            'gpt-4' => 0.03,
            'gpt-4o' => 0.005,
            'gpt-4o-mini' => 0.00015,
            'gpt-4-turbo' => 0.01
        ];
        
        $price_per_1k = $pricing[$model] ?? 0.002;
        return ($tokens / 1000) * $price_per_1k;
    }
}

if (!function_exists('calculateProviderCost')) {
    function calculateProviderCost($tokens, $provider, $model = null) {
        $pricing = [
            'openai' => [
                'gpt-3.5-turbo' => 0.002,
                'gpt-4' => 0.03,
                'gpt-4o' => 0.005,
                'gpt-4o-mini' => 0.00015
            ],
            'claude' => [
                'claude-3-sonnet' => 0.003,
                'claude-3-opus' => 0.015,
                'claude-3-haiku' => 0.00025
            ],
            'gemini' => [
                'gemini-pro' => 0.001,
                'gemini-ultra' => 0.002
            ],
            'smg_free' => 0
        ];
        
        if ($provider === 'smg_free') return 0;
        
        $provider_rates = $pricing[$provider] ?? ['default' => 0.002];
        $rate = $provider_rates[$model] ?? $provider_rates['default'] ?? 0.002;
        
        return ($tokens / 1000) * $rate;
    }
}

// ================================================================
// 📊 FONCTIONS QUOTAS ET STATUS
// ================================================================

if (!function_exists('getQuotaStatus')) {
    function getQuotaStatus($usage_percentage) {
        if ($usage_percentage >= 100) return 'EXCEEDED';
        if ($usage_percentage >= 90) return 'CRITICAL';
        if ($usage_percentage >= 80) return 'WARNING';
        if ($usage_percentage >= 60) return 'NORMAL';
        return 'OK';
    }
}

if (!function_exists('getQuotaStatusColor')) {
    function getQuotaStatusColor($status) {
        $colors = [
            'OK' => '#28a745',
            'NORMAL' => '#17a2b8', 
            'WARNING' => '#ffc107',
            'CRITICAL' => '#fd7e14',
            'EXCEEDED' => '#dc3545'
        ];
        return $colors[$status] ?? '#6c757d';
    }
}

if (!function_exists('getQuotaStatusBadge')) {
    function getQuotaStatusBadge($status) {
        $badges = [
            'OK' => '<span class="badge badge-success">✅ OK</span>',
            'NORMAL' => '<span class="badge badge-info">📊 Normal</span>',
            'WARNING' => '<span class="badge badge-warning">⚠️ Attention</span>',
            'CRITICAL' => '<span class="badge badge-danger">🚨 Critique</span>',
            'EXCEEDED' => '<span class="badge badge-danger">❌ Dépassé</span>'
        ];
        return $badges[$status] ?? '<span class="badge badge-secondary">' . $status . '</span>';
    }
}

// ================================================================
// 📅 FONCTIONS DATES ET FORMATS
// ================================================================

if (!function_exists('formatDate')) {
    function formatDate($date, $format = 'd/m/Y H:i') {
        if (empty($date)) return 'N/A';
        return date($format, strtotime($date));
    }
}

if (!function_exists('getTimeAgo')) {
    function getTimeAgo($date) {
        if (empty($date)) return 'Jamais';
        
        $time_diff = time() - strtotime($date);
        
        if ($time_diff < 60) return 'À l\'instant';
        if ($time_diff < 3600) return floor($time_diff / 60) . ' min';
        if ($time_diff < 86400) return floor($time_diff / 3600) . 'h';
        if ($time_diff < 2592000) return floor($time_diff / 86400) . 'j';
        
        return formatDate($date, 'd/m/Y');
    }
}

// ================================================================
// 🌐 FONCTIONS RÉSEAUX SOCIAUX
// ================================================================

if (!function_exists('getNetworkIcon')) {
    function getNetworkIcon($network) {
        $icons = [
            'instagram' => '📸',
            'facebook' => '📘', 
            'twitter' => '🐦',
            'linkedin' => '💼',
            'tiktok' => '🎵',
            'snapchat' => '👻',
            'youtube' => '📺',
            'pinterest' => '📌',
            'discord' => '💬',
            'telegram' => '📤'
        ];
        return $icons[$network] ?? '📱';
    }
}

if (!function_exists('getStatusBadge')) {
    function getStatusBadge($status) {
        $badges = [
            'pending' => '<span class="badge badge-warning">⏰ En attente</span>',
            'processing' => '<span class="badge badge-info">🔄 En cours</span>',
            'sent' => '<span class="badge badge-success">✅ Envoyé</span>',
            'failed' => '<span class="badge badge-danger">❌ Échec</span>',
            'success' => '<span class="badge badge-success">✅ Succès</span>',
            'retry' => '<span class="badge badge-warning">🔄 Retry</span>',
            'active' => '<span class="badge badge-success">✅ Actif</span>',
            'inactive' => '<span class="badge badge-secondary">⏸️ Inactif</span>'
        ];
        return $badges[$status] ?? '<span class="badge badge-secondary">' . ucfirst($status) . '</span>';
    }
}

// ================================================================
// 🛡️ FONCTIONS VALIDATION ET SÉCURITÉ
// ================================================================

if (!function_exists('validateTokenAmount')) {
    function validateTokenAmount($tokens) {
        $tokens = intval($tokens);
        return max(1, min(100000, $tokens)); // Entre 1 et 100k
    }
}

if (!function_exists('validateTemperature')) {
    function validateTemperature($temp) {
        $temp = floatval($temp);
        return max(0.0, min(2.0, $temp)); // Entre 0.0 et 2.0
    }
}

if (!function_exists('sanitizePrompt')) {
    function sanitizePrompt($prompt) {
        // Nettoyer le prompt tout en gardant les emojis
        $prompt = trim($prompt);
        $prompt = strip_tags($prompt);
        $prompt = htmlspecialchars_decode($prompt);
        return substr($prompt, 0, 5000); // Max 5000 chars
    }
}

// ================================================================
// 📊 FONCTIONS STATISTIQUES
// ================================================================

if (!function_exists('calculateUsagePercentage')) {
    function calculateUsagePercentage($used, $limit) {
        if ($limit <= 0) return 0;
        return min(100, round(($used / $limit) * 100, 1));
    }
}

if (!function_exists('formatFileSize')) {
    function formatFileSize($bytes) {
        if ($bytes >= 1073741824) {
            return round($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return round($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return round($bytes / 1024, 2) . ' KB';
        }
        return $bytes . ' B';
    }
}

// ================================================================
// 🔄 FONCTIONS UTILITAIRES
// ================================================================

if (!function_exists('generateSessionId')) {
    function generateSessionId() {
        return bin2hex(random_bytes(16));
    }
}

if (!function_exists('logDebug')) {
    function logDebug($message, $context = []) {
        $timestamp = date('Y-m-d H:i:s');
        $contextStr = !empty($context) ? ' - ' . json_encode($context) : '';
        error_log("[$timestamp] SMG DEBUG: $message$contextStr");
    }
}

if (!function_exists('isValidJson')) {
    function isValidJson($string) {
        json_decode($string);
        return json_last_error() === JSON_ERROR_NONE;
    }
}

// ================================================================
// 🎯 FONCTIONS SPÉCIALES SMG STUDIO
// ================================================================

if (!function_exists('buildEnrichedPrompt')) {
    function buildEnrichedPrompt($base_prompt, $platform, $content_type, $profile = null) {
        $system_context = isset($profile['system_prompt']) && !empty($profile['system_prompt']) ? 
            $profile['system_prompt'] : 
            "Tu es un expert en création de contenu pour les réseaux sociaux. Tu crées du contenu engageant, authentique et optimisé pour chaque plateforme.";
        
        $platform_constraints = [
            'instagram' => 'Format Instagram: 2200 caractères max, emojis, hashtags pertinents',
            'facebook' => 'Format Facebook: ton convivial, encourage l\'interaction', 
            'linkedin' => 'Format LinkedIn: professionnel, valeur ajoutée, expertise',
            'twitter' => 'Format Twitter: 280 caractères max, hashtags, accrocheur',
            'tiktok' => 'Format TikTok: jeune, dynamique, hashtags tendance',
            'youtube' => 'Format YouTube: description captivante, SEO-friendly'
        ];
        
        $constraint = $platform_constraints[$platform] ?? 'Format adapté aux réseaux sociaux';
        
        return $system_context . "\n\n" . 
               "Contraintes: " . $constraint . "\n\n" .
               "Demande: " . $base_prompt;
    }
}

/**
 * ================================================================
 * ✅ FICHIER CHARGÉ - TOUTES FONCTIONS DISPONIBLES
 * ================================================================
 */
?>