<?php
/**
 * ================================================================
 * 🎬 CLIPMAKER TIMELINE - AVEC CSS UNIFIÉ
 * ================================================================
 * 
 * 🎯 UTILISATION DU CSS UNIFIÉ :
 * - Remplacement des styles inline
 * - Classes harmonisées avec dashboard
 * - Composants cohérents
 * 
 * 💡 BY PAPY 77 - HARMONIE PARFAITE !
 * ================================================================
 */

require_once '../auth.php';
requireLogin();

$user = getCurrentUser();
$project_id = $_GET['project'] ?? null;

// Récupération du projet
$project = null;
if ($project_id) {
    $stmt = $pdo->prepare("
        SELECT * FROM clipmaker_projects 
        WHERE id = ? AND user_id = ?
    ");
    $stmt->execute([$project_id, $user['id']]);
    $project = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Récupération des médias et transitions
$mediaStmt = $pdo->prepare("
    SELECT * FROM clipmaker_media 
    WHERE user_id = ? 
    ORDER BY created_at DESC
");
$mediaStmt->execute([$user['id']]);
$medias = $mediaStmt->fetchAll(PDO::FETCH_ASSOC);

$transitionsStmt = $pdo->prepare("
    SELECT * FROM transitions 
    WHERE actif = 1 
    ORDER BY nom
");
$transitionsStmt->execute();
$transitions = $transitionsStmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🎬 Clipmaker Timeline - SMG Studio</title>
    
    <!-- 🎨 CSS UNIFIÉ -->
    <link rel="stylesheet" href="../assets/css/clipmaker-unified.css">
    
    <!-- 🎬 STYLES SPÉCIFIQUES TIMELINE -->
    <style>
        /* Styles spécifiques timeline qui complètent le CSS unifié */
        .timeline-workspace {
            display: flex;
            height: calc(100vh - 80px);
        }
        
        .media-panel {
            width: 300px;
            background: rgba(255, 255, 255, 0.05);
            backdrop-filter: blur(20px);
            border-right: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .video-player {
            flex: 1;
            background: #000;
            display: flex;
            align-items: center;
            justify-content: center;
            position: relative;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .timeline-zone {
            height: 300px;
            background: rgba(255, 255, 255, 0.05);
            backdrop-filter: blur(20px);
            border-top: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .effects-panel {
            position: fixed;
            right: -350px;
            top: 80px;
            width: 350px;
            height: calc(100vh - 80px);
            background: rgba(45, 55, 72, 0.95);
            backdrop-filter: blur(20px);
            border-left: 1px solid rgba(255, 255, 255, 0.1);
            transition: right 0.3s ease;
            z-index: 1000;
        }
        
        .effects-panel.open {
            right: 0;
        }
        
        .transitions-panel {
            position: fixed;
            bottom: -200px;
            left: 300px;
            right: 0;
            height: 200px;
            background: rgba(45, 55, 72, 0.95);
            backdrop-filter: blur(20px);
            border-top: 1px solid rgba(255, 255, 255, 0.1);
            transition: bottom 0.3s ease;
            z-index: 1000;
        }
        
        .transitions-panel.open {
            bottom: 0;
        }
        
        .timeline-clip {
            position: absolute;
            height: 80%;
            background: linear-gradient(135deg, #667eea, #764ba2);
            border-radius: 6px;
            border: 2px solid transparent;
            cursor: move;
            display: flex;
            align-items: center;
            padding: 0 12px;
            font-size: 12px;
            font-weight: 500;
            top: 10%;
            transition: all 0.3s ease;
            min-width: 80px;
            overflow: hidden;
        }
        
        .timeline-clip:hover {
            border-color: #4CAF50;
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
        }
        
        .timeline-clip.selected {
            border-color: #FFD700;
            box-shadow: 0 0 15px rgba(255, 215, 0, 0.5);
        }
        
        .timeline-clip.video {
            background: linear-gradient(135deg, #667eea, #764ba2);
        }
        
        .timeline-clip.audio {
            background: linear-gradient(135deg, #f093fb, #f5576c);
        }
        
        .timeline-clip.image {
            background: linear-gradient(135deg, #4facfe, #00f2fe);
        }
        
        .timeline-clip.text {
            background: linear-gradient(135deg, #43e97b, #38f9d7);
        }
        
        .resize-handle {
            position: absolute;
            top: 0;
            bottom: 0;
            width: 8px;
            background: rgba(255, 255, 255, 0.3);
            cursor: col-resize;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .resize-handle.left {
            left: 0;
            border-radius: 6px 0 0 6px;
        }
        
        .resize-handle.right {
            right: 0;
            border-radius: 0 6px 6px 0;
        }
        
        .timeline-clip:hover .resize-handle {
            opacity: 1;
        }
        
        .split-indicator {
            position: absolute;
            top: 0;
            bottom: 0;
            width: 2px;
            background: #FF6B6B;
            display: none;
            cursor: col-resize;
            z-index: 100;
        }
        
        .split-indicator.active {
            display: block;
        }
        
        .split-indicator::before {
            content: '✂️';
            position: absolute;
            top: -25px;
            left: -8px;
            background: #FF6B6B;
            color: white;
            padding: 2px 4px;
            border-radius: 3px;
            font-size: 0.8rem;
        }
        
        .export-modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.8);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 9999;
            backdrop-filter: blur(10px);
        }
        
        .export-modal.active {
            display: flex;
        }
        
        .export-content {
            background: rgba(45, 55, 72, 0.95);
            backdrop-filter: blur(20px);
            padding: 40px;
            border-radius: 20px;
            text-align: center;
            min-width: 400px;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .export-progress {
            width: 100%;
            height: 8px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 4px;
            overflow: hidden;
            margin: 20px 0;
        }
        
        .export-progress-bar {
            height: 100%;
            background: linear-gradient(90deg, #4CAF50, #45a049);
            width: 0%;
            transition: width 0.3s ease;
        }
    </style>
</head>

<body>
    <!-- 🎛️ HEADER UNIFIÉ -->
    <div class="clipmaker-header">
        <a href="../clipmaker.php" class="clipmaker-logo">
            <span>🎬</span>
            <span>Clipmaker Timeline</span>
        </a>
        
        <div class="clipmaker-nav">
            <span style="color: #a0aec0;">
                <?= $project ? htmlspecialchars($project['name']) : 'Projet non trouvé' ?>
            </span>
        </div>
        
        <div class="clipmaker-actions">
            <button class="btn-secondary" onclick="toggleEffectsPanel()">
                🎨 Effets
            </button>
            <button class="btn-secondary" onclick="toggleTransitionsPanel()">
                🎭 Transitions
            </button>
            <button class="btn-secondary" onclick="saveProject()">
                💾 Sauvegarder
            </button>
            <button class="btn-primary" onclick="exportProject()">
                🎬 Exporter
            </button>
        </div>
    </div>

    <!-- 🎬 WORKSPACE -->
    <div class="timeline-workspace">
        <!-- 📁 PANNEAU MÉDIAS -->
        <div class="media-panel">
            <div class="clipmaker-section" style="margin: 0; border-radius: 0;">
                <h3>📚 Bibliothèque</h3>
                <div class="form-group">
                    <input type="text" class="form-input" placeholder="Rechercher médias..." id="mediaSearch">
                </div>
                <div class="media-grid" id="mediaList">
                    <?php foreach ($medias as $media): ?>
                    <div class="media-item" draggable="true" data-media-id="<?= $media['id'] ?>" data-media-type="<?= $media['file_type'] ?>">
                        <div class="media-thumbnail">
                            <?= strtoupper(substr($media['file_type'], 0, 3)) ?>
                        </div>
                        <div class="media-info">
                            <div class="media-title"><?= htmlspecialchars($media['original_filename']) ?></div>
                            <div><?= $media['file_type'] ?> • <?= number_format($media['file_size']/1024/1024, 1) ?>MB</div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <!-- 🎥 ZONE PRINCIPALE -->
        <div class="clipmaker-main">
            <!-- 📺 PLAYER VIDEO -->
            <div class="video-player">
                <div style="position: relative; width: 100%; height: 100%; display: flex; align-items: center; justify-content: center;">
                    <video id="mainPlayer" style="max-width: 100%; max-height: 100%; border-radius: 8px; display: none;">
                        <source src="" type="video/mp4">
                        Votre navigateur ne supporte pas la lecture vidéo.
                    </video>
                    <div id="videoOverlay" style="text-align: center; color: #a0aec0;">
                        <div style="font-size: 4rem; margin-bottom: 10px;">🎬</div>
                        <p>Sélectionnez un média pour commencer l'édition</p>
                    </div>
                </div>
                
                <!-- 🎛️ CONTRÔLES PLAYER -->
                <div style="position: absolute; bottom: 0; left: 0; right: 0; background: linear-gradient(transparent, rgba(0,0,0,0.8)); padding: 20px; display: flex; align-items: center; gap: 15px;">
                    <button class="btn-primary" id="playButton" style="width: 50px; height: 50px; border-radius: 50%; padding: 0;">
                        ▶️
                    </button>
                    <div class="clipmaker-progress" style="flex: 1;" id="progressBar">
                        <div class="clipmaker-progress-bar" id="progressFill"></div>
                    </div>
                    <div style="color: #ccc; font-size: 0.9rem; min-width: 80px;" id="timeDisplay">
                        00:00 / 00:00
                    </div>
                </div>
            </div>

            <!-- ⏱️ TIMELINE ZONE -->
            <div class="timeline-zone">
                <div class="timeline-header">
                    <div class="timeline-controls">
                        <button class="timeline-tool active" data-tool="select">🎯 Sélection</button>
                        <button class="timeline-tool" data-tool="cut">✂️ Couper</button>
                        <button class="timeline-tool" data-tool="zoom">🔍 Zoom</button>
                        <button class="timeline-tool" data-tool="transition">🎭 Transition</button>
                        
                        <div class="timeline-zoom">
                            <span>🔍</span>
                            <input type="range" class="zoom-slider" min="0.1" max="3" step="0.1" value="1" id="zoomSlider">
                            <span class="zoom-level" id="zoomLevel">100%</span>
                        </div>
                    </div>
                </div>
                
                <div class="timeline-container" style="flex: 1; overflow: auto;">
                    <div style="position: relative; background: #2a2a2a; min-height: 100%;">
                        <!-- 📏 RÈGLE TEMPORELLE -->
                        <div class="timeline-ruler" id="timelineRuler">
                            <div style="position: absolute; left: 0px; height: 100%; border-left: 1px solid #666; color: #aaa; padding-left: 5px; font-size: 0.8rem; display: flex; align-items: center;">0:00</div>
                            <div style="position: absolute; left: 100px; height: 100%; border-left: 1px solid #666; color: #aaa; padding-left: 5px; font-size: 0.8rem; display: flex; align-items: center;">0:05</div>
                            <div style="position: absolute; left: 200px; height: 100%; border-left: 1px solid #666; color: #aaa; padding-left: 5px; font-size: 0.8rem; display: flex; align-items: center;">0:10</div>
                        </div>
                        
                        <!-- 🎵 PISTES -->
                        <div style="padding: 20px; min-height: 200px; position: relative;">
                            <div class="split-indicator" id="splitIndicator"></div>
                            
                            <div class="timeline-track" data-track="video">
                                <div class="track-label">🎬 Vidéo</div>
                                <div class="track-content">
                                    <div style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; border: 2px dashed #555; border-radius: 6px; display: flex; align-items: center; justify-content: center; color: #aaa; font-size: 0.9rem; transition: all 0.3s ease;" class="drop-zone">
                                        Glissez un média ici
                                    </div>
                                </div>
                            </div>
                            
                            <div class="timeline-track" data-track="audio">
                                <div class="track-label">🎵 Audio</div>
                                <div class="track-content">
                                    <div style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; border: 2px dashed #555; border-radius: 6px; display: flex; align-items: center; justify-content: center; color: #aaa; font-size: 0.9rem; transition: all 0.3s ease;" class="drop-zone">
                                        Glissez un média ici
                                    </div>
                                </div>
                            </div>
                            
                            <div class="timeline-track" data-track="text">
                                <div class="track-label">📝 Texte</div>
                                <div class="track-content">
                                    <div style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; border: 2px dashed #555; border-radius: 6px; display: flex; align-items: center; justify-content: center; color: #aaa; font-size: 0.9rem; transition: all 0.3s ease;" class="drop-zone">
                                        Glissez un média ici
                                    </div>
                                </div>
                            </div>
                            
                            <div class="timeline-track" data-track="effects">
                                <div class="track-label">✨ Effets</div>
                                <div class="track-content">
                                    <div style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; border: 2px dashed #555; border-radius: 6px; display: flex; align-items: center; justify-content: center; color: #aaa; font-size: 0.9rem; transition: all 0.3s ease;" class="drop-zone">
                                        Zone d'effets
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- 🎨 PANNEAU EFFETS -->
    <div class="effects-panel" id="effectsPanel">
        <div class="clipmaker-section" style="margin: 0; border-radius: 0; height: 100%;">
            <div class="flex-between mb-3">
                <h3>🎨 Effets Visuels</h3>
                <button class="btn-secondary btn-small" onclick="toggleEffectsPanel()">×</button>
            </div>
            
            <div class="effect-grid mb-4">
                <div class="effect-item" data-effect="brightness">
                    <div class="effect-icon">☀️</div>
                    <div class="effect-name">Luminosité</div>
                </div>
                <div class="effect-item" data-effect="contrast">
                    <div class="effect-icon">🌗</div>
                    <div class="effect-name">Contraste</div>
                </div>
                <div class="effect-item" data-effect="saturation">
                    <div class="effect-icon">🎨</div>
                    <div class="effect-name">Saturation</div>
                </div>
                <div class="effect-item" data-effect="blur">
                    <div class="effect-icon">🌫️</div>
                    <div class="effect-name">Flou</div>
                </div>
            </div>
            
            <div class="form-group">
                <label class="form-label">Intensité</label>
                <input type="range" class="form-input" min="0" max="100" value="50" id="effectIntensity">
            </div>
            
            <div class="form-group">
                <label class="form-label">Opacité</label>
                <input type="range" class="form-input" min="0" max="100" value="100" id="effectOpacity">
            </div>
            
            <h4>🎵 Effets Audio</h4>
            <div class="clipmaker-card">
                <div class="flex-between mb-2">
                    <span>🎚️ Égaliseur</span>
                    <div class="clipmaker-badge" data-audio-effect="equalizer">OFF</div>
                </div>
                <div class="flex-between mb-2">
                    <span>🔇 Réduction bruit</span>
                    <div class="clipmaker-badge" data-audio-effect="noise-reduction">OFF</div>
                </div>
                <div class="flex-between mb-2">
                    <span>🎙️ Compression</span>
                    <div class="clipmaker-badge" data-audio-effect="compression">OFF</div>
                </div>
                <div class="flex-between">
                    <span>🌊 Reverb</span>
                    <div class="clipmaker-badge" data-audio-effect="reverb">OFF</div>
                </div>
            </div>
            
            <h4>📐 Templates</h4>
            <div class="clipmaker-card">
                <div class="flex-between mb-2">
                    <div>
                        <div style="font-weight: 500;">🎬 Intro Dynamique</div>
                        <div style="font-size: 0.8rem; color: #a0aec0;">Logo + texte + transition</div>
                    </div>
                    <button class="btn-secondary btn-small" data-template="intro">Appliquer</button>
                </div>
                <div class="flex-between mb-2">
                    <div>
                        <div style="font-weight: 500;">🎭 Outro Élégant</div>
                        <div style="font-size: 0.8rem; color: #a0aec0;">Fondu + crédits</div>
                    </div>
                    <button class="btn-secondary btn-small" data-template="outro">Appliquer</button>
                </div>
                <div class="flex-between">
                    <div>
                        <div style="font-weight: 500;">📰 Lower Third</div>
                        <div style="font-size: 0.8rem; color: #a0aec0;">Nom + titre</div>
                    </div>
                    <button class="btn-secondary btn-small" data-template="lower-third">Appliquer</button>
                </div>
            </div>
        </div>
    </div>

    <!-- 🎭 PANNEAU TRANSITIONS -->
    <div class="transitions-panel" id="transitionsPanel">
        <div class="clipmaker-section" style="margin: 0; border-radius: 0; height: 100%;">
            <div class="flex-between mb-3">
                <h3>🎭 Transitions</h3>
                <button class="btn-secondary btn-small" onclick="toggleTransitionsPanel()">×</button>
            </div>
            
            <div style="display: flex; gap: 15px; overflow-x: auto; padding: 10px 0;">
                <?php foreach ($transitions as $transition): ?>
                <div class="clipmaker-card" style="min-width: 120px; text-align: center; cursor: pointer;" 
                     data-transition-id="<?= $transition['id'] ?>" 
                     data-transition-code="<?= $transition['code_ffmpeg'] ?>">
                    <div style="font-size: 1.5rem; margin-bottom: 8px;">🎬</div>
                    <div style="font-size: 0.8rem; font-weight: 500;"><?= htmlspecialchars($transition['nom']) ?></div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>

    <!-- 🔄 MODAL EXPORT -->
    <div class="export-modal" id="exportModal">
        <div class="export-content">
            <h3>🎬 Export en cours...</h3>
            <div class="export-progress">
                <div class="export-progress-bar" id="exportProgressBar"></div>
            </div>
            <div style="color: #a0aec0; font-size: 0.9rem; margin-top: 10px;" id="exportStatus">
                Préparation...
            </div>
            <button class="btn-secondary" onclick="cancelExport()" id="cancelExportBtn" style="margin-top: 20px;">
                Annuler
            </button>
        </div>
    </div>

    <!-- 🎯 JAVASCRIPT TIMELINE -->
    <script>
    class ClipmakerTimeline {
        constructor() {
            this.currentProject = <?= json_encode($project) ?>;
            this.medias = <?= json_encode($medias) ?>;
            this.transitions = <?= json_encode($transitions) ?>;
            this.currentTool = 'select';
            this.isPlaying = false;
            this.currentTime = 0;
            this.selectedClip = null;
            this.clipCounter = 0;
            this.activeEffects = new Map();
            this.audioEffects = new Map();
            this.isDragging = false;
            this.zoomLevel = 1;
            this.pixelsPerSecond = 20;
            this.splitPosition = null;
            this.isExporting = false;
            
            this.init();
        }

        init() {
            this.initializeDragDrop();
            this.initializePlayer();
            this.initializeTools();
            this.initializeSearch();
            this.initializeEffects();
            this.initializeTransitions();
            this.initializeClipSelection();
            this.initializeZoom();
            this.initializeSplitTool();
            this.initializeAudioEffects();
            this.initializeTemplates();
            this.initializeKeyboardShortcuts();
        }

        // 🎬 DRAG & DROP
        initializeDragDrop() {
            const mediaItems = document.querySelectorAll('.media-item');
            const dropZones = document.querySelectorAll('.drop-zone');
            
            mediaItems.forEach(item => {
                item.addEventListener('dragstart', (e) => {
                    e.dataTransfer.setData('text/plain', item.dataset.mediaId);
                    item.classList.add('dragging');
                });
                
                item.addEventListener('dragend', (e) => {
                    item.classList.remove('dragging');
                });
            });
            
            dropZones.forEach(zone => {
                zone.addEventListener('dragover', (e) => {
                    e.preventDefault();
                    zone.style.borderColor = '#4CAF50';
                    zone.style.background = 'rgba(76, 175, 80, 0.1)';
                });
                
                zone.addEventListener('dragleave', (e) => {
                    zone.style.borderColor = '#555';
                    zone.style.background = 'transparent';
                });
                
                zone.addEventListener('drop', (e) => {
                    e.preventDefault();
                    zone.style.borderColor = '#555';
                    zone.style.background = 'transparent';
                    
                    const mediaId = e.dataTransfer.getData('text/plain');
                    const trackType = zone.closest('.timeline-track').dataset.track;
                    
                    const rect = zone.getBoundingClientRect();
                    const x = e.clientX - rect.left;
                    const timePosition = this.pixelToTime(x);
                    
                    this.addMediaToTrack(mediaId, trackType, timePosition);
                });
            });
        }

        // 🎬 AJOUTER MÉDIA À PISTE
        addMediaToTrack(mediaId, trackType, timePosition = 0) {
            const media = this.medias.find(m => m.id == mediaId);
            if (!media) return;
            
            const track = document.querySelector(`[data-track="${trackType}"] .track-content`);
            const dropZone = track.querySelector('.drop-zone');
            
            const clipElement = document.createElement('div');
            clipElement.className = `timeline-clip ${media.file_type}`;
            clipElement.dataset.mediaId = mediaId;
            clipElement.dataset.clipId = ++this.clipCounter;
            clipElement.dataset.startTime = timePosition;
            clipElement.dataset.duration = media.duration || 5;
            
            const width = this.timeToPixel(media.duration || 5);
            clipElement.style.width = width + 'px';
            clipElement.style.left = this.timeToPixel(timePosition) + 'px';
            
            const clipName = media.original_filename.substring(0, 12) + '...';
            clipElement.innerHTML = `
                <div class="resize-handle left"></div>
                <span>${clipName}</span>
                <div class="resize-handle right"></div>
            `;
            
            if (track.children.length === 1) {
                dropZone.style.display = 'none';
            }
            
            track.appendChild(clipElement);
            
            this.initializeClipInteractions(clipElement);
            
            if (media.file_type === 'video' && trackType === 'video') {
                this.loadVideoInPlayer(media.file_path);
            }
            
            this.showNotification(`Clip ajouté: ${media.original_filename}`);
        }

        // 🎯 INTERACTIONS CLIP
        initializeClipInteractions(clipElement) {
            clipElement.addEventListener('click', (e) => {
                e.stopPropagation();
                this.selectClip(clipElement);
            });
            
            clipElement.addEventListener('mousedown', (e) => {
                if (e.target.classList.contains('resize-handle')) {
                    this.startResize(e, clipElement);
                } else {
                    this.startDrag(e, clipElement);
                }
            });
        }

        // 🎯 SÉLECTION CLIP
        selectClip(clipElement) {
            if (this.selectedClip) {
                this.selectedClip.classList.remove('selected');
            }
            
            this.selectedClip = clipElement;
            clipElement.classList.add('selected');
            
            const mediaId = clipElement.dataset.mediaId;
            const media = this.medias.find(m => m.id == mediaId);
            if (media && media.file_type === 'video') {
                this.loadVideoInPlayer(media.file_path);
            }
        }

        // 🎯 DÉSÉLECTION CLIP
        deselectClip() {
            if (this.selectedClip) {
                this.selectedClip.classList.remove('selected');
                this.selectedClip = null;
            }
        }

        // 📺 PLAYER
        initializePlayer() {
            const player = document.getElementById('mainPlayer');
            const playButton = document.getElementById('playButton');
            const progressBar = document.getElementById('progressBar');
            const progressFill = document.getElementById('progressFill');
            const timeDisplay = document.getElementById('timeDisplay');
            
            playButton.addEventListener('click', () => {
                if (this.isPlaying) {
                    this.pauseVideo();
                } else {
                    this.playVideo();
                }
            });
            
            progressBar.addEventListener('click', (e) => {
                const rect = progressBar.getBoundingClientRect();
                const percent = (e.clientX - rect.left) / rect.width;
                this.seekTo(percent);
            });
            
            player.addEventListener('timeupdate', () => {
                this.updateProgress();
            });
        }

        loadVideoInPlayer(filePath) {
            const player = document.getElementById('mainPlayer');
            const overlay = document.getElementById('videoOverlay');
            
            player.src = filePath;
            player.style.display = 'block';
            overlay.style.display = 'none';
            
            player.load();
        }

        playVideo() {
            const player = document.getElementById('mainPlayer');
            const playButton = document.getElementById('playButton');
            
            player.play();
            this.isPlaying = true;
            playButton.textContent = '⏸️';
        }

        pauseVideo() {
            const player = document.getElementById('mainPlayer');
            const playButton = document.getElementById('playButton');
            
            player.pause();
            this.isPlaying = false;
            playButton.textContent = '▶️';
        }

        seekTo(percent) {
            const player = document.getElementById('mainPlayer');
            if (player.duration) {
                player.currentTime = player.duration * percent;
            }
        }

        updateProgress() {
            const player = document.getElementById('mainPlayer');
            const progressFill = document.getElementById('progressFill');
            
            if (player.duration) {
                const percent = (player.currentTime / player.duration) * 100;
                progressFill.style.width = percent + '%';
                this.currentTime = player.currentTime;
                this.updateTimeDisplay();
            }
        }

        updateTimeDisplay() {
            const player = document.getElementById('mainPlayer');
            const timeDisplay = document.getElementById('timeDisplay');
            
            const current = this.formatTime(player.currentTime || 0);
            const duration = this.formatTime(player.duration || 0);
            
            timeDisplay.textContent = `${current} / ${duration}`;
        }

        formatTime(seconds) {
            const mins = Math.floor(seconds / 60);
            const secs = Math.floor(seconds % 60);
            return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
        }

        // 🎛️ OUTILS
        initializeTools() {
            const toolButtons = document.querySelectorAll('.timeline-tool');
            
            toolButtons.forEach(btn => {
                btn.addEventListener('click', (e) => {
                    toolButtons.forEach(b => b.classList.remove('active'));
                    btn.classList.add('active');
                    this.currentTool = btn.dataset.tool;
                });
            });
        }

        // 🔍 RECHERCHE
        initializeSearch() {
            const searchInput = document.getElementById('mediaSearch');
            
            searchInput.addEventListener('input', (e) => {
                const query = e.target.value.toLowerCase();
                const mediaItems = document.querySelectorAll('.media-item');
                
                mediaItems.forEach(item => {
                    const title = item.querySelector('.media-title').textContent.toLowerCase();
                    const visible = title.includes(query);
                    item.style.display = visible ? 'block' : 'none';
                });
            });
        }

        // 🎨 EFFETS
        initializeEffects() {
            const effectItems = document.querySelectorAll('.effect-item');
            
            effectItems.forEach(item => {
                item.addEventListener('click', () => {
                    if (this.selectedClip) {
                        this.applyEffect(item.dataset.effect);
                        item.classList.toggle('active');
                    } else {
                        this.showNotification('Sélectionnez un clip d\'abord');
                    }
                });
            });
        }

        applyEffect(effectType) {
            if (!this.selectedClip) return;
            
            const clipId = this.selectedClip.dataset.clipId;
            
            if (!this.activeEffects.has(clipId)) {
                this.activeEffects.set(clipId, new Set());
            }
            
            const clipEffects = this.activeEffects.get(clipId);
            
            if (clipEffects.has(effectType)) {
                clipEffects.delete(effectType);
                this.showNotification(`Effet ${effectType} retiré`);
            } else {
                clipEffects.add(effectType);
                this.showNotification(`Effet ${effectType} appliqué`);
            }
            
            this.updateClipVisualEffects(this.selectedClip);
        }

        updateClipVisualEffects(clipElement) {
            const clipId = clipElement.dataset.clipId;
            const effects = this.activeEffects.get(clipId);
            
            if (!effects || effects.size === 0) {
                clipElement.style.filter = '';
                return;
            }
            
            let filterString = '';
            
            effects.forEach(effect => {
                switch (effect) {
                    case 'brightness':
                        filterString += 'brightness(1.2) ';
                        break;
                    case 'contrast':
                        filterString += 'contrast(1.3) ';
                        break;
                    case 'saturation':
                        filterString += 'saturate(1.5) ';
                        break;
                    case 'blur':
                        filterString += 'blur(2px) ';
                        break;
                }
            });
            
            clipElement.style.filter = filterString.trim();
        }

        // 🎭 TRANSITIONS
        initializeTransitions() {
            const transitionItems = document.querySelectorAll('[data-transition-id]');
            
            transitionItems.forEach(item => {
                item.addEventListener('click', () => {
                    if (this.selectedClip) {
                        this.applyTransition(item.dataset.transitionId, item.dataset.transitionCode);
                        
                        transitionItems.forEach(t => t.classList.remove('active'));
                        item.classList.add('active');
                        
                        this.selectedClip.classList.add('has-transition');
                        this.showNotification(`Transition appliquée: ${item.dataset.transitionCode}`);
                    } else {
                        this.showNotification('Sélectionnez un clip d\'abord');
                    }
                });
            });
        }

        applyTransition(transitionId, transitionCode) {
            if (!this.selectedClip) return;
            
            this.selectedClip.dataset.transitionId = transitionId;
            this.selectedClip.dataset.transitionCode = transitionCode;
        }

        // 🎵 EFFETS AUDIO
        initializeAudioEffects() {
            const audioToggles = document.querySelectorAll('[data-audio-effect]');
            
            audioToggles.forEach(toggle => {
                toggle.addEventListener('click', () => {
                    const effect = toggle.dataset.audioEffect;
                    
                    if (this.selectedClip) {
                        this.toggleAudioEffect(effect, toggle);
                    } else {
                        this.showNotification('Sélectionnez un clip audio d\'abord');
                    }
                });
            });
        }

        toggleAudioEffect(effectType, toggleElement) {
            const clipId = this.selectedClip.dataset.clipId;
            
            if (!this.audioEffects.has(clipId)) {
                this.audioEffects.set(clipId, new Set());
            }
            
            const clipAudioEffects = this.audioEffects.get(clipId);
            
            if (clipAudioEffects.has(effectType)) {
                clipAudioEffects.delete(effectType);
                toggleElement.textContent = 'OFF';
                toggleElement.style.background = 'rgba(255, 255, 255, 0.1)';
                this.showNotification(`Effet ${effectType} désactivé`);
            } else {
                clipAudioEffects.add(effectType);
                toggleElement.textContent = 'ON';
                toggleElement.style.background = 'linear-gradient(135deg, #48bb78, #38a169)';
                this.showNotification(`Effet ${effectType} activé`);
            }
        }

        // 📐 TEMPLATES
        initializeTemplates() {
            const templateBtns = document.querySelectorAll('[data-template]');
            
            templateBtns.forEach(btn => {
                btn.addEventListener('click', () => {
                    const templateType = btn.dataset.template;
                    this.applyTemplate(templateType);
                });
            });
        }

        applyTemplate(templateType) {
            const trackContent = document.querySelector('[data-track="text"] .track-content');
            const templateClip = document.createElement('div');
            templateClip.className = 'timeline-clip text';
            templateClip.dataset.clipId = ++this.clipCounter;
            templateClip.dataset.startTime = 0;
            templateClip.dataset.duration = 5;
            templateClip.dataset.template = templateType;
            
            templateClip.style.width = this.timeToPixel(5) + 'px';
            templateClip.style.left = '0px';
            
            templateClip.innerHTML = `
                <div class="resize-handle left"></div>
                <span>📐 ${templateType}</span>
                <div class="resize-handle right"></div>
            `;
            
            trackContent.appendChild(templateClip);
            this.initializeClipInteractions(templateClip);
            
            this.showNotification(`Template ${templateType} appliqué !`);
        }

        // 🎯 SÉLECTION CLIP
        initializeClipSelection() {
            document.addEventListener('click', (e) => {
                if (e.target.classList.contains('timeline-clip')) {
                    this.selectClip(e.target);
                } else if (!e.target.closest('.effects-panel') && !e.target.closest('.transitions-panel')) {
                    this.deselectClip();
                }
            });
        }

        // 🔍 ZOOM
        initializeZoom() {
            const zoomSlider = document.getElementById('zoomSlider');
            const zoomLevel = document.getElementById('zoomLevel');
            
            zoomSlider.addEventListener('input', (e) => {
                this.zoomLevel = parseFloat(e.target.value);
                this.pixelsPerSecond = 20 * this.zoomLevel;
                
                zoomLevel.textContent = Math.round(this.zoomLevel * 100) + '%';
                
                this.updateTimelineZoom();
            });
        }

        updateTimelineZoom() {
            const clips = document.querySelectorAll('.timeline-clip');
            
            clips.forEach(clip => {
                const duration = parseFloat(clip.dataset.duration);
                const startTime = parseFloat(clip.dataset.startTime);
                
                const width = this.timeToPixel(duration);
                const left = this.timeToPixel(startTime);
                
                clip.style.width = width + 'px';
                clip.style.left = left + 'px';
            });
        }

        // ✂️ DÉCOUPE
        initializeSplitTool() {
            const splitIndicator = document.getElementById('splitIndicator');
            const trackContainer = document.querySelector('.timeline-zone');
            
            trackContainer.addEventListener('mousemove', (e) => {
                if (this.currentTool === 'cut' && !this.isDragging) {
                    const rect = trackContainer.getBoundingClientRect();
                    const x = e.clientX - rect.left;
                    
                    splitIndicator.style.left = x + 'px';
                    splitIndicator.classList.add('active');
                    
                    this.splitPosition = this.pixelToTime(x);
                }
            });
            
            trackContainer.addEventListener('mouseleave', () => {
                splitIndicator.classList.remove('active');
            });
        }

        // ⌨️ RACCOURCIS
        initializeKeyboardShortcuts() {
            document.addEventListener('keydown', (e) => {
                if (e.target.tagName === 'INPUT') return;
                
                switch (e.key) {
                    case ' ':
                        e.preventDefault();
                        this.isPlaying ? this.pauseVideo() : this.playVideo();
                        break;
                    case 'Delete':
                        if (this.selectedClip) {
                            this.selectedClip.remove();
                            this.selectedClip = null;
                        }
                        break;
                    case 's':
                        if (e.ctrlKey) {
                            e.preventDefault();
                            saveProject();
                        }
                        break;
                }
            });
        }

        // 🎬 DRAG
        startDrag(e, clipElement) {
            this.isDragging = true;
            this.selectClip(clipElement);
            
            const rect = clipElement.getBoundingClientRect();
            this.dragOffset = {
                x: e.clientX - rect.left,
                y: e.clientY - rect.top
            };
            
            clipElement.style.opacity = '0.7';
            
            document.addEventListener('mousemove', this.dragMove.bind(this));
            document.addEventListener('mouseup', this.dragEnd.bind(this));
            
            e.preventDefault();
        }

        dragMove(e) {
            if (!this.isDragging || !this.selectedClip) return;
            
            const timeline = document.querySelector('.timeline-zone');
            const rect = timeline.getBoundingClientRect();
            
            const x = e.clientX - rect.left - this.dragOffset.x;
            const constrainedX = Math.max(0, x);
            
            this.selectedClip.style.left = constrainedX + 'px';
            
            const timePosition = this.pixelToTime(constrainedX);
            this.selectedClip.dataset.startTime = timePosition;
        }

        dragEnd(e) {
            if (!this.isDragging) return;
            
            this.isDragging = false;
            
            if (this.selectedClip) {
                this.selectedClip.style.opacity = '1';
            }
            
            document.removeEventListener('mousemove', this.dragMove);
            document.removeEventListener('mouseup', this.dragEnd);
        }

        // 📏 CONVERSION
        timeToPixel(seconds) {
            return seconds * this.pixelsPerSecond;
        }

        pixelToTime(pixels) {
            return pixels / this.pixelsPerSecond;
        }

        // 💬 NOTIFICATIONS
        showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            notification.textContent = message;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.remove();
            }, 3000);
        }
    }

    // 🚀 FONCTIONS GLOBALES
    function toggleEffectsPanel() {
        const panel = document.getElementById('effectsPanel');
        panel.classList.toggle('open');
    }

    function toggleTransitionsPanel() {
        const panel = document.getElementById('transitionsPanel');
        panel.classList.toggle('open');
    }

    function saveProject() {
        if (!window.timeline) return;
        
        const projectData = {
            clips: [],
            effects: {},
            transitions: {}
        };
        
        const clips = document.querySelectorAll('.timeline-clip');
        clips.forEach(clip => {
            const clipData = {
                id: clip.dataset.clipId,
                mediaId: clip.dataset.mediaId,
                startTime: parseFloat(clip.dataset.startTime),
                duration: parseFloat(clip.dataset.duration),
                track: clip.closest('.timeline-track').dataset.track,
                effects: Array.from(window.timeline.activeEffects.get(clip.dataset.clipId) || []),
                transitionId: clip.dataset.transitionId || null,
                transitionCode: clip.dataset.transitionCode || null
            };
            projectData.clips.push(clipData);
        });
        
        fetch('../api/clipmaker-timeline-api.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                action: 'save-project',
                project_id: window.timeline.currentProject?.id,
                timeline_data: projectData,
                clips: projectData.clips
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                window.timeline.showNotification('✅ Projet sauvegardé !', 'success');
            } else {
                window.timeline.showNotification('❌ Erreur sauvegarde', 'error');
            }
        })
        .catch(error => {
            console.error('Erreur:', error);
            window.timeline.showNotification('❌ Erreur réseau', 'error');
        });
    }

    function exportProject() {
        if (!window.timeline) return;
        
        const clips = document.querySelectorAll('.timeline-clip');
        if (clips.length === 0) {
            window.timeline.showNotification('❌ Aucun clip à exporter', 'error');
            return;
        }
        
        const modal = document.getElementById('exportModal');
        const progressBar = document.getElementById('exportProgressBar');
        const status = document.getElementById('exportStatus');
        
        modal.classList.add('active');
        window.timeline.isExporting = true;
        
        const exportSteps = [
            { progress: 10, status: 'Analyse des clips...' },
            { progress: 25, status: 'Préparation des médias...' },
            { progress: 40, status: 'Application des effets...' },
            { progress: 60, status: 'Rendu vidéo...' },
            { progress: 80, status: 'Encodage audio...' },
            { progress: 95, status: 'Finalisation...' },
            { progress: 100, status: 'Export terminé !' }
        ];
        
        let stepIndex = 0;
        const exportInterval = setInterval(() => {
            if (!window.timeline.isExporting || stepIndex >= exportSteps.length) {
                clearInterval(exportInterval);
                
                if (window.timeline.isExporting) {
                    setTimeout(() => {
                        modal.classList.remove('active');
                        window.timeline.showNotification('✅ Export terminé !', 'success');
                        window.timeline.isExporting = false;
                    }, 1000);
                }
                
                return;
            }
            
            const step = exportSteps[stepIndex];
            progressBar.style.width = step.progress + '%';
            status.textContent = step.status;
            
            stepIndex++;
        }, 800);
    }

    function cancelExport() {
        if (window.timeline.isExporting) {
            window.timeline.isExporting = false;
            
            const modal = document.getElementById('exportModal');
            modal.classList.remove('active');
            
            window.timeline.showNotification('Export annulé', 'warning');
        }
    }

    // 🎬 INITIALISATION
    document.addEventListener('DOMContentLoaded', () => {
        window.timeline = new ClipmakerTimeline();
        console.log('🎬 Clipmaker Timeline avec CSS unifié - Ready!');
    });
    </script>
</body>
</html>