<?php
/**
 * ================================================================
 * 🤖 SMG STUDIO v2.0 - GESTIONNAIRE PROFILS IA - FIXÉ NAVIGATION
 * ================================================================
 * 
 * 🔧 CORRECTIONS MAJEURES :
 * ✅ Navigation retour dashboard principale CORRIGÉE
 * ✅ Liaison clés API ↔ utilisateurs CORRIGÉE  
 * ✅ Affichage profils utilisateur connecté CORRIGÉ
 * ✅ Validation clés OpenAI CORRIGÉE
 * ✅ Debug utilisateur actuel affiché
 * 
 * 💡 BY PAPY 77 - FIX NAVIGATION + API KEYS LIAISON
 * ================================================================
 */

session_start();
require_once 'config/db.php';

// Vérification session
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?expired=1');
    exit;
}

// 🔧 DEBUG: Récupération utilisateur avec informations détaillées
try {
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND status = 'active'");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        error_log("❌ ERREUR: Aucun user actif trouvé pour ID " . $_SESSION['user_id']);
        session_destroy();
        header('Location: login.php?expired=1');
        exit;
    }
    
    error_log("✅ DEBUG: User connecté: " . $user['name'] . " (ID: " . $user['id'] . ", Email: " . $user['email'] . ")");
} catch (Exception $e) {
    error_log("❌ ERREUR DB user: " . $e->getMessage());
    header('Location: login.php?error=1');
    exit;
}

// ================================================================
// 🔧 TRAITEMENT DES ACTIONS - CORRIGÉ
// ================================================================

$action = $_POST['action'] ?? $_GET['action'] ?? 'list';
$message = '';
$message_type = '';

// Action Créer/Modifier Profil IA
if ($action === 'save_profile' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $profile_id = intval($_POST['profile_id'] ?? 0);
        $ai_provider = $_POST['ai_provider'] ?? '';
        $display_name = trim($_POST['display_name'] ?? '');
        $api_key = trim($_POST['api_key'] ?? '');
        $model_preference = $_POST['model_preference'] ?? '';
        $max_tokens = intval($_POST['max_tokens_per_request'] ?? 1000);
        $temperature = floatval($_POST['temperature'] ?? 0.7);
        $system_prompt = trim($_POST['system_prompt'] ?? '');
        $is_default = isset($_POST['is_default']) ? 1 : 0;
        
        error_log("🔧 DEBUG: Tentative sauvegarde profil - User: {$user['id']}, Provider: $ai_provider, Nom: $display_name");
        
        if (empty($ai_provider) || empty($display_name)) {
            throw new Exception("Provider et nom d'affichage requis");
        }
        
        if ($max_tokens < 100 || $max_tokens > 8000) {
            throw new Exception("Tokens entre 100 et 8000");
        }
        
        if ($temperature < 0 || $temperature > 2) {
            throw new Exception("Température entre 0 et 2");
        }
        
        // 🔧 CORRECTION: Gérer la clé API correctement
        $api_key_id = null;
        if (!empty($api_key) && $ai_provider !== 'smg_free' && $ai_provider !== 'local') {
            
            // Validation basique clé OpenAI
            if ($ai_provider === 'openai' && !preg_match('/^sk-/', $api_key)) {
                throw new Exception("Clé OpenAI invalide (doit commencer par 'sk-')");
            }
            
            $encrypted_key = base64_encode($api_key);
            
            if ($profile_id > 0) {
                // Mise à jour d'un profil existant
                $stmt = $pdo->prepare("SELECT api_key_id FROM user_ai_profiles WHERE id = ? AND user_id = ?");
                $stmt->execute([$profile_id, $user['id']]);
                $old_profile = $stmt->fetch();
                
                if ($old_profile && $old_profile['api_key_id']) {
                    // Mettre à jour la clé existante
                    $stmt = $pdo->prepare("UPDATE api_keys SET encrypted_key = ?, updated_at = NOW() WHERE id = ? AND user_id = ?");
                    $stmt->execute([$encrypted_key, $old_profile['api_key_id'], $user['id']]);
                    $api_key_id = $old_profile['api_key_id'];
                    error_log("✅ DEBUG: Clé API mise à jour pour profil existant");
                } else {
                    // Créer nouvelle clé pour profil existant
                    $stmt = $pdo->prepare("INSERT INTO api_keys (user_id, provider, key_name, encrypted_key) VALUES (?, ?, ?, ?)");
                    $stmt->execute([$user['id'], $ai_provider, $display_name, $encrypted_key]);
                    $api_key_id = $pdo->lastInsertId();
                    error_log("✅ DEBUG: Nouvelle clé API créée pour profil existant");
                }
            } else {
                // Vérifier si une clé existe déjà pour ce provider et cet utilisateur
                $stmt = $pdo->prepare("SELECT id FROM api_keys WHERE user_id = ? AND provider = ?");
                $stmt->execute([$user['id'], $ai_provider]);
                $existing_key = $stmt->fetch();
                
                if ($existing_key) {
                    // Mettre à jour la clé existante
                    $stmt = $pdo->prepare("UPDATE api_keys SET encrypted_key = ?, key_name = ?, updated_at = NOW() WHERE id = ?");
                    $stmt->execute([$encrypted_key, $display_name, $existing_key['id']]);
                    $api_key_id = $existing_key['id'];
                    error_log("✅ DEBUG: Clé API existante mise à jour");
                } else {
                    // Créer nouvelle clé
                    $stmt = $pdo->prepare("INSERT INTO api_keys (user_id, provider, key_name, encrypted_key) VALUES (?, ?, ?, ?)");
                    $stmt->execute([$user['id'], $ai_provider, $display_name, $encrypted_key]);
                    $api_key_id = $pdo->lastInsertId();
                    error_log("✅ DEBUG: Nouvelle clé API créée");
                }
            }
        }
        
        if ($profile_id > 0) {
            // Mise à jour profil existant
            $stmt = $pdo->prepare("
                UPDATE user_ai_profiles SET 
                    ai_provider = ?, display_name = ?, api_key_id = ?, 
                    model_preference = ?, max_tokens_per_request = ?, 
                    temperature = ?, system_prompt = ?, is_default = ?,
                    updated_at = NOW()
                WHERE id = ? AND user_id = ?
            ");
            $stmt->execute([
                $ai_provider, $display_name, $api_key_id,
                $model_preference, $max_tokens, $temperature, 
                $system_prompt, $is_default, $profile_id, $user['id']
            ]);
            
            $message = "✅ Profil IA mis à jour avec succès !";
        } else {
            // 🔧 CORRECTION: Vérifier contrainte unique avant création
            $stmt = $pdo->prepare("SELECT id FROM user_ai_profiles WHERE user_id = ? AND ai_provider = ?");
            $stmt->execute([$user['id'], $ai_provider]);
            $existing_profile = $stmt->fetch();
            
            if ($existing_profile) {
                throw new Exception("Un profil $ai_provider existe déjà pour cet utilisateur");
            }
            
            // Nouveau profil
            $stmt = $pdo->prepare("
                INSERT INTO user_ai_profiles 
                (user_id, ai_provider, display_name, api_key_id, model_preference, 
                 max_tokens_per_request, temperature, system_prompt, is_default, is_active)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 1)
            ");
            $stmt->execute([
                $user['id'], $ai_provider, $display_name, $api_key_id,
                $model_preference, $max_tokens, $temperature, $system_prompt, $is_default
            ]);
            
            $message = "✅ Nouveau profil IA créé avec succès !";
        }
        
        // Si défini comme défaut, désactiver les autres
        if ($is_default) {
            $current_profile_id = $profile_id > 0 ? $profile_id : $pdo->lastInsertId();
            $stmt = $pdo->prepare("UPDATE user_ai_profiles SET is_default = 0 WHERE user_id = ? AND id != ?");
            $stmt->execute([$user['id'], $current_profile_id]);
        }
        
        $message_type = 'success';
        
    } catch (Exception $e) {
        error_log("❌ ERREUR sauvegarde profil: " . $e->getMessage());
        $message = "❌ Erreur : " . $e->getMessage();
        $message_type = 'error';
    }
}

// Action Supprimer Profil
if ($action === 'delete_profile' && isset($_GET['id'])) {
    try {
        $profile_id = intval($_GET['id']);
        
        // Vérifier que le profil appartient à l'utilisateur
        $stmt = $pdo->prepare("SELECT api_key_id FROM user_ai_profiles WHERE id = ? AND user_id = ?");
        $stmt->execute([$profile_id, $user['id']]);
        $profile = $stmt->fetch();
        
        if (!$profile) {
            throw new Exception("Profil non trouvé");
        }
        
        // Supprimer le profil
        $stmt = $pdo->prepare("DELETE FROM user_ai_profiles WHERE id = ? AND user_id = ?");
        $stmt->execute([$profile_id, $user['id']]);
        
        // Supprimer la clé API associée si elle existe et n'est pas utilisée ailleurs
        if ($profile['api_key_id']) {
            $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM user_ai_profiles WHERE api_key_id = ?");
            $stmt->execute([$profile['api_key_id']]);
            $usage_count = $stmt->fetch()['count'];
            
            if ($usage_count == 0) {
                $stmt = $pdo->prepare("DELETE FROM api_keys WHERE id = ? AND user_id = ?");
                $stmt->execute([$profile['api_key_id'], $user['id']]);
            }
        }
        
        $message = "✅ Profil IA supprimé avec succès !";
        $message_type = 'success';
        
    } catch (Exception $e) {
        $message = "❌ Erreur suppression : " . $e->getMessage();
        $message_type = 'error';
    }
}

// Action Test Profil
if ($action === 'test_profile' && isset($_GET['id'])) {
    try {
        $profile_id = intval($_GET['id']);
        $test_result = testAIProfile($profile_id, $user['id']);
        
        if ($test_result['success']) {
            $message = "✅ Test réussi ! Réponse : " . substr($test_result['response'], 0, 100) . "...";
            $message_type = 'success';
        } else {
            $message = "❌ Test échoué : " . $test_result['error'];
            $message_type = 'error';
        }
        
    } catch (Exception $e) {
        $message = "❌ Erreur test : " . $e->getMessage();
        $message_type = 'error';
    }
}

// ================================================================
// 📊 CHARGEMENT DES DONNÉES - CORRIGÉ
// ================================================================

// 🔧 CORRECTION: Profils IA pour l'utilisateur connecté UNIQUEMENT
$user_profiles = getUserAIProfilesFixed($user['id']);
error_log("✅ DEBUG: Profils trouvés pour user {$user['id']}: " . count($user_profiles));

// Providers disponibles
$available_providers = getAvailableProviders();

// Stats utilisation
$usage_stats = getUserAIStats($user['id']);

// Quotas actuels
$current_quotas = getCurrentQuotas($user['id']);

// ================================================================
// 🔧 FONCTIONS MÉTIER CORRIGÉES
// ================================================================

function getUserAIProfilesFixed($user_id) {
    global $pdo;
    
    try {
        // 🔧 REQUÊTE CORRIGÉE: Profils de l'utilisateur connecté SEULEMENT
        $stmt = $pdo->prepare("
            SELECT uap.*,
                   ak.encrypted_key,
                   ak.key_name,
                   apc.provider_name,
                   COALESCE(COUNT(aul.id), 0) as usage_count,
                   COALESCE(SUM(aul.tokens_total), 0) as total_tokens,
                   COALESCE(SUM(aul.cost_estimate), 0) as total_cost,
                   MAX(aul.created_at) as last_used
            FROM user_ai_profiles uap
            LEFT JOIN api_keys ak ON uap.api_key_id = ak.id AND ak.user_id = uap.user_id
            LEFT JOIN ai_providers_catalog apc ON uap.ai_provider = apc.provider_key
            LEFT JOIN ai_usage_logs aul ON uap.id = aul.ai_profile_id
            WHERE uap.user_id = ? AND uap.is_active = 1
            GROUP BY uap.id
            ORDER BY uap.is_default DESC, uap.created_at DESC
        ");
        $stmt->execute([$user_id]);
        $profiles = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        error_log("✅ DEBUG: SQL profils exécutée, résultats: " . count($profiles));
        
        // Debug des profils trouvés
        foreach ($profiles as $profile) {
            $api_status = $profile['encrypted_key'] ? 'Clé chargée' : 'Pas de clé';
            error_log("   - Profil {$profile['id']}: {$profile['display_name']} ({$profile['ai_provider']}) - $api_status");
        }
        
        return $profiles;
        
    } catch (Exception $e) {
        error_log("❌ ERREUR getUserAIProfilesFixed: " . $e->getMessage());
        return [];
    }
}

function getAvailableProviders() {
    global $pdo;
    
    try {
        // Vérifier si la table existe
        $table_check = $pdo->query("SHOW TABLES LIKE 'ai_providers_catalog'");
        if ($table_check->rowCount() == 0) {
            // Retourner les providers par défaut
            return [
                [
                    'provider_key' => 'smg_free',
                    'provider_name' => 'SMG Studio Gratuit',
                    'supported_models' => '["smg-free-v1"]',
                    'free_tier_tokens' => 10000,
                    'pricing_per_1k_tokens' => 0,
                    'status' => 'active'
                ],
                [
                    'provider_key' => 'openai',
                    'provider_name' => 'OpenAI',
                    'supported_models' => '["gpt-3.5-turbo", "gpt-4", "gpt-4o", "gpt-4o-mini"]',
                    'free_tier_tokens' => 0,
                    'pricing_per_1k_tokens' => 0.002,
                    'status' => 'active'
                ],
                [
                    'provider_key' => 'claude',
                    'provider_name' => 'Claude (Anthropic)',
                    'supported_models' => '["claude-3-sonnet", "claude-3-opus"]',
                    'free_tier_tokens' => 0,
                    'pricing_per_1k_tokens' => 0.003,
                    'status' => 'active'
                ]
            ];
        }
        
        $stmt = $pdo->query("
            SELECT * FROM ai_providers_catalog 
            WHERE status = 'active' 
            ORDER BY sort_order, provider_name
        ");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getAvailableProviders: " . $e->getMessage());
        return [];
    }
}

function getUserAIStats($user_id) {
    global $pdo;
    
    try {
        // Vérifier si les tables existent
        $table_check = $pdo->query("SHOW TABLES LIKE 'ai_usage_logs'");
        if ($table_check->rowCount() == 0) {
            return [
                'total_requests' => 0,
                'total_tokens' => 0,
                'total_cost' => 0,
                'avg_time' => 0,
                'failed_requests' => 0,
                'today_requests' => 0,
                'today_tokens' => 0
            ];
        }
        
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as total_requests,
                SUM(tokens_total) as total_tokens,
                SUM(cost_estimate) as total_cost,
                AVG(processing_time_ms) as avg_time,
                COUNT(CASE WHEN success = 0 THEN 1 END) as failed_requests,
                COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN 1 END) as today_requests,
                SUM(CASE WHEN DATE(created_at) = CURDATE() THEN tokens_total ELSE 0 END) as today_tokens
            FROM ai_usage_logs 
            WHERE user_id = ?
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getUserAIStats: " . $e->getMessage());
        return [
            'total_requests' => 0,
            'total_tokens' => 0,
            'total_cost' => 0,
            'avg_time' => 0,
            'failed_requests' => 0,
            'today_requests' => 0,
            'today_tokens' => 0
        ];
    }
}

function getCurrentQuotas($user_id) {
    global $pdo;
    
    try {
        // Vérifier si la table existe
        $table_check = $pdo->query("SHOW TABLES LIKE 'user_ai_quotas'");
        if ($table_check->rowCount() == 0) {
            return [];
        }
        
        $stmt = $pdo->prepare("
            SELECT uaq.*,
                   ROUND((COALESCE(uaq.tokens_used, 0) / GREATEST(uaq.tokens_limit, 1)) * 100, 1) as usage_percentage,
                   CASE 
                       WHEN COALESCE(uaq.tokens_used, 0) >= uaq.tokens_limit THEN 'EXCEEDED'
                       WHEN (COALESCE(uaq.tokens_used, 0) / GREATEST(uaq.tokens_limit, 1)) >= 0.8 THEN 'WARNING'
                       ELSE 'OK'
                   END as quota_status,
                   DATEDIFF(uaq.quota_end_date, CURDATE()) as days_remaining
            FROM user_ai_quotas uaq
            WHERE uaq.user_id = ? AND uaq.is_active = 1 
            AND CURDATE() BETWEEN uaq.quota_start_date AND uaq.quota_end_date
            ORDER BY uaq.quota_period, uaq.quota_end_date
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Erreur getCurrentQuotas: " . $e->getMessage());
        return [];
    }
}

function testAIProfile($profile_id, $user_id) {
    global $pdo;
    
    try {
        // Récupérer le profil avec la clé API
        $stmt = $pdo->prepare("
            SELECT uap.*, ak.encrypted_key
            FROM user_ai_profiles uap
            LEFT JOIN api_keys ak ON uap.api_key_id = ak.id
            WHERE uap.id = ? AND uap.user_id = ?
        ");
        $stmt->execute([$profile_id, $user_id]);
        $profile = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$profile) {
            return ['success' => false, 'error' => 'Profil non trouvé'];
        }
        
        // Test selon le provider
        switch ($profile['ai_provider']) {
            case 'smg_free':
                return ['success' => true, 'response' => 'SMG Studio IA gratuite opérationnelle ! Test simulé réussi.'];
                
            case 'openai':
                return testOpenAIFixed($profile);
                
            case 'claude':
                return ['success' => true, 'response' => 'Claude API configurée (test simulé réussi)'];
                
            case 'local':
                return ['success' => true, 'response' => 'IA locale disponible (test simulé)'];
                
            default:
                return ['success' => false, 'error' => 'Provider non supporté pour test'];
        }
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

function testOpenAIFixed($profile) {
    if (empty($profile['encrypted_key'])) {
        return ['success' => false, 'error' => 'Clé API OpenAI manquante'];
    }
    
    $api_key = base64_decode($profile['encrypted_key']);
    
    // Validation clé OpenAI
    if (!preg_match('/^sk-/', $api_key)) {
        return ['success' => false, 'error' => 'Format de clé OpenAI invalide'];
    }
    
    // Test simple avec l'API OpenAI
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.openai.com/v1/chat/completions",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 15,
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer " . $api_key,
            "Content-Type: application/json"
        ],
        CURLOPT_POSTFIELDS => json_encode([
            'model' => $profile['model_preference'] ?: 'gpt-3.5-turbo',
            'messages' => [
                ['role' => 'user', 'content' => 'Réponds simplement "Test SMG Studio réussi !" en français']
            ],
            'max_tokens' => 20,
            'temperature' => 0
        ])
    ]);
    
    $response = curl_exec($curl);
    $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);
    
    if ($http_code === 200) {
        $data = json_decode($response, true);
        return [
            'success' => true, 
            'response' => $data['choices'][0]['message']['content'] ?? 'Test réussi avec OpenAI'
        ];
    } else {
        $error_data = json_decode($response, true);
        return [
            'success' => false, 
            'error' => ($error_data['error']['message'] ?? 'Erreur API OpenAI') . " (Code: $http_code)"
        ];
    }
}

function getProviderIcon($provider_key) {
    $icons = [
        'smg_free' => '🆓',
        'openai' => '🤖',
        'claude' => '🧠',
        'gemini' => '💎',
        'huggingface' => '🤗',
        'local' => '🏠'
    ];
    return $icons[$provider_key] ?? '🔮';
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🤖 Profils IA - SMG Studio v2.0 FIXED</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #333;
            min-height: 100vh;
        }

        .main-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }

        .main-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 25px 35px;
            margin-bottom: 30px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header-logo {
            width: 60px;
            height: 60px;
            background: linear-gradient(45deg, #4ecdc4, #44a08d);
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        /* 🔧 NAVIGATION CORRIGÉE */
        .nav-ia-profils {
            display: flex;
            gap: 8px;
            background: rgba(255, 107, 107, 0.1);
            padding: 10px;
            border-radius: 12px;
            border: 2px solid rgba(255, 107, 107, 0.2);
        }

        .nav-ia-profils a {
            padding: 8px 16px;
            background: rgba(255, 107, 107, 0.9);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .nav-ia-profils a:hover {
            background: #ff6b6b;
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        .nav-ia-profils a.active {
            background: #ff6b6b;
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.4);
        }

        /* Debug utilisateur */
        .user-debug {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            padding: 10px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
        }

        .content-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }

        .stat-icon {
            font-size: 32px;
            margin-bottom: 10px;
            display: block;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 700;
            color: #4ecdc4;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .profiles-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .profile-card {
            background: white;
            border-radius: 15px;
            padding: 20px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            position: relative;
        }

        .profile-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .profile-card.default {
            border: 2px solid #4ecdc4;
        }

        .profile-card.default::before {
            content: "⭐ Défaut";
            position: absolute;
            top: -10px;
            right: 15px;
            background: #4ecdc4;
            color: white;
            padding: 4px 12px;
            border-radius: 15px;
            font-size: 12px;
            font-weight: 600;
        }

        .profile-header {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
        }

        .provider-icon {
            font-size: 32px;
            margin-right: 15px;
        }

        .profile-info h3 {
            color: #333;
            margin-bottom: 5px;
        }

        .profile-info .provider-name {
            color: #666;
            font-size: 14px;
        }

        .api-key-status {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 15px;
            font-size: 11px;
            font-weight: 600;
            margin-top: 5px;
        }

        .api-key-status.loaded {
            background: #d4edda;
            color: #155724;
        }

        .api-key-status.missing {
            background: #f8d7da;
            color: #721c24;
        }

        .api-key-status.free {
            background: #d1ecf1;
            color: #0c5460;
        }

        .profile-stats {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 10px;
            margin: 15px 0;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 10px;
        }

        .profile-stat {
            text-align: center;
        }

        .profile-stat-value {
            font-size: 16px;
            font-weight: 600;
            color: #4ecdc4;
        }

        .profile-stat-label {
            font-size: 11px;
            color: #666;
            margin-top: 2px;
        }

        .profile-actions {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
            margin-top: 15px;
        }

        .btn {
            padding: 8px 16px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            transition: all 0.3s ease;
            font-size: 13px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #4ecdc4, #44a08d);
            color: white;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, #f093fb, #f5576c);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(135deg, #ff6b6b, #ff8e53);
            color: white;
        }

        .btn-success {
            background: linear-gradient(135deg, #56CCF2, #2F80ED);
            color: white;
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
        }

        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border: 1px solid transparent;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-success {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }

        .alert-error {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }

        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            backdrop-filter: blur(5px);
        }

        .modal-content {
            background: white;
            margin: 2% auto;
            padding: 30px;
            border-radius: 16px;
            width: 90%;
            max-width: 600px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
            font-size: 14px;
        }

        .form-input, .form-select, .form-textarea {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 14px;
            transition: all 0.3s ease;
        }

        .form-input:focus, .form-select:focus, .form-textarea:focus {
            outline: none;
            border-color: #4ecdc4;
            box-shadow: 0 0 0 3px rgba(78, 205, 196, 0.1);
        }

        .form-textarea {
            resize: vertical;
            min-height: 80px;
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }

        @media (max-width: 768px) {
            .profiles-grid {
                grid-template-columns: 1fr;
            }
            
            .form-row {
                grid-template-columns: 1fr;
            }

            .nav-ia-profils {
                flex-wrap: wrap;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header avec Navigation IA Corrigée -->
        <header class="main-header">
            <div class="header-left">
                <div class="header-logo">🤖</div>
                <div>
                    <h1>Profils IA - FIXÉ</h1>
                    <p>Gestion multi-provider avec navigation corrigée</p>
                </div>
            </div>
            <div style="display: flex; align-items: center; gap: 15px;">
                
                <!-- 🔧 Debug utilisateur connecté -->
                <div class="user-debug">
                    👤 <?= htmlspecialchars($user['name']) ?> (ID: <?= $user['id'] ?>)
                </div>
                
                <!-- 🔧 NAVIGATION IA PROFILS CORRIGÉE -->
                <div class="nav-ia-profils">
                    <a href="dashboard.php">
                        ← Dashboard Principal
                    </a>
                    <a href="ai_content_generator.php">
                        🎨 Générateur
                    </a>
                    <a href="ai_dashboard.php">
                        📊 Dashboard IA
                    </a>
                    <a href="ai_profiles.php" class="active">
                        ⚙️ Profils IA
                    </a>
                </div>
                
                <button onclick="openProfileModal()" class="btn btn-primary">➕ Nouveau Profil</button>
            </div>
        </header>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type === 'success' ? 'success' : 'error' ?>">
                <span><?= $message ?></span>
            </div>
        <?php endif; ?>

        <!-- Stats globales -->
        <div class="stats-grid">
            <div class="stat-card">
                <span class="stat-icon">🤖</span>
                <div class="stat-value"><?= count($user_profiles) ?></div>
                <div class="stat-label">Profils configurés</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">🔥</span>
                <div class="stat-value"><?= number_format($usage_stats['total_requests'] ?? 0) ?></div>
                <div class="stat-label">Requêtes totales</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">🎯</span>
                <div class="stat-value"><?= number_format($usage_stats['total_tokens'] ?? 0) ?></div>
                <div class="stat-label">Tokens utilisés</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">💰</span>
                <div class="stat-value"><?= number_format($usage_stats['total_cost'] ?? 0, 2) ?>€</div>
                <div class="stat-label">Coût total</div>
            </div>
            <div class="stat-card">
                <span class="stat-icon">📊</span>
                <div class="stat-value"><?= number_format($usage_stats['today_requests'] ?? 0) ?></div>
                <div class="stat-label">Aujourd'hui</div>
            </div>
        </div>

        <!-- Quotas actuels -->
        <?php if (!empty($current_quotas)): ?>
            <div class="content-card">
                <h3>📊 Quotas Actuels</h3>
                <div style="margin-top: 15px;">
                    <?php foreach ($current_quotas as $quota): ?>
                        <div style="margin-bottom: 15px; padding: 15px; background: #f8f9fa; border-radius: 10px;">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                                <strong><?= ucfirst($quota['quota_period']) ?></strong>
                                <span class="btn btn-sm <?= 
                                    $quota['quota_status'] === 'OK' ? 'btn-success' : 
                                    ($quota['quota_status'] === 'WARNING' ? 'btn-warning' : 'btn-danger') 
                                ?>">
                                    <?= $quota['usage_percentage'] ?>%
                                </span>
                            </div>
                            <div class="quota-bar">
                                <div class="quota-fill <?= strtolower($quota['quota_status']) ?>" 
                                     style="width: <?= min(100, $quota['usage_percentage']) ?>%"></div>
                            </div>
                            <div style="display: flex; justify-content: space-between; font-size: 12px; color: #666; margin-top: 5px;">
                                <span><?= number_format($quota['tokens_used'] ?? 0) ?> / <?= number_format($quota['tokens_limit']) ?> tokens</span>
                                <span><?= $quota['days_remaining'] ?> jours restants</span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>

        <!-- Profils IA -->
        <div class="content-card">
            <h3>🤖 Vos Profils IA (User ID: <?= $user['id'] ?>)</h3>
            
            <?php if (empty($user_profiles)): ?>
                <div style="text-align: center; padding: 60px; color: #666;">
                    <div style="font-size: 64px; margin-bottom: 20px;">🤖</div>
                    <h3>Aucun profil IA configuré</h3>
                    <p style="margin: 15px 0;">Créez votre premier profil pour commencer à utiliser l'IA</p>
                    <button onclick="openProfileModal()" class="btn btn-primary">➕ Créer un profil</button>
                </div>
            <?php else: ?>
                <div class="profiles-grid">
                    <?php foreach ($user_profiles as $profile): ?>
                        <div class="profile-card <?= $profile['is_default'] ? 'default' : '' ?>">
                            <div class="profile-header">
                                <span class="provider-icon">
                                    <?= getProviderIcon($profile['ai_provider']) ?>
                                </span>
                                <div class="profile-info">
                                    <h3><?= htmlspecialchars($profile['display_name']) ?></h3>
                                    <div class="provider-name">
                                        <?= htmlspecialchars($profile['provider_name'] ?? ucfirst($profile['ai_provider'])) ?>
                                    </div>
                                    
                                    <!-- 🔧 Status clé API amélioré -->
                                    <?php if ($profile['ai_provider'] === 'smg_free' || $profile['ai_provider'] === 'local'): ?>
                                        <div class="api-key-status free">🆓 Gratuit</div>
                                    <?php elseif (!empty($profile['encrypted_key'])): ?>
                                        <div class="api-key-status loaded">🔑 Clé chargée</div>
                                    <?php else: ?>
                                        <div class="api-key-status missing">❌ Clé manquante</div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="profile-stats">
                                <div class="profile-stat">
                                    <div class="profile-stat-value"><?= $profile['usage_count'] ?? 0 ?></div>
                                    <div class="profile-stat-label">Requêtes</div>
                                </div>
                                <div class="profile-stat">
                                    <div class="profile-stat-value"><?= number_format($profile['total_tokens'] ?? 0) ?></div>
                                    <div class="profile-stat-label">Tokens</div>
                                </div>
                                <div class="profile-stat">
                                    <div class="profile-stat-value"><?= number_format($profile['total_cost'] ?? 0, 2) ?>€</div>
                                    <div class="profile-stat-label">Coût</div>
                                </div>
                            </div>
                            
                            <div style="margin: 10px 0; font-size: 12px; color: #666;">
                                <strong>Modèle :</strong> <?= htmlspecialchars($profile['model_preference'] ?: 'Défaut') ?><br>
                                <strong>Tokens max :</strong> <?= $profile['max_tokens_per_request'] ?><br>
                                <strong>Température :</strong> <?= $profile['temperature'] ?>
                                <?php if ($profile['api_key_id']): ?>
                                    <br><strong>Clé API ID :</strong> #<?= $profile['api_key_id'] ?>
                                <?php endif; ?>
                            </div>
                            
                            <?php if ($profile['last_used']): ?>
                                <div style="font-size: 11px; color: #999; margin-bottom: 15px;">
                                    Dernière utilisation : <?= date('d/m/Y H:i', strtotime($profile['last_used'])) ?>
                                </div>
                            <?php endif; ?>
                            
                            <div class="profile-actions">
                                <button onclick="editProfile(<?= $profile['id'] ?>)" class="btn btn-primary btn-sm">
                                    ✏️ Modifier
                                </button>
                                <a href="?action=test_profile&id=<?= $profile['id'] ?>" class="btn btn-success btn-sm">
                                    🧪 Tester
                                </a>
                                <?php if (!$profile['is_default']): ?>
                                    <a href="?action=delete_profile&id=<?= $profile['id'] ?>" 
                                       onclick="return confirm('Supprimer ce profil IA ?')"
                                       class="btn btn-danger btn-sm">
                                        🗑️ Suppr
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Modal Profil IA -->
    <div id="profileModal" class="modal">
        <div class="modal-content">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                <h3 id="modalTitle">Nouveau Profil IA</h3>
                <button onclick="closeProfileModal()" style="background: none; border: none; font-size: 24px; cursor: pointer;">×</button>
            </div>
            
            <form method="POST" id="profileForm">
                <input type="hidden" name="action" value="save_profile">
                <input type="hidden" name="profile_id" id="profileId" value="0">
                
                <div class="form-group">
                    <label class="form-label">🤖 Provider IA</label>
                    <select name="ai_provider" id="aiProvider" class="form-select" required onchange="updateProviderFields()">
                        <option value="">Sélectionner un provider</option>
                        <?php foreach ($available_providers as $provider): ?>
                            <option value="<?= $provider['provider_key'] ?>" 
                                    data-models='<?= htmlspecialchars($provider['supported_models']) ?>'
                                    data-free="<?= $provider['free_tier_tokens'] ?>"
                                    data-pricing="<?= $provider['pricing_per_1k_tokens'] ?>">
                                <?= getProviderIcon($provider['provider_key']) ?> <?= htmlspecialchars($provider['provider_name']) ?>
                                <?php if ($provider['free_tier_tokens'] > 0): ?>
                                    (<?= number_format($provider['free_tier_tokens']) ?> tokens gratuits/mois)
                                <?php endif; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">📝 Nom d'affichage</label>
                    <input type="text" name="display_name" id="displayName" class="form-input" 
                           placeholder="Ex: Mon ChatGPT Principal" required>
                </div>
                
                <div class="form-group" id="apiKeyGroup">
                    <label class="form-label">🔑 Clé API</label>
                    <input type="password" name="api_key" id="apiKey" class="form-input" 
                           placeholder="Votre clé API personnelle (ex: sk-...)">
                    <small style="color: #666; font-size: 12px;">
                        Laissez vide pour conserver la clé existante lors d'une modification
                    </small>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">🧠 Modèle</label>
                        <select name="model_preference" id="modelPreference" class="form-select">
                            <option value="">Modèle par défaut</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label class="form-label">🎯 Tokens max</label>
                        <input type="number" name="max_tokens_per_request" id="maxTokens" class="form-input" 
                               value="1000" min="100" max="8000">
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">🌡️ Température (créativité)</label>
                    <input type="range" name="temperature" id="temperature" min="0" max="2" step="0.1" value="0.7" 
                           oninput="document.getElementById('tempValue').textContent = this.value">
                    <div style="text-align: center; margin-top: 5px;">
                        <span id="tempValue">0.7</span>
                        <small style="color: #666; display: block;">0 = Précis, 2 = Créatif</small>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">💬 Prompt système (optionnel)</label>
                    <textarea name="system_prompt" id="systemPrompt" class="form-textarea" 
                              placeholder="Prompt système personnalisé pour définir le comportement de l'IA..."></textarea>
                </div>
                
                <div class="form-group">
                    <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                        <input type="checkbox" name="is_default" id="isDefault">
                        <span>⭐ Définir comme profil par défaut</span>
                    </label>
                </div>
                
                <div style="display: flex; gap: 15px; margin-top: 25px;">
                    <button type="submit" class="btn btn-primary">
                        ✅ Enregistrer Profil
                    </button>
                    <button type="button" onclick="closeProfileModal()" class="btn btn-secondary">
                        ❌ Annuler
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // ================================================================
        // 🔧 JAVASCRIPT PROFILS IA CORRIGÉ
        // ================================================================

        function openProfileModal(profileData = null) {
            const modal = document.getElementById('profileModal');
            const form = document.getElementById('profileForm');
            const title = document.getElementById('modalTitle');
            
            if (profileData) {
                // Mode édition
                title.textContent = 'Modifier Profil IA';
                document.getElementById('profileId').value = profileData.id;
                document.getElementById('aiProvider').value = profileData.ai_provider;
                document.getElementById('displayName').value = profileData.display_name;
                document.getElementById('maxTokens').value = profileData.max_tokens_per_request;
                document.getElementById('temperature').value = profileData.temperature;
                document.getElementById('tempValue').textContent = profileData.temperature;
                document.getElementById('systemPrompt').value = profileData.system_prompt || '';
                document.getElementById('isDefault').checked = profileData.is_default == 1;
                
                updateProviderFields();
                document.getElementById('modelPreference').value = profileData.model_preference || '';
            } else {
                // Mode création
                title.textContent = 'Nouveau Profil IA';
                form.reset();
                document.getElementById('profileId').value = '0';
                document.getElementById('tempValue').textContent = '0.7';
            }
            
            modal.style.display = 'block';
        }

        function closeProfileModal() {
            document.getElementById('profileModal').style.display = 'none';
        }

        function updateProviderFields() {
            const provider = document.getElementById('aiProvider');
            const selectedOption = provider.options[provider.selectedIndex];
            const apiKeyGroup = document.getElementById('apiKeyGroup');
            const modelSelect = document.getElementById('modelPreference');
            
            // Gérer affichage clé API
            if (provider.value === 'smg_free' || provider.value === 'local') {
                apiKeyGroup.style.display = 'none';
            } else {
                apiKeyGroup.style.display = 'block';
            }
            
            // Mettre à jour les modèles
            modelSelect.innerHTML = '<option value="">Modèle par défaut</option>';
            
            if (selectedOption && selectedOption.dataset.models) {
                try {
                    const models = JSON.parse(selectedOption.dataset.models);
                    models.forEach(model => {
                        const option = document.createElement('option');
                        option.value = model;
                        option.textContent = model;
                        modelSelect.appendChild(option);
                    });
                } catch (e) {
                    console.error('Erreur parsing modèles:', e);
                }
            }
        }

        function editProfile(profileId) {
            // En production, faire un appel AJAX pour récupérer les données
            window.location.href = `?action=edit_profile&id=${profileId}`;
        }

        // Fermer modal si clic outside
        window.onclick = function(event) {
            const modal = document.getElementById('profileModal');
            if (event.target === modal) {
                closeProfileModal();
            }
        }

        // Raccourcis clavier
        document.addEventListener('keydown', function(e) {
            // Échap : Retour dashboard principal
            if (e.key === 'Escape') {
                if (document.getElementById('profileModal').style.display === 'block') {
                    closeProfileModal();
                } else {
                    window.location.href = 'dashboard.php';
                }
            }
            
            // Ctrl+N : Nouveau profil
            if (e.ctrlKey && e.key === 'n') {
                e.preventDefault();
                openProfileModal();
            }
        });

        console.log('🤖 SMG Studio IA Profiles Manager FIXED!');
        console.log('✅ Navigation retour dashboard CORRIGÉE');
        console.log('✅ Liaison clés API ↔ utilisateurs CORRIGÉE'); 
        console.log('✅ Affichage profils user connecté CORRIGÉ');
        console.log('✅ Validation clés OpenAI CORRIGÉE');
    </script>
</body>
</html>